"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthorizationType = exports.Method = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../aws-cloudwatch");
const core_1 = require("../../core");
const apigateway_canned_metrics_generated_1 = require("./apigateway-canned-metrics.generated");
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const mock_1 = require("./integrations/mock");
const restapi_1 = require("./restapi");
const util_1 = require("./util");
class Method extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_MethodProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Method);
            }
            throw error;
        }
        this.resource = props.resource;
        this.api = props.resource.api;
        this.httpMethod = props.httpMethod.toUpperCase();
        (0, util_1.validateHttpMethod)(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = authorizer?.authorizerId ? authorizer.authorizerId : undefined;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = authorizer?.authorizationType || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if (authorizer?.authorizationType && authorizationTypeOption && authorizer?.authorizationType !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer_1.Authorizer.isAuthorizer(authorizer)) {
            authorizer._attachToApi(this.api);
        }
        this.methodResponses = options.methodResponses ?? [];
        const integration = props.integration ?? this.resource.defaultIntegration ?? new mock_1.MockIntegration();
        const bindResult = integration.bind(this);
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.api.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(bindResult),
            methodResponses: core_1.Lazy.any({ produce: () => this.renderMethodResponses(this.methodResponses) }, { omitEmptyArray: true }),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: this.requestValidatorId(options),
            authorizationScopes: options.authorizationScopes ?? defaultMethodOptions.authorizationScopes,
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        if (restapi_1.RestApiBase._isRestApiBase(props.resource.api)) {
            props.resource.api._attachMethod(this);
        }
        const deployment = props.resource.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({
                method: {
                    ...methodProps,
                    integrationToken: bindResult?.deploymentToken,
                },
            });
        }
    }
    /**
     * The RestApi associated with this Method
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        return this.resource.restApi;
    }
    /**
     * Returns an execute-api ARN for this method:
     *
     *   arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled, or can be explicitly assigned.
     * When not configured, {stage} will be set to '*', as a shorthand for 'all stages'.
     *
     * @attribute
     */
    get methodArn() {
        const stage = this.api.deploymentStage?.stageName;
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     * This stage is used by the AWS Console UI when testing the method.
     */
    get testMethodArn() {
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), 'test-invoke-stage');
    }
    /**
     * Add a method response to this method
     */
    addMethodResponse(methodResponse) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_MethodResponse(methodResponse);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addMethodResponse);
            }
            throw error;
        }
        this.methodResponses.push(methodResponse);
    }
    renderIntegration(bindResult) {
        const options = bindResult.options ?? {};
        let credentials;
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // eslint-disable-next-line max-len
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME, resourceName: '*' });
        }
        return {
            type: bindResult.type,
            uri: bindResult.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: bindResult.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
            timeoutInMillis: options.timeout?.toMilliseconds(),
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
    requestValidatorId(options) {
        if (options.requestValidator && options.requestValidatorOptions) {
            throw new Error('Only one of \'requestValidator\' or \'requestValidatorOptions\' must be specified.');
        }
        if (options.requestValidatorOptions) {
            const validator = this.api.addRequestValidator('validator', options.requestValidatorOptions);
            return validator.requestValidatorId;
        }
        // For backward compatibility
        return options.requestValidator?.requestValidatorId;
    }
    /**
     * Returns the given named metric for this API method
     */
    metric(metricName, stage, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_IStage(stage);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metric);
            }
            throw error;
        }
        return new cloudwatch.Metric({
            namespace: 'AWS/ApiGateway',
            metricName,
            dimensionsMap: { ApiName: this.api.restApiName, Method: this.httpMethod, Resource: this.resource.path, Stage: stage.stageName },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the number of client-side errors captured in a given period.
     *
     * @default - sum over 5 minutes
     */
    metricClientError(stage, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_IStage(stage);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricClientError);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics._4XxErrorSum, stage, props);
    }
    /**
     * Metric for the number of server-side errors captured in a given period.
     *
     * @default - sum over 5 minutes
     */
    metricServerError(stage, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_IStage(stage);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricServerError);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics._5XxErrorSum, stage, props);
    }
    /**
     * Metric for the number of requests served from the API cache in a given period.
     *
     * @default - sum over 5 minutes
     */
    metricCacheHitCount(stage, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_IStage(stage);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricCacheHitCount);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.cacheHitCountSum, stage, props);
    }
    /**
     * Metric for the number of requests served from the backend in a given period,
     * when API caching is enabled.
     *
     * @default - sum over 5 minutes
     */
    metricCacheMissCount(stage, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_IStage(stage);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricCacheMissCount);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.cacheMissCountSum, stage, props);
    }
    /**
     * Metric for the total number API requests in a given period.
     *
     * @default - sample count over 5 minutes
     */
    metricCount(stage, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_IStage(stage);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricCount);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.countSum, stage, {
            statistic: 'SampleCount',
            ...props,
        });
    }
    /**
     * Metric for the time between when API Gateway relays a request to the backend
     * and when it receives a response from the backend.
     *
     * @default - average over 5 minutes.
     */
    metricIntegrationLatency(stage, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_IStage(stage);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricIntegrationLatency);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.integrationLatencyAverage, stage, props);
    }
    /**
     * The time between when API Gateway receives a request from a client
     * and when it returns a response to the client.
     * The latency includes the integration latency and other API Gateway overhead.
     *
     * @default - average over 5 minutes.
     */
    metricLatency(stage, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_IStage(stage);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricLatency);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.latencyAverage, stage, props);
    }
    cannedMetric(fn, stage, props) {
        return new cloudwatch.Metric({
            ...fn({ ApiName: this.api.restApiName, Method: this.httpMethod, Resource: this.resource.path, Stage: stage.stageName }),
            ...props,
        }).attachTo(this);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Method[_a] = { fqn: "aws-cdk-lib.aws_apigateway.Method", version: "2.74.0" };
exports.Method = Method;
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * Open access.
     */
    AuthorizationType["NONE"] = "NONE";
    /**
     * Use AWS IAM permissions.
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Use a custom authorizer.
     */
    AuthorizationType["CUSTOM"] = "CUSTOM";
    /**
     * Use an AWS Cognito user pool.
     */
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
function pathForArn(path) {
    return path.replace(/\{[^\}]*\}/g, '*'); // replace path parameters (like '{bookId}') with asterisk
}
//# sourceMappingURL=data:application/json;base64,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