"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = exports.StageBase = exports.MethodLoggingLevel = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../aws-cloudwatch");
const core_1 = require("../../core");
const access_log_1 = require("./access-log");
const api_key_1 = require("./api-key");
const apigateway_canned_metrics_generated_1 = require("./apigateway-canned-metrics.generated");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
const util_1 = require("./util");
var MethodLoggingLevel;
(function (MethodLoggingLevel) {
    MethodLoggingLevel["OFF"] = "OFF";
    MethodLoggingLevel["ERROR"] = "ERROR";
    MethodLoggingLevel["INFO"] = "INFO";
})(MethodLoggingLevel = exports.MethodLoggingLevel || (exports.MethodLoggingLevel = {}));
/**
 * Base class for an ApiGateway Stage
 */
class StageBase extends core_1.Resource {
    /**
     * Add an ApiKey to this stage
     */
    addApiKey(id, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_ApiKeyOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addApiKey);
            }
            throw error;
        }
        return new api_key_1.ApiKey(this, id, {
            stages: [this],
            ...options,
        });
    }
    /**
     * Returns the invoke URL for a certain path.
     * @param path The resource path
     */
    urlForPath(path = '/') {
        if (!path.startsWith('/')) {
            throw new Error(`Path must begin with "/": ${path}`);
        }
        return `https://${this.restApi.restApiId}.execute-api.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}/${this.stageName}${path}`;
    }
    /**
     * Returns the resource ARN for this stage:
     *
     *   arn:aws:apigateway:{region}::/restapis/{restApiId}/stages/{stageName}
     *
     * Note that this is separate from the execute-api ARN for methods and resources
     * within this stage.
     *
     * @attribute
     */
    get stageArn() {
        return core_1.Stack.of(this).formatArn({
            arnFormat: core_1.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME,
            service: 'apigateway',
            account: '',
            resource: 'restapis',
            resourceName: `${this.restApi.restApiId}/stages/${this.stageName}`,
        });
    }
    /**
     * Returns the given named metric for this stage
     */
    metric(metricName, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metric);
            }
            throw error;
        }
        return new cloudwatch.Metric({
            namespace: 'AWS/ApiGateway',
            metricName,
            dimensionsMap: { ApiName: this.restApi.restApiName, Stage: this.stageName },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the number of client-side errors captured in a given period.
     *
     * @default - sum over 5 minutes
     */
    metricClientError(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricClientError);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics._4XxErrorSum, props);
    }
    /**
     * Metric for the number of server-side errors captured in a given period.
     *
     * @default - sum over 5 minutes
     */
    metricServerError(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricServerError);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics._5XxErrorSum, props);
    }
    /**
     * Metric for the number of requests served from the API cache in a given period.
     *
     * @default - sum over 5 minutes
     */
    metricCacheHitCount(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricCacheHitCount);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.cacheHitCountSum, props);
    }
    /**
     * Metric for the number of requests served from the backend in a given period,
     * when API caching is enabled.
     *
     * @default - sum over 5 minutes
     */
    metricCacheMissCount(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricCacheMissCount);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.cacheMissCountSum, props);
    }
    /**
     * Metric for the total number API requests in a given period.
     *
     * @default - sample count over 5 minutes
     */
    metricCount(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricCount);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.countSum, {
            statistic: 'SampleCount',
            ...props,
        });
    }
    /**
     * Metric for the time between when API Gateway relays a request to the backend
     * and when it receives a response from the backend.
     *
     * @default - average over 5 minutes.
     */
    metricIntegrationLatency(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricIntegrationLatency);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.integrationLatencyAverage, props);
    }
    /**
     * The time between when API Gateway receives a request from a client
     * and when it returns a response to the client.
     * The latency includes the integration latency and other API Gateway overhead.
     *
     * @default - average over 5 minutes.
     */
    metricLatency(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricLatency);
            }
            throw error;
        }
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.latencyAverage, props);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ ApiName: this.restApi.restApiName, Stage: this.stageName }),
            ...props,
        }).attachTo(this);
    }
}
_a = JSII_RTTI_SYMBOL_1;
StageBase[_a] = { fqn: "aws-cdk-lib.aws_apigateway.StageBase", version: "2.74.0" };
exports.StageBase = StageBase;
class Stage extends StageBase {
    /**
     * Import a Stage by its attributes
     */
    static fromStageAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_StageAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromStageAttributes);
            }
            throw error;
        }
        class Import extends StageBase {
            constructor() {
                super(...arguments);
                this.stageName = attrs.stageName;
                this.restApi = attrs.restApi;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_StageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Stage);
            }
            throw error;
        }
        this.enableCacheCluster = props.cacheClusterEnabled;
        const methodSettings = this.renderMethodSettings(props); // this can mutate `this.cacheClusterEnabled`
        // custom access logging
        let accessLogSetting;
        const accessLogDestination = props.accessLogDestination;
        const accessLogFormat = props.accessLogFormat;
        if (!accessLogDestination && !accessLogFormat) {
            accessLogSetting = undefined;
        }
        else {
            if (accessLogFormat !== undefined &&
                !core_1.Token.isUnresolved(accessLogFormat.toString()) &&
                !/.*\$context.(requestId|extendedRequestId)\b.*/.test(accessLogFormat.toString())) {
                throw new Error('Access log must include either `AccessLogFormat.contextRequestId()` or `AccessLogFormat.contextExtendedRequestId()`');
            }
            if (accessLogFormat !== undefined && accessLogDestination === undefined) {
                throw new Error('Access log format is specified without a destination');
            }
            accessLogSetting = {
                destinationArn: accessLogDestination?.bind(this).destinationArn,
                format: accessLogFormat?.toString() ? accessLogFormat?.toString() : access_log_1.AccessLogFormat.clf().toString(),
            };
        }
        // enable cache cluster if cacheClusterSize is set
        if (props.cacheClusterSize !== undefined) {
            if (this.enableCacheCluster === undefined) {
                this.enableCacheCluster = true;
            }
            else if (this.enableCacheCluster === false) {
                throw new Error(`Cannot set "cacheClusterSize" to ${props.cacheClusterSize} and "cacheClusterEnabled" to "false"`);
            }
        }
        const cacheClusterSize = this.enableCacheCluster ? (props.cacheClusterSize || '0.5') : undefined;
        const resource = new apigateway_generated_1.CfnStage(this, 'Resource', {
            stageName: props.stageName || 'prod',
            accessLogSetting,
            cacheClusterEnabled: this.enableCacheCluster,
            cacheClusterSize,
            clientCertificateId: props.clientCertificateId,
            deploymentId: props.deployment.deploymentId,
            restApiId: props.deployment.api.restApiId,
            description: props.description,
            documentationVersion: props.documentationVersion,
            variables: props.variables,
            tracingEnabled: props.tracingEnabled,
            methodSettings,
        });
        this.stageName = resource.ref;
        this.restApi = props.deployment.api;
        if (restapi_1.RestApiBase._isRestApiBase(this.restApi)) {
            this.restApi._attachStage(this);
        }
    }
    renderMethodSettings(props) {
        const settings = new Array();
        const self = this;
        // extract common method options from the stage props
        const commonMethodOptions = {
            metricsEnabled: props.metricsEnabled,
            loggingLevel: props.loggingLevel,
            dataTraceEnabled: props.dataTraceEnabled,
            throttlingBurstLimit: props.throttlingBurstLimit,
            throttlingRateLimit: props.throttlingRateLimit,
            cachingEnabled: props.cachingEnabled,
            cacheTtl: props.cacheTtl,
            cacheDataEncrypted: props.cacheDataEncrypted,
        };
        // if any of them are defined, add an entry for '/*/*'.
        const hasCommonOptions = Object.keys(commonMethodOptions).map(v => commonMethodOptions[v]).filter(x => x !== undefined).length > 0;
        if (hasCommonOptions) {
            settings.push(renderEntry('/*/*', commonMethodOptions));
        }
        if (props.methodOptions) {
            for (const path of Object.keys(props.methodOptions)) {
                settings.push(renderEntry(path, props.methodOptions[path]));
            }
        }
        return settings.length === 0 ? undefined : settings;
        function renderEntry(path, options) {
            if (options.cachingEnabled) {
                if (self.enableCacheCluster === undefined) {
                    self.enableCacheCluster = true;
                }
                else if (self.enableCacheCluster === false) {
                    throw new Error(`Cannot enable caching for method ${path} since cache cluster is disabled on stage`);
                }
            }
            const { httpMethod, resourcePath } = (0, util_1.parseMethodOptionsPath)(path);
            return {
                httpMethod,
                resourcePath,
                cacheDataEncrypted: options.cacheDataEncrypted,
                cacheTtlInSeconds: options.cacheTtl && options.cacheTtl.toSeconds(),
                cachingEnabled: options.cachingEnabled,
                dataTraceEnabled: options.dataTraceEnabled ?? false,
                loggingLevel: options.loggingLevel,
                metricsEnabled: options.metricsEnabled,
                throttlingBurstLimit: options.throttlingBurstLimit,
                throttlingRateLimit: options.throttlingRateLimit,
            };
        }
    }
}
_b = JSII_RTTI_SYMBOL_1;
Stage[_b] = { fqn: "aws-cdk-lib.aws_apigateway.Stage", version: "2.74.0" };
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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