"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsagePlan = exports.Period = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const cx_api_1 = require("../../cx-api");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
class UsagePlanBase extends core_1.Resource {
    /**
     * Adds an ApiKey.
     *
     * @param apiKey the api key to associate with this usage plan
     * @param options options that control the behaviour of this method
     */
    addApiKey(apiKey, options) {
        let id;
        const prefix = 'UsagePlanKeyResource';
        if (core_1.FeatureFlags.of(this).isEnabled(cx_api_1.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID)) {
            id = `${prefix}:${core_1.Names.nodeUniqueId(apiKey.node)}`;
        }
        else {
            // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodified.
            id = this.node.tryFindChild(prefix) ? `${prefix}:${core_1.Names.nodeUniqueId(apiKey.node)}` : prefix;
        }
        const resource = new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* UsagePlanKeyType.API_KEY */,
            usagePlanId: this.usagePlanId,
        });
        if (options?.overrideLogicalId) {
            resource.overrideLogicalId(options?.overrideLogicalId);
        }
    }
}
class UsagePlan extends UsagePlanBase {
    /**
     * Import an externally defined usage plan using its ARN.
     *
     * @param scope  the construct that will "own" the imported usage plan.
     * @param id     the id of the imported usage plan in the construct tree.
     * @param usagePlanId the id of an existing usage plan.
     */
    static fromUsagePlanId(scope, id, usagePlanId) {
        class Import extends UsagePlanBase {
            constructor() {
                super(scope, id);
                this.usagePlanId = usagePlanId;
            }
        }
        return new Import();
    }
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_UsagePlanProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, UsagePlan);
            }
            throw error;
        }
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.any({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Adds an apiStage.
     * @param apiStage
     */
    addApiStage(apiStage) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_UsagePlanPerApiStage(apiStage);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addApiStage);
            }
            throw error;
        }
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle,
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            (0, util_1.validateInteger)(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined,
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            (0, util_1.validateInteger)(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            (0, util_1.validateDouble)(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit,
                rateLimit: rateLimit,
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
_a = JSII_RTTI_SYMBOL_1;
UsagePlan[_a] = { fqn: "aws-cdk-lib.aws_apigateway.UsagePlan", version: "2.74.0" };
exports.UsagePlan = UsagePlan;
//# sourceMappingURL=data:application/json;base64,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