"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonSchemaMapper = exports.validateDouble = exports.validateInteger = exports.parseAwsApiCall = exports.parseMethodOptionsPath = exports.validateHttpMethod = exports.ALL_METHODS = void 0;
const url_1 = require("url");
const jsonSchema = require("./json-schema");
exports.ALL_METHODS = ['OPTIONS', 'GET', 'PUT', 'POST', 'DELETE', 'PATCH', 'HEAD'];
const ALLOWED_METHODS = ['ANY', ...exports.ALL_METHODS];
function validateHttpMethod(method, messagePrefix = '') {
    if (!ALLOWED_METHODS.includes(method)) {
        throw new Error(`${messagePrefix}Invalid HTTP method "${method}". Allowed methods: ${ALLOWED_METHODS.join(',')}`);
    }
}
exports.validateHttpMethod = validateHttpMethod;
function parseMethodOptionsPath(originalPath) {
    if (!originalPath.startsWith('/')) {
        throw new Error(`Method options path must start with '/': ${originalPath}`);
    }
    const path = originalPath.slice(1); // trim trailing '/'
    const components = path.split('/');
    if (components.length < 2) {
        throw new Error(`Method options path must include at least two components: /{resource}/{method} (i.e. /foo/bar/GET): ${path}`);
    }
    const httpMethod = components.pop().toUpperCase(); // last component is an HTTP method
    if (httpMethod !== '*') {
        validateHttpMethod(httpMethod, `${originalPath}: `);
    }
    let resourcePath = '/~1' + components.join('~1');
    if (components.length === 1 && components[0] === '*') {
        resourcePath = '/*';
    }
    else if (components.length === 1 && components[0] === '') {
        resourcePath = '/';
    }
    return {
        httpMethod,
        resourcePath,
    };
}
exports.parseMethodOptionsPath = parseMethodOptionsPath;
function parseAwsApiCall(path, action, actionParams) {
    if (actionParams && !action) {
        throw new Error('"actionParams" requires that "action" will be set');
    }
    if (path && action) {
        throw new Error(`"path" and "action" are mutually exclusive (path="${path}", action="${action}")`);
    }
    if (path) {
        return {
            apiType: 'path',
            apiValue: path,
        };
    }
    if (action) {
        if (actionParams) {
            action += '&' + (0, url_1.format)({ query: actionParams }).slice(1);
        }
        return {
            apiType: 'action',
            apiValue: action,
        };
    }
    throw new Error('Either "path" or "action" are required');
}
exports.parseAwsApiCall = parseAwsApiCall;
function validateInteger(property, messagePrefix) {
    if (property && !Number.isInteger(property)) {
        throw new Error(`${messagePrefix} should be an integer`);
    }
}
exports.validateInteger = validateInteger;
function validateDouble(property, messagePrefix) {
    if (property && isNaN(property) && isNaN(parseFloat(property.toString()))) {
        throw new Error(`${messagePrefix} should be an double`);
    }
}
exports.validateDouble = validateDouble;
class JsonSchemaMapper {
    /**
     * Transforms naming of some properties to prefix with a $, where needed
     * according to the JSON schema spec
     * @param schema The JsonSchema object to transform for CloudFormation output
     */
    static toCfnJsonSchema(schema) {
        const result = JsonSchemaMapper._toCfnJsonSchema(schema);
        if (!('$schema' in result)) {
            result.$schema = jsonSchema.JsonSchemaVersion.DRAFT4;
        }
        return result;
    }
    static _toCfnJsonSchema(schema, preserveKeys = false) {
        if (schema == null || typeof schema !== 'object') {
            return schema;
        }
        if (Array.isArray(schema)) {
            return schema.map(entry => JsonSchemaMapper._toCfnJsonSchema(entry));
        }
        return Object.assign({}, ...Object.entries(schema).map(([key, value]) => {
            const mapKey = !preserveKeys && (key in JsonSchemaMapper.SchemaPropsWithPrefix);
            const newKey = mapKey ? JsonSchemaMapper.SchemaPropsWithPrefix[key] : key;
            // If keys were preserved, don't consider SchemaPropsWithUserDefinedChildren for those keys (they are user-defined!)
            const newValue = JsonSchemaMapper._toCfnJsonSchema(value, !preserveKeys && JsonSchemaMapper.SchemaPropsWithUserDefinedChildren[key]);
            return { [newKey]: newValue };
        }));
    }
}
JsonSchemaMapper.SchemaPropsWithPrefix = {
    schema: '$schema',
    ref: '$ref',
};
// The value indicates whether direct children should be key-mapped.
JsonSchemaMapper.SchemaPropsWithUserDefinedChildren = {
    definitions: true,
    properties: true,
    patternProperties: true,
    dependencies: true,
};
exports.JsonSchemaMapper = JsonSchemaMapper;
//# sourceMappingURL=data:application/json;base64,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