"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepScalingPolicy = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_common_1 = require("../../aws-autoscaling-common");
const cloudwatch = require("../../aws-cloudwatch");
const constructs_1 = require("constructs");
const step_scaling_action_1 = require("./step-scaling-action");
/**
 * Define a scaling strategy which scales depending on absolute values of some metric.
 *
 * You can specify the scaling behavior for various values of the metric.
 *
 * Implemented using one or more CloudWatch alarms and Step Scaling Policies.
 */
class StepScalingPolicy extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_applicationautoscaling_StepScalingPolicyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, StepScalingPolicy);
            }
            throw error;
        }
        if (props.scalingSteps.length < 2) {
            throw new Error('You must supply at least 2 intervals for autoscaling');
        }
        if (props.datapointsToAlarm !== undefined && props.datapointsToAlarm < 1) {
            throw new RangeError(`datapointsToAlarm cannot be less than 1, got: ${props.datapointsToAlarm}`);
        }
        const adjustmentType = props.adjustmentType || step_scaling_action_1.AdjustmentType.CHANGE_IN_CAPACITY;
        const changesAreAbsolute = adjustmentType === step_scaling_action_1.AdjustmentType.EXACT_CAPACITY;
        const intervals = (0, aws_autoscaling_common_1.normalizeIntervals)(props.scalingSteps, changesAreAbsolute);
        const alarms = (0, aws_autoscaling_common_1.findAlarmThresholds)(intervals);
        if (alarms.lowerAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.lowerAlarmIntervalIndex].upper;
            this.lowerAction = new step_scaling_action_1.StepScalingAction(this, 'LowerPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: props.metricAggregationType ?? aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.lowerAlarmIntervalIndex; i >= 0; i--) {
                this.lowerAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: i !== 0 ? intervals[i].lower - threshold : undefined,
                    upperBound: intervals[i].upper - threshold,
                });
            }
            this.lowerAlarm = new cloudwatch.Alarm(this, 'LowerAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Lower threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: props.evaluationPeriods ?? 1,
                datapointsToAlarm: props.datapointsToAlarm,
                threshold,
            });
            this.lowerAlarm.addAlarmAction(new StepScalingAlarmAction(this.lowerAction));
        }
        if (alarms.upperAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.upperAlarmIntervalIndex].lower;
            this.upperAction = new step_scaling_action_1.StepScalingAction(this, 'UpperPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: props.metricAggregationType ?? aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.upperAlarmIntervalIndex; i < intervals.length; i++) {
                this.upperAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: intervals[i].lower - threshold,
                    upperBound: i !== intervals.length - 1 ? intervals[i].upper - threshold : undefined, // Extend last interval to +infinity
                });
            }
            this.upperAlarm = new cloudwatch.Alarm(this, 'UpperAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Upper threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: props.evaluationPeriods ?? 1,
                datapointsToAlarm: props.datapointsToAlarm,
                threshold,
            });
            this.upperAlarm.addAlarmAction(new StepScalingAlarmAction(this.upperAction));
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
StepScalingPolicy[_a] = { fqn: "aws-cdk-lib.aws_applicationautoscaling.StepScalingPolicy", version: "2.74.0" };
exports.StepScalingPolicy = StepScalingPolicy;
function aggregationTypeFromMetric(metric) {
    const statistic = metric.toMetricConfig().metricStat?.statistic;
    if (statistic == null) {
        return undefined;
    } // Math expression, don't know aggregation, leave default
    switch (statistic) {
        case 'Average':
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
        case 'Minimum':
            return step_scaling_action_1.MetricAggregationType.MINIMUM;
        case 'Maximum':
            return step_scaling_action_1.MetricAggregationType.MAXIMUM;
        default:
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
    }
}
/**
 * Use a StepScalingAction as an Alarm Action
 *
 * This class is here and not in aws-cloudwatch-actions because this library
 * needs to use the class, and otherwise we'd have a circular dependency:
 *
 * aws-autoscaling -> aws-cloudwatch-actions (for using the Action)
 * aws-cloudwatch-actions -> aws-autoscaling (for the definition of IStepScalingAction)
 */
class StepScalingAlarmAction {
    constructor(stepScalingAction) {
        this.stepScalingAction = stepScalingAction;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.stepScalingAction.scalingPolicyArn };
    }
}
//# sourceMappingURL=data:application/json;base64,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