"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceDiscovery = exports.DnsResponseType = exports.IpPreference = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Enum of supported IP preferences.
 * Used to dictate the IP version for mesh wide and virtual node service discovery.
 * Also used to specify the IP version that a sidecar Envoy uses when sending traffic to a local application.
 */
var IpPreference;
(function (IpPreference) {
    /**
     * Use IPv4 when sending traffic to a local application.
     * Only use IPv4 for service discovery.
     */
    IpPreference["IPV4_ONLY"] = "IPv4_ONLY";
    /**
     * Use IPv4 when sending traffic to a local application.
     * First attempt to use IPv4 and fall back to IPv6 for service discovery.
     */
    IpPreference["IPV4_PREFERRED"] = "IPv4_PREFERRED";
    /**
     * Use IPv6 when sending traffic to a local application.
     * Only use IPv6 for service discovery.
     */
    IpPreference["IPV6_ONLY"] = "IPv6_ONLY";
    /**
     * Use IPv6 when sending traffic to a local application.
     * First attempt to use IPv6 and fall back to IPv4 for service discovery.
     */
    IpPreference["IPV6_PREFERRED"] = "IPv6_PREFERRED";
})(IpPreference = exports.IpPreference || (exports.IpPreference = {}));
/**
 * Enum of DNS service discovery response type
 */
var DnsResponseType;
(function (DnsResponseType) {
    /**
     * DNS resolver returns a loadbalanced set of endpoints and the traffic would be sent to the given endpoints.
     * It would not drain existing connections to other endpoints that are not part of this list.
     */
    DnsResponseType["LOAD_BALANCER"] = "LOADBALANCER";
    /**
     * DNS resolver is returning all the endpoints.
     * This also means that if an endpoint is missing, it would drain the current connections to the missing endpoint.
     */
    DnsResponseType["ENDPOINTS"] = "ENDPOINTS";
})(DnsResponseType = exports.DnsResponseType || (exports.DnsResponseType = {}));
/**
 * Provides the Service Discovery method a VirtualNode uses
 */
class ServiceDiscovery {
    /**
     * Returns DNS based service discovery
     *
     * @param hostname
     * @param responseType Specifies the DNS response type for the virtual node.
     *  The default is `DnsResponseType.LOAD_BALANCER`.
     * @param ipPreference No IP preference is applied to the Virtual Node.
     */
    static dns(hostname, responseType, ipPreference) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_appmesh_DnsResponseType(responseType);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_appmesh_IpPreference(ipPreference);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.dns);
            }
            throw error;
        }
        return new DnsServiceDiscovery(hostname, responseType, ipPreference);
    }
    /**
     * Returns Cloud Map based service discovery
     *
     * @param service The AWS Cloud Map Service to use for service discovery
     * @param instanceAttributes A string map that contains attributes with values that you can use to
     *  filter instances by any custom attribute that you specified when you
     *  registered the instance. Only instances that match all of the specified
     *  key/value pairs will be returned.
     * @param ipPreference No IP preference is applied to the Virtual Node.
     */
    static cloudMap(service, instanceAttributes, ipPreference) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_servicediscovery_IService(service);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_appmesh_IpPreference(ipPreference);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.cloudMap);
            }
            throw error;
        }
        return new CloudMapServiceDiscovery(service, instanceAttributes, ipPreference);
    }
}
_a = JSII_RTTI_SYMBOL_1;
ServiceDiscovery[_a] = { fqn: "aws-cdk-lib.aws_appmesh.ServiceDiscovery", version: "2.74.0" };
exports.ServiceDiscovery = ServiceDiscovery;
class DnsServiceDiscovery extends ServiceDiscovery {
    constructor(hostname, responseType, ipPreference) {
        super();
        this.hostname = hostname;
        this.responseType = responseType;
        this.ipPreference = ipPreference;
    }
    bind(_scope) {
        return {
            dns: {
                hostname: this.hostname,
                responseType: this.responseType,
                ipPreference: this.ipPreference,
            },
        };
    }
}
class CloudMapServiceDiscovery extends ServiceDiscovery {
    constructor(service, instanceAttributes, ipPreference) {
        super();
        this.service = service;
        this.instanceAttributes = instanceAttributes;
        this.ipPreference = ipPreference;
    }
    bind(_scope) {
        return {
            cloudmap: {
                namespaceName: this.service.namespace.namespaceName,
                serviceName: this.service.serviceName,
                attributes: renderAttributes(this.instanceAttributes),
                ipPreference: this.ipPreference,
            },
        };
    }
}
function renderAttributes(attrs) {
    if (attrs === undefined) {
        return undefined;
    }
    return Object.entries(attrs).map(([key, value]) => ({ key, value }));
}
//# sourceMappingURL=data:application/json;base64,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