"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlApiBase = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const data_source_1 = require("./data-source");
const resolver_1 = require("./resolver");
/**
 * Base Class for GraphQL API
 */
class GraphqlApiBase extends core_1.Resource {
    /**
     * add a new dummy data source to this API. Useful for pipeline resolvers
     * and for backend changes that don't require a data source.
     *
     * @param id The data source's id
     * @param options The optional configuration for this data source
     */
    addNoneDataSource(id, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_appsync_DataSourceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addNoneDataSource);
            }
            throw error;
        }
        return new data_source_1.NoneDataSource(this, id, {
            api: this,
            name: options?.name,
            description: options?.description,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     *
     * @param id The data source's id
     * @param table The DynamoDB table backing this data source
     * @param options The optional configuration for this data source
     */
    addDynamoDbDataSource(id, table, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_dynamodb_ITable(table);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_appsync_DataSourceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addDynamoDbDataSource);
            }
            throw error;
        }
        return new data_source_1.DynamoDbDataSource(this, id, {
            api: this,
            table,
            name: options?.name,
            description: options?.description,
        });
    }
    /**
     * add a new http data source to this API
     *
     * @param id The data source's id
     * @param endpoint The http endpoint
     * @param options The optional configuration for this data source
     */
    addHttpDataSource(id, endpoint, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_appsync_HttpDataSourceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addHttpDataSource);
            }
            throw error;
        }
        return new data_source_1.HttpDataSource(this, id, {
            api: this,
            endpoint,
            name: options?.name,
            description: options?.description,
            authorizationConfig: options?.authorizationConfig,
        });
    }
    /**
     * add a new Lambda data source to this API
     *
     * @param id The data source's id
     * @param lambdaFunction The Lambda function to call to interact with this data source
     * @param options The optional configuration for this data source
     */
    addLambdaDataSource(id, lambdaFunction, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_IFunction(lambdaFunction);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_appsync_DataSourceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLambdaDataSource);
            }
            throw error;
        }
        return new data_source_1.LambdaDataSource(this, id, {
            api: this,
            lambdaFunction,
            name: options?.name,
            description: options?.description,
        });
    }
    /**
     * add a new Rds data source to this API
     * @param id The data source's id
     * @param serverlessCluster The serverless cluster to interact with this data source
     * @param secretStore The secret store that contains the username and password for the serverless cluster
     * @param databaseName The optional name of the database to use within the cluster
     * @param options The optional configuration for this data source
     */
    addRdsDataSource(id, serverlessCluster, secretStore, databaseName, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_IServerlessCluster(serverlessCluster);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_ISecret(secretStore);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_appsync_DataSourceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRdsDataSource);
            }
            throw error;
        }
        return new data_source_1.RdsDataSource(this, id, {
            api: this,
            name: options?.name,
            description: options?.description,
            serverlessCluster,
            secretStore,
            databaseName,
        });
    }
    /**
     * add a new elasticsearch data source to this API
     *
     * @deprecated - use `addOpenSearchDataSource`
     * @param id The data source's id
     * @param domain The elasticsearch domain for this data source
     * @param options The optional configuration for this data source
     */
    addElasticsearchDataSource(id, domain, options) {
        try {
            jsiiDeprecationWarnings.print("aws-cdk-lib.aws_appsync.GraphqlApiBase#addElasticsearchDataSource", "- use `addOpenSearchDataSource`");
            jsiiDeprecationWarnings.aws_cdk_lib_aws_elasticsearch_IDomain(domain);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_appsync_DataSourceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addElasticsearchDataSource);
            }
            throw error;
        }
        return new data_source_1.ElasticsearchDataSource(this, id, {
            api: this,
            name: options?.name,
            description: options?.description,
            domain,
        });
    }
    /**
     * add a new OpenSearch data source to this API
     *
     * @param id The data source's id
     * @param domain The OpenSearch domain for this data source
     * @param options The optional configuration for this data source
     */
    addOpenSearchDataSource(id, domain, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_opensearchservice_IDomain(domain);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_appsync_DataSourceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addOpenSearchDataSource);
            }
            throw error;
        }
        return new data_source_1.OpenSearchDataSource(this, id, {
            api: this,
            name: options?.name,
            description: options?.description,
            domain,
        });
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(id, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_appsync_ExtendedResolverProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.createResolver);
            }
            throw error;
        }
        return new resolver_1.Resolver(this, id, {
            api: this,
            ...props,
        });
    }
    /**
     * Add schema dependency if not imported
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_CfnResource(construct);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSchemaDependency);
            }
            throw error;
        }
        construct;
        return false;
    }
}
_a = JSII_RTTI_SYMBOL_1;
GraphqlApiBase[_a] = { fqn: "aws-cdk-lib.aws_appsync.GraphqlApiBase", version: "2.74.0" };
exports.GraphqlApiBase = GraphqlApiBase;
//# sourceMappingURL=data:application/json;base64,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