"use strict";
var _a, _b, _c, _d, _e, _f, _g;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = exports.ScalingProcess = exports.ScalingEvents = exports.ScalingEvent = exports.UpdateType = exports.AutoScalingGroup = exports.GroupMetric = exports.GroupMetrics = exports.UpdatePolicy = exports.Signals = exports.SpotAllocationStrategy = exports.OnDemandAllocationStrategy = exports.Monitoring = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const elbv2 = require("../../aws-elasticloadbalancingv2");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const aspects_1 = require("./aspects");
const autoscaling_generated_1 = require("./autoscaling.generated");
const lifecycle_hook_1 = require("./lifecycle-hook");
const scheduled_action_1 = require("./scheduled-action");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
const volume_1 = require("./volume");
const warm_pool_1 = require("./warm-pool");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The monitoring mode for instances launched in an autoscaling group
 */
var Monitoring;
(function (Monitoring) {
    /**
     * Generates metrics every 5 minutes
     */
    Monitoring[Monitoring["BASIC"] = 0] = "BASIC";
    /**
     * Generates metrics every minute
     */
    Monitoring[Monitoring["DETAILED"] = 1] = "DETAILED";
})(Monitoring = exports.Monitoring || (exports.Monitoring = {}));
/**
 * Indicates how to allocate instance types to fulfill On-Demand capacity.
 */
var OnDemandAllocationStrategy;
(function (OnDemandAllocationStrategy) {
    /**
     * This strategy uses the order of instance types in the LaunchTemplateOverrides to define the launch
     * priority of each instance type. The first instance type in the array is prioritized higher than the
     * last. If all your On-Demand capacity cannot be fulfilled using your highest priority instance, then
     * the Auto Scaling group launches the remaining capacity using the second priority instance type, and
     * so on.
     */
    OnDemandAllocationStrategy["PRIORITIZED"] = "prioritized";
})(OnDemandAllocationStrategy = exports.OnDemandAllocationStrategy || (exports.OnDemandAllocationStrategy = {}));
/**
 * Indicates how to allocate instance types to fulfill Spot capacity.
 */
var SpotAllocationStrategy;
(function (SpotAllocationStrategy) {
    /**
     * The Auto Scaling group launches instances using the Spot pools with the lowest price, and evenly
     * allocates your instances across the number of Spot pools that you specify.
     */
    SpotAllocationStrategy["LOWEST_PRICE"] = "lowest-price";
    /**
     * The Auto Scaling group launches instances using Spot pools that are optimally chosen based on the
     * available Spot capacity.
     *
     * Recommended.
     */
    SpotAllocationStrategy["CAPACITY_OPTIMIZED"] = "capacity-optimized";
    /**
     * When you use this strategy, you need to set the order of instance types in the list of launch template
     * overrides from highest to lowest priority (from first to last in the list). Amazon EC2 Auto Scaling
     * honors the instance type priorities on a best-effort basis but optimizes for capacity first.
     */
    SpotAllocationStrategy["CAPACITY_OPTIMIZED_PRIORITIZED"] = "capacity-optimized-prioritized";
    /**
     * The price and capacity optimized allocation strategy looks at both price and
     * capacity to select the Spot Instance pools that are the least likely to be
     * interrupted and have the lowest possible price.
     */
    SpotAllocationStrategy["PRICE_CAPACITY_OPTIMIZED"] = "price-capacity-optimized";
})(SpotAllocationStrategy = exports.SpotAllocationStrategy || (exports.SpotAllocationStrategy = {}));
/**
 * Configure whether the AutoScalingGroup waits for signals
 *
 * If you do configure waiting for signals, you should make sure the instances
 * invoke `cfn-signal` somewhere in their UserData to signal that they have
 * started up (either successfully or unsuccessfully).
 *
 * Signals are used both during intial creation and subsequent updates.
 */
class Signals {
    /**
     * Wait for the desiredCapacity of the AutoScalingGroup amount of signals to have been received
     *
     * If no desiredCapacity has been configured, wait for minCapacity signals intead.
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     */
    static waitForAll(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_SignalsOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.waitForAll);
            }
            throw error;
        }
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy(renderOptions) {
                return this.doRender(options, renderOptions.desiredCapacity ?? renderOptions.minCapacity);
            }
        }();
    }
    /**
     * Wait for the minCapacity of the AutoScalingGroup amount of signals to have been received
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     */
    static waitForMinCapacity(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_SignalsOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.waitForMinCapacity);
            }
            throw error;
        }
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy(renderOptions) {
                return this.doRender(options, renderOptions.minCapacity);
            }
        }();
    }
    /**
     * Wait for a specific amount of signals to have been received
     *
     * You should send one signal per instance, so this represents the number of
     * instances to wait for.
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     */
    static waitForCount(count, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_SignalsOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.waitForCount);
            }
            throw error;
        }
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy() {
                return this.doRender(options, count);
            }
        }();
    }
    /**
     * Helper to render the actual creation policy, as the logic between them is quite similar
     */
    doRender(options, count) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_SignalsOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.doRender);
            }
            throw error;
        }
        const minSuccessfulInstancesPercent = validatePercentage(options.minSuccessPercentage);
        return {
            ...options.minSuccessPercentage !== undefined ? { autoScalingCreationPolicy: { minSuccessfulInstancesPercent } } : {},
            resourceSignal: {
                count,
                timeout: options.timeout?.toIsoString(),
            },
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
Signals[_a] = { fqn: "aws-cdk-lib.aws_autoscaling.Signals", version: "2.74.0" };
exports.Signals = Signals;
/**
 * How existing instances should be updated
 */
class UpdatePolicy {
    /**
     * Create a new AutoScalingGroup and switch over to it
     */
    static replacingUpdate() {
        return new class extends UpdatePolicy {
            _renderUpdatePolicy() {
                return {
                    autoScalingReplacingUpdate: { willReplace: true },
                };
            }
        }();
    }
    /**
     * Replace the instances in the AutoScalingGroup one by one, or in batches
     */
    static rollingUpdate(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_RollingUpdateOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.rollingUpdate);
            }
            throw error;
        }
        const minSuccessPercentage = validatePercentage(options.minSuccessPercentage);
        return new class extends UpdatePolicy {
            _renderUpdatePolicy(renderOptions) {
                return {
                    autoScalingRollingUpdate: {
                        maxBatchSize: options.maxBatchSize,
                        minInstancesInService: options.minInstancesInService,
                        suspendProcesses: options.suspendProcesses ?? DEFAULT_SUSPEND_PROCESSES,
                        minSuccessfulInstancesPercent: minSuccessPercentage ?? renderOptions.creationPolicy?.autoScalingCreationPolicy?.minSuccessfulInstancesPercent,
                        waitOnResourceSignals: options.waitOnResourceSignals ?? renderOptions.creationPolicy?.resourceSignal !== undefined ? true : undefined,
                        pauseTime: options.pauseTime?.toIsoString() ?? renderOptions.creationPolicy?.resourceSignal?.timeout,
                    },
                };
            }
        }();
    }
}
_b = JSII_RTTI_SYMBOL_1;
UpdatePolicy[_b] = { fqn: "aws-cdk-lib.aws_autoscaling.UpdatePolicy", version: "2.74.0" };
exports.UpdatePolicy = UpdatePolicy;
/**
 * A set of group metrics
 */
class GroupMetrics {
    /**
     * Report all group metrics.
     */
    static all() {
        return new GroupMetrics();
    }
    constructor(...metrics) {
        /**
         * @internal
         */
        this._metrics = new Set();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_GroupMetric(metrics);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GroupMetrics);
            }
            throw error;
        }
        metrics?.forEach(metric => this._metrics.add(metric));
    }
}
_c = JSII_RTTI_SYMBOL_1;
GroupMetrics[_c] = { fqn: "aws-cdk-lib.aws_autoscaling.GroupMetrics", version: "2.74.0" };
exports.GroupMetrics = GroupMetrics;
/**
 * Group metrics that an Auto Scaling group sends to Amazon CloudWatch.
 */
class GroupMetric {
    constructor(name) {
        this.name = name;
    }
}
_d = JSII_RTTI_SYMBOL_1;
GroupMetric[_d] = { fqn: "aws-cdk-lib.aws_autoscaling.GroupMetric", version: "2.74.0" };
/**
 * The minimum size of the Auto Scaling group
 */
GroupMetric.MIN_SIZE = new GroupMetric('GroupMinSize');
/**
 * The maximum size of the Auto Scaling group
 */
GroupMetric.MAX_SIZE = new GroupMetric('GroupMaxSize');
/**
 * The number of instances that the Auto Scaling group attempts to maintain
 */
GroupMetric.DESIRED_CAPACITY = new GroupMetric('GroupDesiredCapacity');
/**
 * The number of instances that are running as part of the Auto Scaling group
 * This metric does not include instances that are pending or terminating
 */
GroupMetric.IN_SERVICE_INSTANCES = new GroupMetric('GroupInServiceInstances');
/**
 * The number of instances that are pending
 * A pending instance is not yet in service, this metric does not include instances that are in service or terminating
 */
GroupMetric.PENDING_INSTANCES = new GroupMetric('GroupPendingInstances');
/**
 * The number of instances that are in a Standby state
 * Instances in this state are still running but are not actively in service
 */
GroupMetric.STANDBY_INSTANCES = new GroupMetric('GroupStandbyInstances');
/**
 * The number of instances that are in the process of terminating
 * This metric does not include instances that are in service or pending
 */
GroupMetric.TERMINATING_INSTANCES = new GroupMetric('GroupTerminatingInstances');
/**
 * The total number of instances in the Auto Scaling group
 * This metric identifies the number of instances that are in service, pending, and terminating
 */
GroupMetric.TOTAL_INSTANCES = new GroupMetric('GroupTotalInstances');
exports.GroupMetric = GroupMetric;
class AutoScalingGroupBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
        this.hasCalledScaleOnRequestCount = false;
    }
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id, props) {
        return new lifecycle_hook_1.LifecycleHook(this, `LifecycleHook${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Add a pool of pre-initialized EC2 instances that sits alongside an Auto Scaling group
     */
    addWarmPool(options) {
        return new warm_pool_1.WarmPool(this, 'WarmPool', {
            autoScalingGroup: this,
            ...options,
        });
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, props) {
        return new scheduled_action_1.ScheduledAction(this, `ScheduledAction${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_IN,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_OUT,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target request handling rate
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     */
    scaleOnRequestCount(id, props) {
        if (this.albTargetGroup === undefined) {
            throw new Error('Attach the AutoScalingGroup to a non-imported Application Load Balancer before calling scaleOnRequestCount()');
        }
        const resourceLabel = `${this.albTargetGroup.firstLoadBalancerFullName}/${this.albTargetGroup.targetGroupFullName}`;
        if ((props.targetRequestsPerMinute === undefined) === (props.targetRequestsPerSecond === undefined)) {
            throw new Error('Specify exactly one of \'targetRequestsPerMinute\' or \'targetRequestsPerSecond\'');
        }
        let rpm;
        if (props.targetRequestsPerSecond !== undefined) {
            if (core_1.Token.isUnresolved(props.targetRequestsPerSecond)) {
                throw new Error('\'targetRequestsPerSecond\' cannot be an unresolved value; use \'targetRequestsPerMinute\' instead.');
            }
            rpm = props.targetRequestsPerSecond * 60;
        }
        else {
            rpm = props.targetRequestsPerMinute;
        }
        const policy = new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            targetValue: rpm,
            resourceLabel,
            ...props,
        });
        policy.node.addDependency(this.albTargetGroup.loadBalancerAttached);
        this.hasCalledScaleOnRequestCount = true;
        return policy;
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            customMetric: props.metric,
            ...props,
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, autoScalingGroup: this });
    }
    addUserData(..._commands) {
    }
}
/**
 * A Fleet represents a managed set of EC2 instances
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 */
class AutoScalingGroup extends AutoScalingGroupBase {
    static fromAutoScalingGroupName(scope, id, autoScalingGroupName) {
        class Import extends AutoScalingGroupBase {
            constructor() {
                super(...arguments);
                this.autoScalingGroupName = autoScalingGroupName;
                this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                    service: 'autoscaling',
                    resource: 'autoScalingGroup:*:autoScalingGroupName',
                    resourceName: this.autoScalingGroupName,
                });
                this.osType = ec2.OperatingSystemType.UNKNOWN;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.autoScalingGroupName,
        });
        this.loadBalancerNames = [];
        this.targetGroupArns = [];
        this.groupMetrics = [];
        this.notifications = [];
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_AutoScalingGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AutoScalingGroup);
            }
            throw error;
        }
        this.newInstancesProtectedFromScaleIn = props.newInstancesProtectedFromScaleIn;
        if (props.initOptions && !props.init) {
            throw new Error('Setting \'initOptions\' requires that \'init\' is also set');
        }
        if (props.groupMetrics) {
            this.groupMetrics.push(...props.groupMetrics);
        }
        let launchConfig = undefined;
        if (props.launchTemplate || props.mixedInstancesPolicy) {
            this.verifyNoLaunchConfigPropIsGiven(props);
            const bareLaunchTemplate = props.launchTemplate;
            const mixedInstancesPolicy = props.mixedInstancesPolicy;
            if (bareLaunchTemplate && mixedInstancesPolicy) {
                throw new Error('Setting \'mixedInstancesPolicy\' must not be set when \'launchTemplate\' is set');
            }
            if (bareLaunchTemplate && bareLaunchTemplate instanceof ec2.LaunchTemplate) {
                if (!bareLaunchTemplate.instanceType) {
                    throw new Error('Setting \'launchTemplate\' requires its \'instanceType\' to be set');
                }
                if (!bareLaunchTemplate.imageId) {
                    throw new Error('Setting \'launchTemplate\' requires its \'machineImage\' to be set');
                }
                this.launchTemplate = bareLaunchTemplate;
            }
            if (mixedInstancesPolicy && mixedInstancesPolicy.launchTemplate instanceof ec2.LaunchTemplate) {
                if (!mixedInstancesPolicy.launchTemplate.imageId) {
                    throw new Error('Setting \'mixedInstancesPolicy.launchTemplate\' requires its \'machineImage\' to be set');
                }
                this.launchTemplate = mixedInstancesPolicy.launchTemplate;
            }
            this._role = this.launchTemplate?.role;
            this.grantPrincipal = this._role || new iam.UnknownPrincipal({ resource: this });
            this.osType = this.launchTemplate?.osType ?? ec2.OperatingSystemType.UNKNOWN;
        }
        else {
            if (!props.machineImage) {
                throw new Error('Setting \'machineImage\' is required when \'launchTemplate\' and \'mixedInstancesPolicy\' is not set');
            }
            if (!props.instanceType) {
                throw new Error('Setting \'instanceType\' is required when \'launchTemplate\' and \'mixedInstancesPolicy\' is not set');
            }
            this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'InstanceSecurityGroup', {
                vpc: props.vpc,
                allowAllOutbound: props.allowAllOutbound !== false,
            });
            this._connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
            this.securityGroups = [this.securityGroup];
            core_1.Tags.of(this).add(NAME_TAG, this.node.path);
            this._role = props.role || new iam.Role(this, 'InstanceRole', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            });
            this.grantPrincipal = this._role;
            if (props.ssmSessionPermissions) {
                this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
            }
            const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
                roles: [this.role.roleName],
            });
            // use delayed evaluation
            const imageConfig = props.machineImage.getImage(this);
            this._userData = props.userData ?? imageConfig.userData;
            const userDataToken = core_1.Lazy.string({ produce: () => core_1.Fn.base64(this.userData.render()) });
            const securityGroupsToken = core_1.Lazy.list({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
            launchConfig = new autoscaling_generated_1.CfnLaunchConfiguration(this, 'LaunchConfig', {
                imageId: imageConfig.imageId,
                keyName: props.keyName,
                instanceType: props.instanceType.toString(),
                instanceMonitoring: (props.instanceMonitoring !== undefined ? (props.instanceMonitoring === Monitoring.DETAILED) : undefined),
                securityGroups: securityGroupsToken,
                iamInstanceProfile: iamProfile.ref,
                userData: userDataToken,
                associatePublicIpAddress: props.associatePublicIpAddress,
                spotPrice: props.spotPrice,
                blockDeviceMappings: (props.blockDevices !== undefined ?
                    synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined),
            });
            launchConfig.node.addDependency(this.role);
            this.osType = imageConfig.osType;
        }
        // desiredCapacity just reflects what the user has supplied.
        const desiredCapacity = props.desiredCapacity;
        const minCapacity = props.minCapacity ?? 1;
        const maxCapacity = props.maxCapacity ?? desiredCapacity ?? Math.max(minCapacity, 1);
        (0, core_1.withResolved)(minCapacity, maxCapacity, (min, max) => {
            if (min > max) {
                throw new Error(`minCapacity (${min}) should be <= maxCapacity (${max})`);
            }
        });
        (0, core_1.withResolved)(desiredCapacity, minCapacity, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Should have minCapacity (${min}) <= desiredCapacity (${desired})`);
            }
        });
        (0, core_1.withResolved)(desiredCapacity, maxCapacity, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (max < desired) {
                throw new Error(`Should have desiredCapacity (${desired}) <= maxCapacity (${max})`);
            }
        });
        if (desiredCapacity !== undefined) {
            core_1.Annotations.of(this).addWarning('desiredCapacity has been configured. Be aware this will reset the size of your AutoScalingGroup on every deployment. See https://github.com/aws/aws-cdk/issues/5215');
        }
        this.maxInstanceLifetime = props.maxInstanceLifetime;
        // See https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html for details on max instance lifetime.
        if (this.maxInstanceLifetime && !this.maxInstanceLifetime.isUnresolved() &&
            (this.maxInstanceLifetime.toSeconds() !== 0) &&
            (this.maxInstanceLifetime.toSeconds() < 86400 || this.maxInstanceLifetime.toSeconds() > 31536000)) {
            throw new Error('maxInstanceLifetime must be between 1 and 365 days (inclusive)');
        }
        if (props.notificationsTopic && props.notifications) {
            throw new Error('Cannot set \'notificationsTopic\' and \'notifications\', \'notificationsTopic\' is deprecated use \'notifications\' instead');
        }
        if (props.notificationsTopic) {
            this.notifications = [{
                    topic: props.notificationsTopic,
                }];
        }
        if (props.notifications) {
            this.notifications = props.notifications.map(nc => ({
                topic: nc.topic,
                scalingEvents: nc.scalingEvents ?? ScalingEvents.ALL,
            }));
        }
        const { subnetIds, hasPublic } = props.vpc.selectSubnets(props.vpcSubnets);
        const asgProps = {
            autoScalingGroupName: this.physicalName,
            cooldown: props.cooldown?.toSeconds().toString(),
            minSize: core_1.Tokenization.stringifyNumber(minCapacity),
            maxSize: core_1.Tokenization.stringifyNumber(maxCapacity),
            desiredCapacity: desiredCapacity !== undefined ? core_1.Tokenization.stringifyNumber(desiredCapacity) : undefined,
            loadBalancerNames: core_1.Lazy.list({ produce: () => this.loadBalancerNames }, { omitEmpty: true }),
            targetGroupArns: core_1.Lazy.list({ produce: () => this.targetGroupArns }, { omitEmpty: true }),
            notificationConfigurations: this.renderNotificationConfiguration(),
            metricsCollection: core_1.Lazy.any({ produce: () => this.renderMetricsCollection() }),
            vpcZoneIdentifier: subnetIds,
            healthCheckType: props.healthCheck && props.healthCheck.type,
            healthCheckGracePeriod: props.healthCheck && props.healthCheck.gracePeriod && props.healthCheck.gracePeriod.toSeconds(),
            maxInstanceLifetime: this.maxInstanceLifetime ? this.maxInstanceLifetime.toSeconds() : undefined,
            newInstancesProtectedFromScaleIn: core_1.Lazy.any({ produce: () => this.newInstancesProtectedFromScaleIn }),
            terminationPolicies: props.terminationPolicies,
            defaultInstanceWarmup: props.defaultInstanceWarmup?.toSeconds(),
            capacityRebalance: props.capacityRebalance,
            ...this.getLaunchSettings(launchConfig, props.launchTemplate, props.mixedInstancesPolicy),
        };
        if (!hasPublic && props.associatePublicIpAddress) {
            throw new Error("To set 'associatePublicIpAddress: true' you must select Public subnets (vpcSubnets: { subnetType: SubnetType.PUBLIC })");
        }
        this.autoScalingGroup = new autoscaling_generated_1.CfnAutoScalingGroup(this, 'ASG', asgProps);
        this.autoScalingGroupName = this.getResourceNameAttribute(this.autoScalingGroup.ref),
            this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                service: 'autoscaling',
                resource: 'autoScalingGroup:*:autoScalingGroupName',
                resourceName: this.autoScalingGroupName,
            });
        this.node.defaultChild = this.autoScalingGroup;
        this.applyUpdatePolicies(props, { desiredCapacity, minCapacity });
        if (props.init) {
            this.applyCloudFormationInit(props.init, props.initOptions);
        }
        this.spotPrice = props.spotPrice;
        if (props.requireImdsv2) {
            core_1.Aspects.of(this).add(new aspects_1.AutoScalingGroupRequireImdsv2Aspect());
        }
        this.node.addValidation({ validate: () => this.validateTargetGroup() });
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_ISecurityGroup(securityGroup);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSecurityGroup);
            }
            throw error;
        }
        if (!this.securityGroups) {
            throw new Error('You cannot add security groups when the Auto Scaling Group is created from a Launch Template.');
        }
        this.securityGroups.push(securityGroup);
    }
    /**
     * Attach to a classic load balancer
     */
    attachToClassicLB(loadBalancer) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_elasticloadbalancing_LoadBalancer(loadBalancer);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.attachToClassicLB);
            }
            throw error;
        }
        this.loadBalancerNames.push(loadBalancer.loadBalancerName);
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToApplicationTargetGroup(targetGroup) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_elasticloadbalancingv2_IApplicationTargetGroup(targetGroup);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.attachToApplicationTargetGroup);
            }
            throw error;
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        if (targetGroup instanceof elbv2.ApplicationTargetGroup) {
            // Copy onto self if it's a concrete type. We need this for autoscaling
            // based on request count, which we cannot do with an imported TargetGroup.
            this.albTargetGroup = targetGroup;
        }
        targetGroup.registerConnectable(this);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToNetworkTargetGroup(targetGroup) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_elasticloadbalancingv2_INetworkTargetGroup(targetGroup);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.attachToNetworkTargetGroup);
            }
            throw error;
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by instances of this fleet.
     */
    addToRolePolicy(statement) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToRolePolicy);
            }
            throw error;
        }
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Use a CloudFormation Init configuration at instance startup
     *
     * This does the following:
     *
     * - Attaches the CloudFormation Init metadata to the AutoScalingGroup resource.
     * - Add commands to the UserData to run `cfn-init` and `cfn-signal`.
     * - Update the instance's CreationPolicy to wait for `cfn-init` to finish
     *   before reporting success.
     */
    applyCloudFormationInit(init, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_CloudFormationInit(init);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_ApplyCloudFormationInitOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.applyCloudFormationInit);
            }
            throw error;
        }
        if (!this.autoScalingGroup.cfnOptions.creationPolicy?.resourceSignal) {
            throw new Error('When applying CloudFormationInit, you must also configure signals by supplying \'signals\' at instantiation time.');
        }
        init.attach(this.autoScalingGroup, {
            platform: this.osType,
            instanceRole: this.role,
            userData: this.userData,
            configSets: options.configSets,
            embedFingerprint: options.embedFingerprint,
            printLog: options.printLog,
            ignoreFailures: options.ignoreFailures,
            includeRole: options.includeRole,
            includeUrl: options.includeUrl,
        });
    }
    /**
     * Ensures newly-launched instances are protected from scale-in.
     */
    protectNewInstancesFromScaleIn() {
        this.newInstancesProtectedFromScaleIn = true;
    }
    /**
     * Returns `true` if newly-launched instances are protected from scale-in.
     */
    areNewInstancesProtectedFromScaleIn() {
        return this.newInstancesProtectedFromScaleIn === true;
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        if (this._connections) {
            return this._connections;
        }
        if (this.launchTemplate) {
            return this.launchTemplate.connections;
        }
        throw new Error('AutoScalingGroup can only be used as IConnectable if it is not created from an imported Launch Template.');
    }
    /**
     * The Base64-encoded user data to make available to the launched EC2 instances.
     *
     * @throws an error if a launch template is given and it does not provide a non-null `userData`
     */
    get userData() {
        if (this._userData) {
            return this._userData;
        }
        if (this.launchTemplate?.userData) {
            return this.launchTemplate.userData;
        }
        throw new Error('The provided launch template does not expose its user data.');
    }
    /**
     * The IAM Role in the instance profile
     *
     * @throws an error if a launch template is given
     */
    get role() {
        if (this._role) {
            return this._role;
        }
        throw new Error('The provided launch template does not expose or does not define its role.');
    }
    verifyNoLaunchConfigPropIsGiven(props) {
        if (props.machineImage) {
            throw new Error('Setting \'machineImage\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.instanceType) {
            throw new Error('Setting \'instanceType\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.role) {
            throw new Error('Setting \'role\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.userData) {
            throw new Error('Setting \'userData\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.securityGroup) {
            throw new Error('Setting \'securityGroup\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.keyName) {
            throw new Error('Setting \'keyName\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.instanceMonitoring) {
            throw new Error('Setting \'instanceMonitoring\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.associatePublicIpAddress !== undefined) {
            throw new Error('Setting \'associatePublicIpAddress\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.spotPrice) {
            throw new Error('Setting \'spotPrice\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.blockDevices) {
            throw new Error('Setting \'blockDevices\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
    }
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    applyUpdatePolicies(props, signalOptions) {
        // Make sure people are not using the old and new properties together
        const oldProps = [
            'updateType',
            'rollingUpdateConfiguration',
            'resourceSignalCount',
            'resourceSignalTimeout',
            'replacingUpdateMinSuccessfulInstancesPercent',
        ];
        for (const prop of oldProps) {
            if ((props.signals || props.updatePolicy) && props[prop] !== undefined) {
                throw new Error(`Cannot set 'signals'/'updatePolicy' and '${prop}' together. Prefer 'signals'/'updatePolicy'`);
            }
        }
        // Reify updatePolicy to `rollingUpdate` default in case it is combined with `init`
        props = {
            ...props,
            updatePolicy: props.updatePolicy ?? (props.init ? UpdatePolicy.rollingUpdate() : undefined),
        };
        if (props.signals || props.updatePolicy) {
            this.applyNewSignalUpdatePolicies(props, signalOptions);
        }
        else {
            this.applyLegacySignalUpdatePolicies(props);
        }
        // The following is technically part of the "update policy" but it's also a completely
        // separate aspect of rolling/replacing update, so it's just its own top-level property.
        // Default is 'true' because that's what you're most likely to want
        if (props.ignoreUnmodifiedSizeProperties !== false) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: true },
            };
        }
        if (props.signals && !props.init) {
            // To be able to send a signal using `cfn-init`, the execution role needs
            // `cloudformation:SignalResource`. Normally the binding of CfnInit would
            // grant that permissions and another one, but if the user wants to use
            // `signals` without `init`, add the permissions here.
            //
            // If they call `applyCloudFormationInit()` after construction, nothing bad
            // happens either, we'll just have a duplicate statement which doesn't hurt.
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: ['cloudformation:SignalResource'],
                resources: [core_1.Aws.STACK_ID],
            }));
        }
    }
    /**
     * Use 'signals' and 'updatePolicy' to determine the creation and update policies
     */
    applyNewSignalUpdatePolicies(props, signalOptions) {
        this.autoScalingGroup.cfnOptions.creationPolicy = props.signals?.renderCreationPolicy(signalOptions);
        this.autoScalingGroup.cfnOptions.updatePolicy = props.updatePolicy?._renderUpdatePolicy({
            creationPolicy: this.autoScalingGroup.cfnOptions.creationPolicy,
        });
    }
    applyLegacySignalUpdatePolicies(props) {
        if (props.updateType === UpdateType.REPLACING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingReplacingUpdate: {
                    willReplace: true,
                },
            };
            if (props.replacingUpdateMinSuccessfulInstancesPercent !== undefined) {
                // Yes, this goes on CreationPolicy, not as a process parameter to ReplacingUpdate.
                // It's a little confusing, but the docs seem to explicitly state it will only be used
                // during the update?
                //
                // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html
                this.autoScalingGroup.cfnOptions.creationPolicy = {
                    ...this.autoScalingGroup.cfnOptions.creationPolicy,
                    autoScalingCreationPolicy: {
                        minSuccessfulInstancesPercent: validatePercentage(props.replacingUpdateMinSuccessfulInstancesPercent),
                    },
                };
            }
        }
        else if (props.updateType === UpdateType.ROLLING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingRollingUpdate: renderRollingUpdateConfig(props.rollingUpdateConfiguration),
            };
        }
        if (props.resourceSignalCount !== undefined || props.resourceSignalTimeout !== undefined) {
            this.autoScalingGroup.cfnOptions.creationPolicy = {
                ...this.autoScalingGroup.cfnOptions.creationPolicy,
                resourceSignal: {
                    count: props.resourceSignalCount,
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toIsoString(),
                },
            };
        }
    }
    renderNotificationConfiguration() {
        if (this.notifications.length === 0) {
            return undefined;
        }
        return this.notifications.map(notification => ({
            topicArn: notification.topic.topicArn,
            notificationTypes: notification.scalingEvents ? notification.scalingEvents._types : ScalingEvents.ALL._types,
        }));
    }
    renderMetricsCollection() {
        if (this.groupMetrics.length === 0) {
            return undefined;
        }
        return this.groupMetrics.map(group => ({
            granularity: '1Minute',
            metrics: group._metrics?.size !== 0 ? [...group._metrics].map(m => m.name) : undefined,
        }));
    }
    getLaunchSettings(launchConfig, launchTemplate, mixedInstancesPolicy) {
        if (launchConfig) {
            return {
                launchConfigurationName: launchConfig.ref,
            };
        }
        if (launchTemplate) {
            return {
                launchTemplate: this.convertILaunchTemplateToSpecification(launchTemplate),
            };
        }
        if (mixedInstancesPolicy) {
            let instancesDistribution = undefined;
            if (mixedInstancesPolicy.instancesDistribution) {
                const dist = mixedInstancesPolicy.instancesDistribution;
                instancesDistribution = {
                    onDemandAllocationStrategy: dist.onDemandAllocationStrategy?.toString(),
                    onDemandBaseCapacity: dist.onDemandBaseCapacity,
                    onDemandPercentageAboveBaseCapacity: dist.onDemandPercentageAboveBaseCapacity,
                    spotAllocationStrategy: dist.spotAllocationStrategy?.toString(),
                    spotInstancePools: dist.spotInstancePools,
                    spotMaxPrice: dist.spotMaxPrice,
                };
            }
            return {
                mixedInstancesPolicy: {
                    instancesDistribution,
                    launchTemplate: {
                        launchTemplateSpecification: this.convertILaunchTemplateToSpecification(mixedInstancesPolicy.launchTemplate),
                        ...(mixedInstancesPolicy.launchTemplateOverrides ? {
                            overrides: mixedInstancesPolicy.launchTemplateOverrides.map(override => {
                                if (override.weightedCapacity && Math.floor(override.weightedCapacity) !== override.weightedCapacity) {
                                    throw new Error('Weight must be an integer');
                                }
                                return {
                                    instanceType: override.instanceType.toString(),
                                    launchTemplateSpecification: override.launchTemplate
                                        ? this.convertILaunchTemplateToSpecification(override.launchTemplate)
                                        : undefined,
                                    weightedCapacity: override.weightedCapacity?.toString(),
                                };
                            }),
                        } : {}),
                    },
                },
            };
        }
        throw new Error('Either launchConfig, launchTemplate or mixedInstancesPolicy needs to be specified.');
    }
    convertILaunchTemplateToSpecification(launchTemplate) {
        if (launchTemplate.launchTemplateId) {
            return {
                launchTemplateId: launchTemplate.launchTemplateId,
                version: launchTemplate.versionNumber,
            };
        }
        else {
            return {
                launchTemplateName: launchTemplate.launchTemplateName,
                version: launchTemplate.versionNumber,
            };
        }
    }
    validateTargetGroup() {
        const errors = new Array();
        if (this.hasCalledScaleOnRequestCount && this.targetGroupArns.length > 1) {
            errors.push('Cannon use multiple target groups if `scaleOnRequestCount()` is being used.');
        }
        return errors;
    }
}
_e = JSII_RTTI_SYMBOL_1;
AutoScalingGroup[_e] = { fqn: "aws-cdk-lib.aws_autoscaling.AutoScalingGroup", version: "2.74.0" };
exports.AutoScalingGroup = AutoScalingGroup;
/**
 * The type of update to perform on instances in this AutoScalingGroup
 *
 * @deprecated Use UpdatePolicy instead
 */
var UpdateType;
(function (UpdateType) {
    /**
     * Don't do anything
     */
    UpdateType["NONE"] = "None";
    /**
     * Replace the entire AutoScalingGroup
     *
     * Builds a new AutoScalingGroup first, then delete the old one.
     */
    UpdateType["REPLACING_UPDATE"] = "Replace";
    /**
     * Replace the instances in the AutoScalingGroup.
     */
    UpdateType["ROLLING_UPDATE"] = "RollingUpdate";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
/**
 * Fleet scaling events
 */
var ScalingEvent;
(function (ScalingEvent) {
    /**
     * Notify when an instance was launched
     */
    ScalingEvent["INSTANCE_LAUNCH"] = "autoscaling:EC2_INSTANCE_LAUNCH";
    /**
     * Notify when an instance was terminated
     */
    ScalingEvent["INSTANCE_TERMINATE"] = "autoscaling:EC2_INSTANCE_TERMINATE";
    /**
     * Notify when an instance failed to terminate
     */
    ScalingEvent["INSTANCE_TERMINATE_ERROR"] = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR";
    /**
     * Notify when an instance failed to launch
     */
    ScalingEvent["INSTANCE_LAUNCH_ERROR"] = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    /**
     * Send a test notification to the topic
     */
    ScalingEvent["TEST_NOTIFICATION"] = "autoscaling:TEST_NOTIFICATION";
})(ScalingEvent = exports.ScalingEvent || (exports.ScalingEvent = {}));
/**
 * A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS
 * or create a custom group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 */
class ScalingEvents {
    constructor(...types) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_ScalingEvent(types);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ScalingEvents);
            }
            throw error;
        }
        this._types = types;
    }
}
_f = JSII_RTTI_SYMBOL_1;
ScalingEvents[_f] = { fqn: "aws-cdk-lib.aws_autoscaling.ScalingEvents", version: "2.74.0" };
/**
 * Fleet scaling errors
 */
ScalingEvents.ERRORS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * All fleet scaling events
 */
ScalingEvents.ALL = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * Fleet scaling launch events
 */
ScalingEvents.LAUNCH_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR);
/**
 * Fleet termination launch events
 */
ScalingEvents.TERMINATION_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
exports.ScalingEvents = ScalingEvents;
var ScalingProcess;
(function (ScalingProcess) {
    ScalingProcess["LAUNCH"] = "Launch";
    ScalingProcess["TERMINATE"] = "Terminate";
    ScalingProcess["HEALTH_CHECK"] = "HealthCheck";
    ScalingProcess["REPLACE_UNHEALTHY"] = "ReplaceUnhealthy";
    ScalingProcess["AZ_REBALANCE"] = "AZRebalance";
    ScalingProcess["ALARM_NOTIFICATION"] = "AlarmNotification";
    ScalingProcess["SCHEDULED_ACTIONS"] = "ScheduledActions";
    ScalingProcess["ADD_TO_LOAD_BALANCER"] = "AddToLoadBalancer";
})(ScalingProcess = exports.ScalingProcess || (exports.ScalingProcess = {}));
// Recommended list of processes to suspend from here:
// https://aws.amazon.com/premiumsupport/knowledge-center/auto-scaling-group-rolling-updates/
const DEFAULT_SUSPEND_PROCESSES = [ScalingProcess.HEALTH_CHECK, ScalingProcess.REPLACE_UNHEALTHY, ScalingProcess.AZ_REBALANCE,
    ScalingProcess.ALARM_NOTIFICATION, ScalingProcess.SCHEDULED_ACTIONS];
/**
 * Health check settings
 */
class HealthCheck {
    /**
     * Use EC2 for health checks
     *
     * @param options EC2 health check options
     */
    static ec2(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_Ec2HealthCheckOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.ec2);
            }
            throw error;
        }
        return new HealthCheck(HealthCheckType.EC2, options.grace);
    }
    /**
     * Use ELB for health checks.
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options
     */
    static elb(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_ElbHealthCheckOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.elb);
            }
            throw error;
        }
        return new HealthCheck(HealthCheckType.ELB, options.grace);
    }
    constructor(type, gracePeriod) {
        this.type = type;
        this.gracePeriod = gracePeriod;
    }
}
_g = JSII_RTTI_SYMBOL_1;
HealthCheck[_g] = { fqn: "aws-cdk-lib.aws_autoscaling.HealthCheck", version: "2.74.0" };
exports.HealthCheck = HealthCheck;
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["EC2"] = "EC2";
    HealthCheckType["ELB"] = "ELB";
})(HealthCheckType || (HealthCheckType = {}));
/**
 * Render the rolling update configuration into the appropriate object
 */
function renderRollingUpdateConfig(config = {}) {
    const waitOnResourceSignals = config.minSuccessfulInstancesPercent !== undefined;
    const pauseTime = config.pauseTime || (waitOnResourceSignals ? core_1.Duration.minutes(5) : core_1.Duration.seconds(0));
    return {
        maxBatchSize: config.maxBatchSize,
        minInstancesInService: config.minInstancesInService,
        minSuccessfulInstancesPercent: validatePercentage(config.minSuccessfulInstancesPercent),
        waitOnResourceSignals,
        pauseTime: pauseTime && pauseTime.toIsoString(),
        suspendProcesses: config.suspendProcesses ?? DEFAULT_SUSPEND_PROCESSES,
    };
}
function validatePercentage(x) {
    if (x === undefined || (0 <= x && x <= 100)) {
        return x;
    }
    throw new Error(`Expected: a percentage 0..100, got: ${x}`);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (volume === volume_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
            return {
                deviceName,
                noDevice: true,
            };
        }
        if (ebs) {
            const { iops, volumeType, throughput } = ebs;
            if (throughput) {
                const throughputRange = { Min: 125, Max: 1000 };
                const { Min, Max } = throughputRange;
                if (volumeType != volume_1.EbsDeviceVolumeType.GP3) {
                    throw new Error('throughput property requires volumeType: EbsDeviceVolumeType.GP3');
                }
                if (throughput < Min || throughput > Max) {
                    throw new Error(`throughput property takes a minimum of ${Min} and a maximum of ${Max}`);
                }
                const maximumThroughputRatio = 0.25;
                if (iops) {
                    const iopsRatio = (throughput / iops);
                    if (iopsRatio > maximumThroughputRatio) {
                        throw new Error(`Throughput (MiBps) to iops ratio of ${iopsRatio} is too high; maximum is ${maximumThroughputRatio} MiBps per iops`);
                    }
                }
            }
            if (!iops) {
                if (volumeType === volume_1.EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== volume_1.EbsDeviceVolumeType.IO1) {
                core_1.Annotations.of(construct).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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