"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupPlan = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const backup_generated_1 = require("./backup.generated");
const rule_1 = require("./rule");
const selection_1 = require("./selection");
const vault_1 = require("./vault");
/**
 * A backup plan
 */
class BackupPlan extends core_1.Resource {
    /**
     * Import an existing backup plan
     */
    static fromBackupPlanId(scope, id, backupPlanId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.backupPlanId = backupPlanId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Daily with 35 day retention
     */
    static daily35DayRetention(scope, id, backupVault) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.daily35DayRetention);
            }
            throw error;
        }
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        return plan;
    }
    /**
     * Daily and monthly with 1 year retention
     */
    static dailyMonthly1YearRetention(scope, id, backupVault) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.dailyMonthly1YearRetention);
            }
            throw error;
        }
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.monthly1Year());
        return plan;
    }
    /**
     * Daily, weekly and monthly with 5 year retention
     */
    static dailyWeeklyMonthly5YearRetention(scope, id, backupVault) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.dailyWeeklyMonthly5YearRetention);
            }
            throw error;
        }
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.weekly());
        plan.addRule(rule_1.BackupPlanRule.monthly5Year());
        return plan;
    }
    /**
     * Daily, weekly and monthly with 7 year retention
     */
    static dailyWeeklyMonthly7YearRetention(scope, id, backupVault) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.dailyWeeklyMonthly7YearRetention);
            }
            throw error;
        }
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.weekly());
        plan.addRule(rule_1.BackupPlanRule.monthly7Year());
        return plan;
    }
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.rules = [];
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_BackupPlanProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BackupPlan);
            }
            throw error;
        }
        const plan = new backup_generated_1.CfnBackupPlan(this, 'Resource', {
            backupPlan: {
                advancedBackupSettings: this.advancedBackupSettings(props),
                backupPlanName: props.backupPlanName || id,
                backupPlanRule: core_1.Lazy.any({ produce: () => this.rules }, { omitEmptyArray: true }),
            },
        });
        this.backupPlanId = plan.attrBackupPlanId;
        this.backupPlanArn = plan.attrBackupPlanArn;
        this.versionId = plan.attrVersionId;
        this._backupVault = props.backupVault;
        for (const rule of props.backupPlanRules || []) {
            this.addRule(rule);
        }
        this.node.addValidation({ validate: () => this.validatePlan() });
    }
    advancedBackupSettings(props) {
        if (!props.windowsVss) {
            return undefined;
        }
        return [{
                backupOptions: {
                    WindowsVSS: 'enabled',
                },
                resourceType: 'EC2',
            }];
    }
    /**
     * Adds a rule to a plan
     *
     * @param rule the rule to add
     */
    addRule(rule) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_BackupPlanRule(rule);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRule);
            }
            throw error;
        }
        let vault;
        if (rule.props.backupVault) {
            vault = rule.props.backupVault;
        }
        else if (this._backupVault) {
            vault = this._backupVault;
        }
        else {
            this._backupVault = new vault_1.BackupVault(this, 'Vault');
            vault = this._backupVault;
        }
        this.rules.push({
            completionWindowMinutes: rule.props.completionWindow?.toMinutes(),
            lifecycle: (rule.props.deleteAfter || rule.props.moveToColdStorageAfter) && {
                deleteAfterDays: rule.props.deleteAfter?.toDays(),
                moveToColdStorageAfterDays: rule.props.moveToColdStorageAfter?.toDays(),
            },
            ruleName: rule.props.ruleName ?? `${this.node.id}Rule${this.rules.length}`,
            scheduleExpression: rule.props.scheduleExpression?.expressionString,
            startWindowMinutes: rule.props.startWindow?.toMinutes(),
            enableContinuousBackup: rule.props.enableContinuousBackup,
            targetBackupVault: vault.backupVaultName,
            copyActions: rule.props.copyActions?.map(this.planCopyActions),
        });
    }
    planCopyActions(props) {
        return {
            destinationBackupVaultArn: props.destinationBackupVault.backupVaultArn,
            lifecycle: (props.deleteAfter || props.moveToColdStorageAfter) && {
                deleteAfterDays: props.deleteAfter?.toDays(),
                moveToColdStorageAfterDays: props.moveToColdStorageAfter?.toDays(),
            },
        };
    }
    /**
     * The backup vault where backups are stored if not defined at
     * the rule level
     */
    get backupVault() {
        if (!this._backupVault) {
            // This cannot happen but is here to make TypeScript happy
            throw new Error('No backup vault!');
        }
        return this._backupVault;
    }
    /**
     * Adds a selection to this plan
     */
    addSelection(id, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_BackupSelectionOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSelection);
            }
            throw error;
        }
        return new selection_1.BackupSelection(this, id, {
            backupPlan: this,
            ...options,
        });
    }
    validatePlan() {
        if (this.rules.length === 0) {
            return ['A backup plan must have at least 1 rule.'];
        }
        return [];
    }
}
_a = JSII_RTTI_SYMBOL_1;
BackupPlan[_a] = { fqn: "aws-cdk-lib.aws_backup.BackupPlan", version: "2.74.0" };
exports.BackupPlan = BackupPlan;
//# sourceMappingURL=data:application/json;base64,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