"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupPlanRule = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const events = require("../../aws-events");
const core_1 = require("../../core");
/**
 * A backup plan rule
 */
class BackupPlanRule {
    /**
     * Daily with 35 days retention
     */
    static daily(backupVault) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.daily);
            }
            throw error;
        }
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Daily',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
            }),
            deleteAfter: core_1.Duration.days(35),
        });
    }
    /**
     * Weekly with 3 months retention
     */
    static weekly(backupVault) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.weekly);
            }
            throw error;
        }
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Weekly',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
                weekDay: 'SAT',
            }),
            deleteAfter: core_1.Duration.days(30 * 3),
        });
    }
    /**
     * Monthly 1 year retention, move to cold storage after 1 month
     */
    static monthly1Year(backupVault) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.monthly1Year);
            }
            throw error;
        }
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly1Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30),
            deleteAfter: core_1.Duration.days(365),
        });
    }
    /**
     * Monthly 5 year retention, move to cold storage after 3 months
     */
    static monthly5Year(backupVault) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.monthly5Year);
            }
            throw error;
        }
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly5Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 5),
        });
    }
    /**
     * Monthly 7 year retention, move to cold storage after 3 months
     */
    static monthly7Year(backupVault) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.monthly7Year);
            }
            throw error;
        }
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly7Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 7),
        });
    }
    /** @param props Rule properties */
    constructor(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_BackupPlanRuleProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BackupPlanRule);
            }
            throw error;
        }
        if (props.deleteAfter && props.moveToColdStorageAfter &&
            props.deleteAfter.toDays() < props.moveToColdStorageAfter.toDays()) {
            throw new Error('`deleteAfter` must be greater than `moveToColdStorageAfter`');
        }
        if (props.scheduleExpression && !/^cron/.test(props.scheduleExpression.expressionString)) {
            throw new Error('`scheduleExpression` must be of type `cron`');
        }
        const deleteAfter = (props.enableContinuousBackup && !props.deleteAfter) ? core_1.Duration.days(35) : props.deleteAfter;
        if (props.enableContinuousBackup && props.moveToColdStorageAfter) {
            throw new Error('`moveToColdStorageAfter` must not be specified if `enableContinuousBackup` is enabled');
        }
        if (props.enableContinuousBackup && props.deleteAfter &&
            (props.deleteAfter?.toDays() < 1 || props.deleteAfter?.toDays() > 35)) {
            throw new Error(`'deleteAfter' must be between 1 and 35 days if 'enableContinuousBackup' is enabled, but got ${props.deleteAfter.toHumanString()}`);
        }
        if (props.copyActions && props.copyActions.length > 0) {
            props.copyActions.forEach(copyAction => {
                if (copyAction.deleteAfter && !core_1.Token.isUnresolved(copyAction.deleteAfter) &&
                    copyAction.moveToColdStorageAfter && !core_1.Token.isUnresolved(copyAction.moveToColdStorageAfter) &&
                    copyAction.deleteAfter.toDays() < copyAction.moveToColdStorageAfter.toDays() + 90) {
                    throw new Error([
                        '\'deleteAfter\' must at least 90 days later than corresponding \'moveToColdStorageAfter\'',
                        `received 'deleteAfter: ${copyAction.deleteAfter.toDays()}' and 'moveToColdStorageAfter: ${copyAction.moveToColdStorageAfter.toDays()}'`,
                    ].join('\n'));
                }
            });
        }
        this.props = {
            ...props,
            deleteAfter,
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
BackupPlanRule[_a] = { fqn: "aws-cdk-lib.aws_backup.BackupPlanRule", version: "2.74.0" };
exports.BackupPlanRule = BackupPlanRule;
//# sourceMappingURL=data:application/json;base64,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