"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupVault = exports.BackupVaultEvents = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const backup_generated_1 = require("./backup.generated");
/**
 * Backup vault events. Some events are no longer supported and will not return
 * statuses or notifications.
 *
 * @see https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultNotifications.html#API_PutBackupVaultNotifications_RequestBody
 */
var BackupVaultEvents;
(function (BackupVaultEvents) {
    /** BACKUP_JOB_STARTED */
    BackupVaultEvents["BACKUP_JOB_STARTED"] = "BACKUP_JOB_STARTED";
    /** BACKUP_JOB_COMPLETED */
    BackupVaultEvents["BACKUP_JOB_COMPLETED"] = "BACKUP_JOB_COMPLETED";
    /** BACKUP_JOB_SUCCESSFUL */
    BackupVaultEvents["BACKUP_JOB_SUCCESSFUL"] = "BACKUP_JOB_SUCCESSFUL";
    /** BACKUP_JOB_FAILED */
    BackupVaultEvents["BACKUP_JOB_FAILED"] = "BACKUP_JOB_FAILED";
    /** BACKUP_JOB_EXPIRED */
    BackupVaultEvents["BACKUP_JOB_EXPIRED"] = "BACKUP_JOB_EXPIRED";
    /** RESTORE_JOB_STARTED */
    BackupVaultEvents["RESTORE_JOB_STARTED"] = "RESTORE_JOB_STARTED";
    /** RESTORE_JOB_COMPLETED */
    BackupVaultEvents["RESTORE_JOB_COMPLETED"] = "RESTORE_JOB_COMPLETED";
    /** RESTORE_JOB_SUCCESSFUL */
    BackupVaultEvents["RESTORE_JOB_SUCCESSFUL"] = "RESTORE_JOB_SUCCESSFUL";
    /** RESTORE_JOB_FAILED */
    BackupVaultEvents["RESTORE_JOB_FAILED"] = "RESTORE_JOB_FAILED";
    /** COPY_JOB_STARTED */
    BackupVaultEvents["COPY_JOB_STARTED"] = "COPY_JOB_STARTED";
    /** COPY_JOB_SUCCESSFUL */
    BackupVaultEvents["COPY_JOB_SUCCESSFUL"] = "COPY_JOB_SUCCESSFUL";
    /** COPY_JOB_FAILED */
    BackupVaultEvents["COPY_JOB_FAILED"] = "COPY_JOB_FAILED";
    /** RECOVERY_POINT_MODIFIED */
    BackupVaultEvents["RECOVERY_POINT_MODIFIED"] = "RECOVERY_POINT_MODIFIED";
    /** BACKUP_PLAN_CREATED */
    BackupVaultEvents["BACKUP_PLAN_CREATED"] = "BACKUP_PLAN_CREATED";
    /** BACKUP_PLAN_MODIFIED */
    BackupVaultEvents["BACKUP_PLAN_MODIFIED"] = "BACKUP_PLAN_MODIFIED";
    /** S3_BACKUP_OBJECT_FAILED */
    BackupVaultEvents["S3_BACKUP_OBJECT_FAILED"] = "S3_BACKUP_OBJECT_FAILED";
    /** BACKUP_PLAN_MODIFIED */
    BackupVaultEvents["S3_RESTORE_OBJECT_FAILED"] = "S3_RESTORE_OBJECT_FAILED";
})(BackupVaultEvents = exports.BackupVaultEvents || (exports.BackupVaultEvents = {}));
class BackupVaultBase extends core_1.Resource {
    /**
     * Grant the actions defined in actions to the given grantee
     * on this Backup Vault resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        for (const action of actions) {
            if (action.indexOf('*') >= 0) {
                throw new Error("AWS Backup access policies don't support a wildcard in the Action key.");
            }
        }
        return iam.Grant.addToPrincipal({
            grantee: grantee,
            actions: actions,
            resourceArns: [this.backupVaultArn],
        });
    }
}
/**
 * A backup vault
 */
class BackupVault extends BackupVaultBase {
    /**
     * Import an existing backup vault by name
     */
    static fromBackupVaultName(scope, id, backupVaultName) {
        const backupVaultArn = core_1.Stack.of(scope).formatArn({
            service: 'backup',
            resource: 'backup-vault',
            resourceName: backupVaultName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        return BackupVault.fromBackupVaultArn(scope, id, backupVaultArn);
    }
    /**
     * Import an existing backup vault by arn
     */
    static fromBackupVaultArn(scope, id, backupVaultArn) {
        const parsedArn = core_1.Stack.of(scope).splitArn(backupVaultArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!parsedArn.resourceName) {
            throw new Error(`Backup Vault Arn ${backupVaultArn} does not have a resource name.`);
        }
        class Import extends BackupVaultBase {
            constructor() {
                super(...arguments);
                this.backupVaultName = parsedArn.resourceName;
                this.backupVaultArn = backupVaultArn;
            }
        }
        return new Import(scope, id, {
            account: parsedArn.account,
            region: parsedArn.region,
        });
    }
    constructor(scope, id, props = {}) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_backup_BackupVaultProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BackupVault);
            }
            throw error;
        }
        if (props.backupVaultName && !/^[a-zA-Z0-9\-_]{2,50}$/.test(props.backupVaultName)) {
            throw new Error('Expected vault name to match pattern `^[a-zA-Z0-9\-_]{2,50}$`');
        }
        let notifications;
        if (props.notificationTopic) {
            notifications = {
                backupVaultEvents: props.notificationEvents || Object.values(BackupVaultEvents),
                snsTopicArn: props.notificationTopic.topicArn,
            };
            props.notificationTopic.grantPublish(new iam.ServicePrincipal('backup.amazonaws.com'));
        }
        this.accessPolicy = props.accessPolicy ?? new iam.PolicyDocument();
        if (props.blockRecoveryPointDeletion) {
            this.blockRecoveryPointDeletion();
        }
        const vault = new backup_generated_1.CfnBackupVault(this, 'Resource', {
            backupVaultName: props.backupVaultName || this.uniqueVaultName(),
            accessPolicy: core_1.Lazy.any({ produce: () => this.accessPolicy.toJSON() }),
            encryptionKeyArn: props.encryptionKey && props.encryptionKey.keyArn,
            notifications,
            lockConfiguration: renderLockConfiguration(props.lockConfiguration),
        });
        vault.applyRemovalPolicy(props.removalPolicy);
        this.backupVaultName = vault.attrBackupVaultName;
        this.backupVaultArn = vault.attrBackupVaultArn;
    }
    /**
     * Adds a statement to the vault access policy
     */
    addToAccessPolicy(statement) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToAccessPolicy);
            }
            throw error;
        }
        this.accessPolicy.addStatements(statement);
    }
    /**
     * Adds a statement to the vault access policy that prevents anyone
     * from deleting a recovery point.
     */
    blockRecoveryPointDeletion() {
        this.addToAccessPolicy(new iam.PolicyStatement({
            effect: iam.Effect.DENY,
            actions: [
                'backup:DeleteRecoveryPoint',
                'backup:UpdateRecoveryPointLifecycle',
            ],
            principals: [new iam.AnyPrincipal()],
            resources: ['*'],
        }));
    }
    uniqueVaultName() {
        // Max length of 50 chars, get the last 50 chars
        const id = core_1.Names.uniqueId(this);
        return id.substring(Math.max(id.length - 50, 0), id.length);
    }
}
_a = JSII_RTTI_SYMBOL_1;
BackupVault[_a] = { fqn: "aws-cdk-lib.aws_backup.BackupVault", version: "2.74.0" };
exports.BackupVault = BackupVault;
function renderLockConfiguration(config) {
    if (!config) {
        return undefined;
    }
    if (config.changeableFor && config.changeableFor.toHours() < 72) {
        throw new Error(`AWS Backup enforces a 72-hour cooling-off period before Vault Lock takes effect and becomes immutable, got ${config.changeableFor.toHours()} hours`);
    }
    if (config.maxRetention) {
        if (config.maxRetention.toDays() > 36500) {
            throw new Error(`The longest maximum retention period you can specify is 36500 days, got ${config.maxRetention.toDays()} days`);
        }
        if (config.maxRetention.toDays() <= config.minRetention.toDays()) {
            throw new Error(`The maximum retention period (${config.maxRetention.toDays()} days) must be greater than the minimum retention period (${config.minRetention.toDays()} days)`);
        }
    }
    if (config.minRetention.toHours() < 24) {
        throw new Error(`The shortest minimum retention period you can specify is 1 day, got ${config.minRetention.toHours()} hours`);
    }
    return {
        minRetentionDays: config.minRetention.toDays(),
        maxRetentionDays: config.maxRetention?.toDays(),
        changeableForDays: config.changeableFor?.toDays(),
    };
}
//# sourceMappingURL=data:application/json;base64,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