"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = exports.CertificateValidation = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const certificate_base_1 = require("./certificate-base");
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * How to validate a certificate
 */
class CertificateValidation {
    /**
     * Validate the certificate with DNS
     *
     * IMPORTANT: If `hostedZone` is not specified, DNS records must be added
     * manually and the stack will not complete creating until the records are
     * added.
     *
     * @param hostedZone the hosted zone where DNS records must be created
     */
    static fromDns(hostedZone) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_route53_IHostedZone(hostedZone);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDns);
            }
            throw error;
        }
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZone,
        });
    }
    /**
     * Validate the certificate with automatically created DNS records in multiple
     * Amazon Route 53 hosted zones.
     *
     * @param hostedZones a map of hosted zones where DNS records must be created
     * for the domains in the certificate
     */
    static fromDnsMultiZone(hostedZones) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZones,
        });
    }
    /**
     * Validate the certificate with Email
     *
     * IMPORTANT: if you are creating a certificate as part of your stack, the stack
     * will not complete creating until you read and follow the instructions in the
     * email that you will receive.
     *
     * ACM will send validation emails to the following addresses:
     *
     *  admin@domain.com
     *  administrator@domain.com
     *  hostmaster@domain.com
     *  postmaster@domain.com
     *  webmaster@domain.com
     *
     * For every domain that you register.
     *
     * @param validationDomains a map of validation domains to use for domains in the certificate
     */
    static fromEmail(validationDomains) {
        return new CertificateValidation({
            method: ValidationMethod.EMAIL,
            validationDomains,
        });
    }
    /** @param props Certification validation properties */
    constructor(props) {
        this.props = props;
        this.method = props.method ?? ValidationMethod.EMAIL;
    }
}
_a = JSII_RTTI_SYMBOL_1;
CertificateValidation[_a] = { fqn: "aws-cdk-lib.aws_certificatemanager.CertificateValidation", version: "2.74.0" };
exports.CertificateValidation = CertificateValidation;
/**
 * A certificate managed by AWS Certificate Manager
 */
class Certificate extends certificate_base_1.CertificateBase {
    /**
     * Import a certificate
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends certificate_base_1.CertificateBase {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_certificatemanager_CertificateProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Certificate);
            }
            throw error;
        }
        let validation;
        if (props.validation) {
            validation = props.validation;
        }
        else { // Deprecated props
            if (props.validationMethod === ValidationMethod.DNS) {
                validation = CertificateValidation.fromDns();
            }
            else {
                validation = CertificateValidation.fromEmail(props.validationDomains);
            }
        }
        // check if domain name is 64 characters or less
        if (!core_1.Token.isUnresolved(props.domainName) && props.domainName.length > 64) {
            throw new Error('Domain name must be 64 characters or less');
        }
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        let certificateTransparencyLoggingPreference;
        if (props.transparencyLoggingEnabled !== undefined) {
            certificateTransparencyLoggingPreference = props.transparencyLoggingEnabled ? 'ENABLED' : 'DISABLED';
        }
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: renderDomainValidation(validation, allDomainNames),
            validationMethod: validation.method,
            certificateTransparencyLoggingPreference,
        });
        core_1.Tags.of(cert).add(NAME_TAG, props.certificateName || this.node.path.slice(0, 255));
        this.certificateArn = cert.ref;
    }
}
_b = JSII_RTTI_SYMBOL_1;
Certificate[_b] = { fqn: "aws-cdk-lib.aws_certificatemanager.Certificate", version: "2.74.0" };
exports.Certificate = Certificate;
/**
 * Method used to assert ownership of the domain
 */
var ValidationMethod;
(function (ValidationMethod) {
    /**
     * Send email to a number of email addresses associated with the domain
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html
     */
    ValidationMethod["EMAIL"] = "EMAIL";
    /**
     * Validate ownership by adding appropriate DNS records
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html
     */
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
// eslint-disable-next-line max-len
function renderDomainValidation(validation, domainNames) {
    const domainValidation = [];
    switch (validation.method) {
        case ValidationMethod.DNS:
            for (const domainName of getUniqueDnsDomainNames(domainNames)) {
                const hostedZone = validation.props.hostedZones?.[domainName] ?? validation.props.hostedZone;
                if (hostedZone) {
                    domainValidation.push({ domainName, hostedZoneId: hostedZone.hostedZoneId });
                }
            }
            break;
        case ValidationMethod.EMAIL:
            for (const domainName of domainNames) {
                const validationDomain = validation.props.validationDomains?.[domainName];
                if (!validationDomain && core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                domainValidation.push({ domainName, validationDomain: validationDomain ?? (0, util_1.apexDomain)(domainName) });
            }
            break;
        default:
            throw new Error(`Unknown validation method ${validation.method}`);
    }
    return domainValidation.length !== 0 ? domainValidation : undefined;
}
/**
 * Removes wildcard domains (*.example.com) where the base domain (example.com) is present.
 * This is because the DNS validation treats them as the same thing, and the automated CloudFormation
 * DNS validation errors out with the duplicate records.
 */
function getUniqueDnsDomainNames(domainNames) {
    return domainNames.filter(domain => {
        return core_1.Token.isUnresolved(domain) || !domain.startsWith('*.') || !domainNames.includes(domain.replace('*.', ''));
    });
}
//# sourceMappingURL=data:application/json;base64,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