"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SlackChannelConfiguration = exports.LoggingLevel = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../aws-cloudwatch");
const iam = require("../../aws-iam");
const logs = require("../../aws-logs");
const cdk = require("../../core");
const chatbot_generated_1 = require("./chatbot.generated");
/**
 * Logging levels include ERROR, INFO, or NONE.
 */
var LoggingLevel;
(function (LoggingLevel) {
    /**
     * ERROR
     */
    LoggingLevel["ERROR"] = "ERROR";
    /**
     * INFO
     */
    LoggingLevel["INFO"] = "INFO";
    /**
     * NONE
     */
    LoggingLevel["NONE"] = "NONE";
})(LoggingLevel = exports.LoggingLevel || (exports.LoggingLevel = {}));
/**
 * Either a new or imported Slack channel configuration
 */
class SlackChannelConfigurationBase extends cdk.Resource {
    /**
     * Adds extra permission to iam-role of Slack channel configuration
     * @param statement
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Return the given named metric for this SlackChannelConfiguration
     */
    metric(metricName, props) {
        // AWS Chatbot publishes metrics to us-east-1 regardless of stack region
        // https://docs.aws.amazon.com/chatbot/latest/adminguide/monitoring-cloudwatch.html
        return new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            dimensionsMap: {
                ConfigurationName: this.slackChannelConfigurationName,
            },
            metricName,
            ...props,
        });
    }
    bindAsNotificationRuleTarget(_scope) {
        return {
            targetType: 'AWSChatbotSlack',
            targetAddress: this.slackChannelConfigurationArn,
        };
    }
}
/**
 * A new Slack channel configuration
 */
class SlackChannelConfiguration extends SlackChannelConfigurationBase {
    /**
     * Import an existing Slack channel configuration provided an ARN
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param slackChannelConfigurationArn configuration ARN (i.e. arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack)
     *
     * @returns a reference to the existing Slack channel configuration
     */
    static fromSlackChannelConfigurationArn(scope, id, slackChannelConfigurationArn) {
        const re = /^slack-channel\//;
        const resourceName = cdk.Arn.extractResourceName(slackChannelConfigurationArn, 'chat-configuration');
        if (!cdk.Token.isUnresolved(slackChannelConfigurationArn) && !re.test(resourceName)) {
            throw new Error('The ARN of a Slack integration must be in the form: arn:aws:chatbot:{region}:{account}:chat-configuration/slack-channel/{slackChannelName}');
        }
        class Import extends SlackChannelConfigurationBase {
            constructor(s, i) {
                super(s, i);
                /**
                 * @attribute
                 */
                this.slackChannelConfigurationArn = slackChannelConfigurationArn;
                this.role = undefined;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
                // handle slackChannelConfigurationName as specified above
                if (cdk.Token.isUnresolved(slackChannelConfigurationArn)) {
                    this.slackChannelConfigurationName = cdk.Fn.select(1, cdk.Fn.split('slack-channel/', resourceName));
                }
                else {
                    this.slackChannelConfigurationName = resourceName.substring('slack-channel/'.length);
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for All SlackChannelConfigurations
     */
    static metricAll(metricName, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricAll);
            }
            throw error;
        }
        // AWS Chatbot publishes metrics to us-east-1 regardless of stack region
        // https://docs.aws.amazon.com/chatbot/latest/adminguide/monitoring-cloudwatch.html
        return new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            metricName,
            ...props,
        });
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.slackChannelConfigurationName,
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_chatbot_SlackChannelConfigurationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SlackChannelConfiguration);
            }
            throw error;
        }
        this.role = props.role || new iam.Role(this, 'ConfigurationRole', {
            assumedBy: new iam.ServicePrincipal('chatbot.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        this.notificationTopics = props.notificationTopics ?? [];
        const configuration = new chatbot_generated_1.CfnSlackChannelConfiguration(this, 'Resource', {
            configurationName: props.slackChannelConfigurationName,
            iamRoleArn: this.role.roleArn,
            slackWorkspaceId: props.slackWorkspaceId,
            slackChannelId: props.slackChannelId,
            snsTopicArns: cdk.Lazy.list({ produce: () => this.notificationTopics.map(topic => topic.topicArn) }, { omitEmpty: true }),
            loggingLevel: props.loggingLevel?.toString(),
            guardrailPolicies: cdk.Lazy.list({ produce: () => props.guardrailPolicies?.map(policy => policy.managedPolicyArn) }, { omitEmpty: true }),
        });
        // Log retention
        // AWS Chatbot publishes logs to us-east-1 regardless of stack region https://docs.aws.amazon.com/chatbot/latest/adminguide/cloudwatch-logs.html
        if (props.logRetention) {
            new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/chatbot/${props.slackChannelConfigurationName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logGroupRegion: 'us-east-1',
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
        }
        this.slackChannelConfigurationArn = configuration.ref;
        this.slackChannelConfigurationName = props.slackChannelConfigurationName;
    }
    /**
     * Adds a SNS topic that deliver notifications to AWS Chatbot.
     * @param notificationTopic
     */
    addNotificationTopic(notificationTopic) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_sns_ITopic(notificationTopic);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addNotificationTopic);
            }
            throw error;
        }
        this.notificationTopics.push(notificationTopic);
    }
}
_a = JSII_RTTI_SYMBOL_1;
SlackChannelConfiguration[_a] = { fqn: "aws-cdk-lib.aws_chatbot.SlackChannelConfiguration", version: "2.74.0" };
exports.SlackChannelConfiguration = SlackChannelConfiguration;
//# sourceMappingURL=data:application/json;base64,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