"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudfront = require("../../aws-cloudfront");
const iam = require("../../aws-iam");
const cdk = require("../../core");
const http_origin_1 = require("./http-origin");
/**
 * An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 */
class S3Origin {
    constructor(bucket, props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_IBucket(bucket);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_origins_S3OriginProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, S3Origin);
            }
            throw error;
        }
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    bind(scope, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_OriginBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        return this.origin.bind(scope, options);
    }
}
_a = JSII_RTTI_SYMBOL_1;
S3Origin[_a] = { fqn: "aws-cdk-lib.aws_cloudfront_origins.S3Origin", version: "2.74.0" };
exports.S3Origin = S3Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, { originAccessIdentity, ...props }) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
        if (originAccessIdentity) {
            this.originAccessIdentity = originAccessIdentity;
        }
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            // Using a bucket from another stack creates a cyclic reference with
            // the bucket taking a dependency on the generated S3CanonicalUserId for the grant principal,
            // and the distribution having a dependency on the bucket's domain name.
            // Fix this by parenting the OAI in the bucket's stack when cross-stack usage is detected.
            const bucketStack = cdk.Stack.of(this.bucket);
            const bucketInDifferentStack = bucketStack !== cdk.Stack.of(scope);
            const oaiScope = bucketInDifferentStack ? bucketStack : scope;
            const oaiId = bucketInDifferentStack ? `${cdk.Names.uniqueId(scope)}S3Origin` : 'S3Origin';
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(oaiScope, oaiId, {
                comment: `Identity for ${options.originId}`,
            });
        }
        // Used rather than `grantRead` because `grantRead` will grant overly-permissive policies.
        // Only GetObject is needed to retrieve objects for the distribution.
        // This also excludes KMS permissions; currently, OAI only supports SSE-S3 for buckets.
        // Source: https://aws.amazon.com/blogs/networking-and-content-delivery/serving-sse-kms-encrypted-content-from-s3-using-cloudfront/
        this.bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.bucket.arnForObjects('*')],
            actions: ['s3:GetObject'],
            principals: [this.originAccessIdentity.grantPrincipal],
        }));
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityId}` };
    }
}
//# sourceMappingURL=data:application/json;base64,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