"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaEdgeEventType = exports.CachedMethods = exports.AllowedMethods = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.OriginProtocolPolicy = exports.ViewerProtocolPolicy = exports.PriceClass = exports.HttpVersion = exports.Distribution = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const s3 = require("../../aws-s3");
const core_1 = require("../../core");
const cx_api_1 = require("../../cx-api");
const constructs_1 = require("constructs");
const cloudfront_generated_1 = require("./cloudfront.generated");
const cache_behavior_1 = require("./private/cache-behavior");
const utils_1 = require("./private/utils");
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 */
class Distribution extends core_1.Resource {
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_DistributionAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDistributionAttributes);
            }
            throw error;
        }
        return new class extends core_1.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
            grant(grantee, ...actions) {
                return iam.Grant.addToPrincipal({ grantee, actions, resourceArns: [(0, utils_1.formatDistributionArn)(this)] });
            }
            grantCreateInvalidation(grantee) {
                return this.grant(grantee, 'cloudfront:CreateInvalidation');
            }
        }();
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.additionalBehaviors = [];
        this.boundOrigins = [];
        this.originGroups = [];
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_DistributionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Distribution);
            }
            throw error;
        }
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).splitArn(props.certificate.certificateArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`Distribution certificates must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
            if ((props.domainNames ?? []).length === 0) {
                throw new Error('Must specify at least one domain name to use a certificate with a distribution');
            }
        }
        const originId = this.addOrigin(props.defaultBehavior.origin);
        this.defaultBehavior = new cache_behavior_1.CacheBehavior(originId, { pathPattern: '*', ...props.defaultBehavior });
        if (props.additionalBehaviors) {
            Object.entries(props.additionalBehaviors).forEach(([pathPattern, behaviorOptions]) => {
                this.addBehavior(pathPattern, behaviorOptions.origin, behaviorOptions);
            });
        }
        this.certificate = props.certificate;
        this.errorResponses = props.errorResponses ?? [];
        // Comments have an undocumented limit of 128 characters
        const trimmedComment = props.comment && props.comment.length > 128
            ? `${props.comment.slice(0, 128 - 3)}...`
            : props.comment;
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'Resource', {
            distributionConfig: {
                enabled: props.enabled ?? true,
                origins: core_1.Lazy.any({ produce: () => this.renderOrigins() }),
                originGroups: core_1.Lazy.any({ produce: () => this.renderOriginGroups() }),
                defaultCacheBehavior: this.defaultBehavior._renderBehavior(),
                aliases: props.domainNames,
                cacheBehaviors: core_1.Lazy.any({ produce: () => this.renderCacheBehaviors() }),
                comment: trimmedComment,
                customErrorResponses: this.renderErrorResponses(),
                defaultRootObject: props.defaultRootObject,
                httpVersion: props.httpVersion ?? HttpVersion.HTTP2,
                ipv6Enabled: props.enableIpv6 ?? true,
                logging: this.renderLogging(props),
                priceClass: props.priceClass ?? undefined,
                restrictions: this.renderRestrictions(props.geoRestriction),
                viewerCertificate: this.certificate ? this.renderViewerCertificate(this.certificate, props.minimumProtocolVersion, props.sslSupportMethod) : undefined,
                webAclId: props.webAclId,
            },
        });
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param origin the origin to use for this behavior
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern, origin, behaviorOptions = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_IOrigin(origin);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_AddBehaviorOptions(behaviorOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addBehavior);
            }
            throw error;
        }
        if (pathPattern === '*') {
            throw new Error('Only the default behavior can have a path pattern of \'*\'');
        }
        const originId = this.addOrigin(origin);
        this.additionalBehaviors.push(new cache_behavior_1.CacheBehavior(originId, { pathPattern, ...behaviorOptions }));
    }
    /**
     * Adds an IAM policy statement associated with this distribution to an IAM
     * principal's policy.
     *
     * @param identity The principal
     * @param actions The set of actions to allow (i.e. "cloudfront:ListInvalidations")
     */
    grant(identity, ...actions) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grant);
            }
            throw error;
        }
        return iam.Grant.addToPrincipal({ grantee: identity, actions, resourceArns: [(0, utils_1.formatDistributionArn)(this)] });
    }
    /**
     * Grant to create invalidations for this bucket to an IAM principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantCreateInvalidation(identity) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantCreateInvalidation);
            }
            throw error;
        }
        return this.grant(identity, 'cloudfront:CreateInvalidation');
    }
    addOrigin(origin, isFailoverOrigin = false) {
        const ORIGIN_ID_MAX_LENGTH = 128;
        const existingOrigin = this.boundOrigins.find(boundOrigin => boundOrigin.origin === origin);
        if (existingOrigin) {
            return existingOrigin.originGroupId ?? existingOrigin.originId;
        }
        else {
            const originIndex = this.boundOrigins.length + 1;
            const scope = new constructs_1.Construct(this, `Origin${originIndex}`);
            const generatedId = core_1.Names.uniqueId(scope).slice(-ORIGIN_ID_MAX_LENGTH);
            const originBindConfig = origin.bind(scope, { originId: generatedId });
            const originId = originBindConfig.originProperty?.id ?? generatedId;
            const duplicateId = this.boundOrigins.find(boundOrigin => boundOrigin.originProperty?.id === originBindConfig.originProperty?.id);
            if (duplicateId) {
                throw new Error(`Origin with id ${duplicateId.originProperty?.id} already exists. OriginIds must be unique within a distribution`);
            }
            if (!originBindConfig.failoverConfig) {
                this.boundOrigins.push({ origin, originId, ...originBindConfig });
            }
            else {
                if (isFailoverOrigin) {
                    throw new Error('An Origin cannot use an Origin with its own failover configuration as its fallback origin!');
                }
                const groupIndex = this.originGroups.length + 1;
                const originGroupId = core_1.Names.uniqueId(new constructs_1.Construct(this, `OriginGroup${groupIndex}`)).slice(-ORIGIN_ID_MAX_LENGTH);
                this.boundOrigins.push({ origin, originId, originGroupId, ...originBindConfig });
                const failoverOriginId = this.addOrigin(originBindConfig.failoverConfig.failoverOrigin, true);
                this.addOriginGroup(originGroupId, originBindConfig.failoverConfig.statusCodes, originId, failoverOriginId);
                return originGroupId;
            }
            return originBindConfig.originProperty?.id ?? originId;
        }
    }
    addOriginGroup(originGroupId, statusCodes, originId, failoverOriginId) {
        statusCodes = statusCodes ?? [500, 502, 503, 504];
        if (statusCodes.length === 0) {
            throw new Error('fallbackStatusCodes cannot be empty');
        }
        this.originGroups.push({
            failoverCriteria: {
                statusCodes: {
                    items: statusCodes,
                    quantity: statusCodes.length,
                },
            },
            id: originGroupId,
            members: {
                items: [
                    { originId },
                    { originId: failoverOriginId },
                ],
                quantity: 2,
            },
        });
    }
    renderOrigins() {
        const renderedOrigins = [];
        this.boundOrigins.forEach(boundOrigin => {
            if (boundOrigin.originProperty) {
                renderedOrigins.push(boundOrigin.originProperty);
            }
        });
        return renderedOrigins;
    }
    renderOriginGroups() {
        return this.originGroups.length === 0
            ? undefined
            : {
                items: this.originGroups,
                quantity: this.originGroups.length,
            };
    }
    renderCacheBehaviors() {
        if (this.additionalBehaviors.length === 0) {
            return undefined;
        }
        return this.additionalBehaviors.map(behavior => behavior._renderBehavior());
    }
    renderErrorResponses() {
        if (this.errorResponses.length === 0) {
            return undefined;
        }
        return this.errorResponses.map(errorConfig => {
            if (!errorConfig.responseHttpStatus && !errorConfig.ttl && !errorConfig.responsePagePath) {
                throw new Error('A custom error response without either a \'responseHttpStatus\', \'ttl\' or \'responsePagePath\' is not valid.');
            }
            return {
                errorCachingMinTtl: errorConfig.ttl?.toSeconds(),
                errorCode: errorConfig.httpStatus,
                responseCode: errorConfig.responsePagePath
                    ? errorConfig.responseHttpStatus ?? errorConfig.httpStatus
                    : errorConfig.responseHttpStatus,
                responsePagePath: errorConfig.responsePagePath,
            };
        });
    }
    renderLogging(props) {
        if (!props.enableLogging && !props.logBucket) {
            return undefined;
        }
        if (props.enableLogging === false && props.logBucket) {
            throw new Error('Explicitly disabled logging but provided a logging bucket.');
        }
        const bucket = props.logBucket ?? new s3.Bucket(this, 'LoggingBucket', {
            encryption: s3.BucketEncryption.S3_MANAGED,
        });
        return {
            bucket: bucket.bucketRegionalDomainName,
            includeCookies: props.logIncludesCookies,
            prefix: props.logFilePrefix,
        };
    }
    renderRestrictions(geoRestriction) {
        return geoRestriction ? {
            geoRestriction: {
                restrictionType: geoRestriction.restrictionType,
                locations: geoRestriction.locations,
            },
        } : undefined;
    }
    renderViewerCertificate(certificate, minimumProtocolVersionProp, sslSupportMethodProp) {
        const defaultVersion = core_1.FeatureFlags.of(this).isEnabled(cx_api_1.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021)
            ? SecurityPolicyProtocol.TLS_V1_2_2021 : SecurityPolicyProtocol.TLS_V1_2_2019;
        const minimumProtocolVersion = minimumProtocolVersionProp ?? defaultVersion;
        const sslSupportMethod = sslSupportMethodProp ?? SSLMethod.SNI;
        return {
            acmCertificateArn: certificate.certificateArn,
            minimumProtocolVersion: minimumProtocolVersion,
            sslSupportMethod: sslSupportMethod,
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
Distribution[_a] = { fqn: "aws-cdk-lib.aws_cloudfront.Distribution", version: "2.74.0" };
exports.Distribution = Distribution;
/** Maximum HTTP version to support */
var HttpVersion;
(function (HttpVersion) {
    /** HTTP 1.1 */
    HttpVersion["HTTP1_1"] = "http1.1";
    /** HTTP 2 */
    HttpVersion["HTTP2"] = "http2";
    /** HTTP 2 and HTTP 3 */
    HttpVersion["HTTP2_AND_3"] = "http2and3";
    /** HTTP 3 */
    HttpVersion["HTTP3"] = "http3";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
var PriceClass;
(function (PriceClass) {
    /** USA, Canada, Europe, & Israel */
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    /** All locations */
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    /** HTTPS only */
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    /** Will redirect HTTP requests to HTTPS */
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    /** Both HTTP and HTTPS supported */
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    /** Connect on HTTP only */
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    /** Connect with the same protocol as the viewer */
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    /** Connect on HTTPS only */
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
    SecurityPolicyProtocol["TLS_V1_2_2019"] = "TLSv1.2_2019";
    SecurityPolicyProtocol["TLS_V1_2_2021"] = "TLSv1.2_2021";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
class AllowedMethods {
    constructor(methods) { this.methods = methods; }
}
_b = JSII_RTTI_SYMBOL_1;
AllowedMethods[_b] = { fqn: "aws-cdk-lib.aws_cloudfront.AllowedMethods", version: "2.74.0" };
/** HEAD and GET */
AllowedMethods.ALLOW_GET_HEAD = new AllowedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
AllowedMethods.ALLOW_GET_HEAD_OPTIONS = new AllowedMethods(['GET', 'HEAD', 'OPTIONS']);
/** All supported HTTP methods */
AllowedMethods.ALLOW_ALL = new AllowedMethods(['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE']);
exports.AllowedMethods = AllowedMethods;
/**
 * The HTTP methods that the Behavior will cache requests on.
 */
class CachedMethods {
    constructor(methods) { this.methods = methods; }
}
_c = JSII_RTTI_SYMBOL_1;
CachedMethods[_c] = { fqn: "aws-cdk-lib.aws_cloudfront.CachedMethods", version: "2.74.0" };
/** HEAD and GET */
CachedMethods.CACHE_GET_HEAD = new CachedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
CachedMethods.CACHE_GET_HEAD_OPTIONS = new CachedMethods(['GET', 'HEAD', 'OPTIONS']);
exports.CachedMethods = CachedMethods;
/**
 * The type of events that a Lambda@Edge function can be invoked in response to.
 */
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing response
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
//# sourceMappingURL=data:application/json;base64,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