"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginRequestQueryStringBehavior = exports.OriginRequestHeaderBehavior = exports.OriginRequestCookieBehavior = exports.OriginRequestPolicy = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * A Origin Request Policy configuration.
 *
 * @resource AWS::CloudFront::OriginRequestPolicy
 */
class OriginRequestPolicy extends core_1.Resource {
    /** Imports a Origin Request Policy from its id. */
    static fromOriginRequestPolicyId(scope, id, originRequestPolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.originRequestPolicyId = originRequestPolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed origin request policy. */
    static fromManagedOriginRequestPolicy(managedOriginRequestPolicyId) {
        return new class {
            constructor() {
                this.originRequestPolicyId = managedOriginRequestPolicyId;
            }
        }();
    }
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.originRequestPolicyName,
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_OriginRequestPolicyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, OriginRequestPolicy);
            }
            throw error;
        }
        const originRequestPolicyName = props.originRequestPolicyName ?? core_1.Names.uniqueId(this);
        if (!core_1.Token.isUnresolved(originRequestPolicyName) && !originRequestPolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'originRequestPolicyName' can only include '-', '_', and alphanumeric characters, got: '${props.originRequestPolicyName}'`);
        }
        const cookies = props.cookieBehavior ?? OriginRequestCookieBehavior.none();
        const headers = props.headerBehavior ?? OriginRequestHeaderBehavior.none();
        const queryStrings = props.queryStringBehavior ?? OriginRequestQueryStringBehavior.none();
        const resource = new cloudfront_generated_1.CfnOriginRequestPolicy(this, 'Resource', {
            originRequestPolicyConfig: {
                name: originRequestPolicyName,
                comment: props.comment,
                cookiesConfig: {
                    cookieBehavior: cookies.behavior,
                    cookies: cookies.cookies,
                },
                headersConfig: {
                    headerBehavior: headers.behavior,
                    headers: headers.headers,
                },
                queryStringsConfig: {
                    queryStringBehavior: queryStrings.behavior,
                    queryStrings: queryStrings.queryStrings,
                },
            },
        });
        this.originRequestPolicyId = resource.ref;
    }
}
_a = JSII_RTTI_SYMBOL_1;
OriginRequestPolicy[_a] = { fqn: "aws-cdk-lib.aws_cloudfront.OriginRequestPolicy", version: "2.74.0" };
/** This policy includes only the User-Agent and Referer headers. It doesn’t include any query strings or cookies. */
OriginRequestPolicy.USER_AGENT_REFERER_HEADERS = OriginRequestPolicy.fromManagedOriginRequestPolicy('acba4595-bd28-49b8-b9fe-13317c0390fa');
/** This policy includes the header that enables cross-origin resource sharing (CORS) requests when the origin is a custom origin. */
OriginRequestPolicy.CORS_CUSTOM_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('59781a5b-3903-41f3-afcb-af62929ccde1');
/** This policy includes the headers that enable cross-origin resource sharing (CORS) requests when the origin is an Amazon S3 bucket. */
OriginRequestPolicy.CORS_S3_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('88a5eaf4-2fd4-4709-b370-b4c650ea3fcf');
/** This policy includes all values (query strings, headers, and cookies) in the viewer request. */
OriginRequestPolicy.ALL_VIEWER = OriginRequestPolicy.fromManagedOriginRequestPolicy('216adef6-5c7f-47e4-b989-5492eafa07d3');
/** This policy is designed for use with an origin that is an AWS Elemental MediaTailor endpoint. */
OriginRequestPolicy.ELEMENTAL_MEDIA_TAILOR = OriginRequestPolicy.fromManagedOriginRequestPolicy('775133bc-15f2-49f9-abea-afb2e0bf67d2');
/** This policy includes all values (headers, cookies, and query strings) in the viewer request, and all CloudFront headers that were released through June 2022 (CloudFront headers released after June 2022 are not included). */
OriginRequestPolicy.ALL_VIEWER_AND_CLOUDFRONT_2022 = OriginRequestPolicy.fromManagedOriginRequestPolicy('33f36d7e-f396-46d9-90e0-52428a34d9dc');
/** This policy includes all values (query strings, and cookies) except the header in the viewer request. */
OriginRequestPolicy.ALL_VIEWER_EXCEPT_HOST_HEADER = OriginRequestPolicy.fromManagedOriginRequestPolicy('b689b0a8-53d0-40ab-baf2-68738e2966ac');
exports.OriginRequestPolicy = OriginRequestPolicy;
/**
 * Determines whether any cookies in viewer requests (and if so, which cookies)
 * are included in requests that CloudFront sends to the origin.
 */
class OriginRequestCookieBehavior {
    /**
     * Cookies in viewer requests are not included in requests that CloudFront sends to the origin.
     * Any cookies that are listed in a CachePolicy are still included in origin requests.
     */
    static none() { return new OriginRequestCookieBehavior('none'); }
    /** All cookies in viewer requests are included in requests that CloudFront sends to the origin. */
    static all() { return new OriginRequestCookieBehavior('all'); }
    /** Only the provided `cookies` are included in requests that CloudFront sends to the origin. */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new OriginRequestCookieBehavior('whitelist', cookies);
    }
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
}
_b = JSII_RTTI_SYMBOL_1;
OriginRequestCookieBehavior[_b] = { fqn: "aws-cdk-lib.aws_cloudfront.OriginRequestCookieBehavior", version: "2.74.0" };
exports.OriginRequestCookieBehavior = OriginRequestCookieBehavior;
/**
 * Determines whether any HTTP headers (and if so, which headers) are included in requests that CloudFront sends to the origin.
 */
class OriginRequestHeaderBehavior {
    /**
     * HTTP headers are not included in requests that CloudFront sends to the origin.
     * Any headers that are listed in a CachePolicy are still included in origin requests.
     */
    static none() { return new OriginRequestHeaderBehavior('none'); }
    /**
     * All HTTP headers in viewer requests are included in requests that CloudFront sends to the origin.
     * Additionally, any additional CloudFront headers provided are included; the additional headers are added by CloudFront.
     * @see https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-cloudfront-headers.html
     */
    static all(...cloudfrontHeaders) {
        if (cloudfrontHeaders.length > 0) {
            if (!cloudfrontHeaders.every(header => header.startsWith('CloudFront-'))) {
                throw new Error('additional CloudFront headers passed to `OriginRequestHeaderBehavior.all()` must begin with \'CloudFront-\'');
            }
            return new OriginRequestHeaderBehavior('allViewerAndWhitelistCloudFront', cloudfrontHeaders);
        }
        else {
            return new OriginRequestHeaderBehavior('allViewer');
        }
    }
    /** Listed headers are included in requests that CloudFront sends to the origin. */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        if (headers.map(header => header.toLowerCase()).some(header => ['authorization', 'accept-encoding'].includes(header))) {
            throw new Error('you cannot pass `Authorization` or `Accept-Encoding` as header values; use a CachePolicy to forward these headers instead');
        }
        return new OriginRequestHeaderBehavior('whitelist', headers);
    }
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
}
_c = JSII_RTTI_SYMBOL_1;
OriginRequestHeaderBehavior[_c] = { fqn: "aws-cdk-lib.aws_cloudfront.OriginRequestHeaderBehavior", version: "2.74.0" };
exports.OriginRequestHeaderBehavior = OriginRequestHeaderBehavior;
/**
 * Determines whether any URL query strings in viewer requests (and if so, which query strings)
 * are included in requests that CloudFront sends to the origin.
 */
class OriginRequestQueryStringBehavior {
    /**
     * Query strings in viewer requests are not included in requests that CloudFront sends to the origin.
     * Any query strings that are listed in a CachePolicy are still included in origin requests.
     */
    static none() { return new OriginRequestQueryStringBehavior('none'); }
    /** All query strings in viewer requests are included in requests that CloudFront sends to the origin. */
    static all() { return new OriginRequestQueryStringBehavior('all'); }
    /** Only the provided `queryStrings` are included in requests that CloudFront sends to the origin. */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new OriginRequestQueryStringBehavior('whitelist', queryStrings);
    }
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
}
_d = JSII_RTTI_SYMBOL_1;
OriginRequestQueryStringBehavior[_d] = { fqn: "aws-cdk-lib.aws_cloudfront.OriginRequestQueryStringBehavior", version: "2.74.0" };
exports.OriginRequestQueryStringBehavior = OriginRequestQueryStringBehavior;
//# sourceMappingURL=data:application/json;base64,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