"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginBase = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 */
class OriginBase {
    constructor(domainName, props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_OriginProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, OriginBase);
            }
            throw error;
        }
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, props.connectionTimeout?.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        validateCustomHeaders(props.customHeaders);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
        this.originShieldRegion = props.originShieldRegion;
        this.originId = props.originId;
        this.originShieldEnabled = props.originShieldEnabled ?? true;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(_scope, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_OriginBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            originProperty: {
                domainName: this.domainName,
                id: this.originId ?? options.originId,
                originPath: this.originPath,
                connectionAttempts: this.connectionAttempts,
                connectionTimeout: this.connectionTimeout?.toSeconds(),
                originCustomHeaders: this.renderCustomHeaders(),
                s3OriginConfig,
                customOriginConfig,
                originShield: this.renderOriginShield(this.originShieldEnabled, this.originShieldRegion),
            },
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.slice(0, -1);
        }
        return path;
    }
    /**
     * Takes origin shield region and converts to CfnDistribution.OriginShieldProperty
     */
    renderOriginShield(originShieldEnabled, originShieldRegion) {
        if (!originShieldEnabled) {
            return { enabled: false };
        }
        return originShieldRegion ? { enabled: true, originShieldRegion } : undefined;
    }
}
_a = JSII_RTTI_SYMBOL_1;
OriginBase[_a] = { fqn: "aws-cdk-lib.aws_cloudfront.OriginBase", version: "2.74.0" };
exports.OriginBase = OriginBase;
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
/**
 * Throws an error if custom header assignment is prohibited by CloudFront.
 * @link: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html#add-origin-custom-headers-denylist
 */
function validateCustomHeaders(customHeaders) {
    if (!customHeaders || Object.entries(customHeaders).length === 0) {
        return;
    }
    const customHeaderKeys = Object.keys(customHeaders);
    const prohibitedHeaderKeys = [
        'Cache-Control', 'Connection', 'Content-Length', 'Cookie', 'Host', 'If-Match', 'If-Modified-Since', 'If-None-Match', 'If-Range', 'If-Unmodified-Since',
        'Max-Forwards', 'Pragma', 'Proxy-Authorization', 'Proxy-Connection', 'Range', 'Request-Range', 'TE', 'Trailer', 'Transfer-Encoding', 'Upgrade', 'Via',
        'X-Real-Ip',
    ];
    const prohibitedHeaderKeyPrefixes = [
        'X-Amz-', 'X-Edge-',
    ];
    const prohibitedHeadersKeysMatches = customHeaderKeys.filter(customKey => {
        return prohibitedHeaderKeys.map((prohibitedKey) => prohibitedKey.toLowerCase()).includes(customKey.toLowerCase());
    });
    const prohibitedHeaderPrefixMatches = customHeaderKeys.filter(customKey => {
        return prohibitedHeaderKeyPrefixes.some(prohibitedKeyPrefix => customKey.toLowerCase().startsWith(prohibitedKeyPrefix.toLowerCase()));
    });
    if (prohibitedHeadersKeysMatches.length !== 0) {
        throw new Error(`The following headers cannot be configured as custom origin headers: ${prohibitedHeadersKeysMatches.join(', ')}`);
    }
    if (prohibitedHeaderPrefixMatches.length !== 0) {
        throw new Error(`The following headers cannot be used as prefixes for custom origin headers: ${prohibitedHeaderPrefixMatches.join(', ')}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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