"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontWebDistribution = exports.ViewerCertificate = exports.CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedMethods = exports.OriginSslPolicy = exports.FailoverStatusCode = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const certificatemanager = require("../../aws-certificatemanager");
const iam = require("../../aws-iam");
const s3 = require("../../aws-s3");
const cdk = require("../../core");
const cloudfront_generated_1 = require("./cloudfront.generated");
const distribution_1 = require("./distribution");
const utils_1 = require("./private/utils");
/**
 * HTTP status code to failover to second origin
 */
var FailoverStatusCode;
(function (FailoverStatusCode) {
    /**
     * Forbidden (403)
     */
    FailoverStatusCode[FailoverStatusCode["FORBIDDEN"] = 403] = "FORBIDDEN";
    /**
     * Not found (404)
     */
    FailoverStatusCode[FailoverStatusCode["NOT_FOUND"] = 404] = "NOT_FOUND";
    /**
     * Internal Server Error (500)
     */
    FailoverStatusCode[FailoverStatusCode["INTERNAL_SERVER_ERROR"] = 500] = "INTERNAL_SERVER_ERROR";
    /**
     * Bad Gateway (502)
     */
    FailoverStatusCode[FailoverStatusCode["BAD_GATEWAY"] = 502] = "BAD_GATEWAY";
    /**
     * Service Unavailable (503)
     */
    FailoverStatusCode[FailoverStatusCode["SERVICE_UNAVAILABLE"] = 503] = "SERVICE_UNAVAILABLE";
    /**
     * Gateway Timeout (504)
     */
    FailoverStatusCode[FailoverStatusCode["GATEWAY_TIMEOUT"] = 504] = "GATEWAY_TIMEOUT";
})(FailoverStatusCode = exports.FailoverStatusCode || (exports.FailoverStatusCode = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_certificatemanager_ICertificate(certificate);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_ViewerCertificateOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAcmCertificate);
            }
            throw error;
        }
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_ViewerCertificateOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromIamCertificate);
            }
            throw error;
        }
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a `SecurityPolicyProtocol.TLS_V1` security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
}
_a = JSII_RTTI_SYMBOL_1;
ViewerCertificate[_a] = { fqn: "aws-cdk-lib.aws_cloudfront.ViewerCertificate", version: "2.74.0" };
exports.ViewerCertificate = ViewerCertificate;
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * const sourceBucket = new s3.Bucket(this, 'Bucket');
 *
 * const distribution = new cloudfront.CloudFrontWebDistribution(this, 'MyDistribution', {
 *   originConfigs: [
 *     {
 *       s3OriginSource: {
 *       s3BucketSource: sourceBucket,
 *       },
 *       behaviors : [ {isDefaultBehavior: true}],
 *     },
 *   ],
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 * @resource AWS::CloudFront::Distribution
 */
class CloudFrontWebDistribution extends cdk.Resource {
    /**
     * Creates a construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_CloudFrontWebDistributionAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDistributionAttributes);
            }
            throw error;
        }
        return new class extends cdk.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
            grant(grantee, ...actions) {
                return iam.Grant.addToPrincipal({ grantee, actions, resourceArns: [(0, utils_1.formatDistributionArn)(this)] });
            }
            grantCreateInvalidation(identity) {
                return this.grant(identity, 'cloudfront:CreateInvalidation');
            }
        }();
    }
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [distribution_1.SSLMethod.SNI]: [
                distribution_1.SecurityPolicyProtocol.TLS_V1, distribution_1.SecurityPolicyProtocol.TLS_V1_1_2016,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2016, distribution_1.SecurityPolicyProtocol.TLS_V1_2_2018,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2_2019, distribution_1.SecurityPolicyProtocol.TLS_V1_2_2021,
            ],
            [distribution_1.SSLMethod.VIP]: [distribution_1.SecurityPolicyProtocol.SSL_V3, distribution_1.SecurityPolicyProtocol.TLS_V1],
        };
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudfront_CloudFrontWebDistributionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudFrontWebDistribution);
            }
            throw error;
        }
        // Comments have an undocumented limit of 128 characters
        const trimmedComment = props.comment && props.comment.length > 128
            ? `${props.comment.slice(0, 128 - 3)}...`
            : props.comment;
        const behaviors = [];
        const origins = [];
        const originGroups = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            let originId = `origin${originIndex}`;
            const originProperty = this.toOriginProperty(originConfig, originId);
            if (originConfig.failoverCustomOriginSource || originConfig.failoverS3OriginSource) {
                const originSecondaryId = `originSecondary${originIndex}`;
                const originSecondaryProperty = this.toOriginProperty({
                    s3OriginSource: originConfig.failoverS3OriginSource,
                    customOriginSource: originConfig.failoverCustomOriginSource,
                    originPath: originConfig.originPath,
                    originHeaders: originConfig.originHeaders,
                    originShieldRegion: originConfig.originShieldRegion,
                }, originSecondaryId);
                const originGroupsId = `OriginGroup${originIndex}`;
                const failoverCodes = originConfig.failoverCriteriaStatusCodes ?? [500, 502, 503, 504];
                originGroups.push({
                    id: originGroupsId,
                    members: {
                        items: [{ originId }, { originId: originSecondaryId }],
                        quantity: 2,
                    },
                    failoverCriteria: {
                        statusCodes: {
                            items: failoverCodes,
                            quantity: failoverCodes.length,
                        },
                    },
                });
                originId = originGroupsId;
                origins.push(originSecondaryProperty);
            }
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        const originGroupsDistConfig = originGroups.length > 0
            ? {
                items: originGroups,
                quantity: originGroups.length,
            }
            : undefined;
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        let distributionConfig = {
            comment: trimmedComment,
            enabled: props.enabled ?? true,
            defaultRootObject: props.defaultRootObject ?? 'index.html',
            httpVersion: props.httpVersion || distribution_1.HttpVersion.HTTP2,
            priceClass: props.priceClass || distribution_1.PriceClass.PRICE_CLASS_100,
            ipv6Enabled: props.enableIpV6 ?? true,
            // eslint-disable-next-line max-len
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
            origins,
            originGroups: originGroupsDistConfig,
            defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy),
            cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined,
        };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // eslint-disable-next-line max-len
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = {
                ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket', {
                encryption: s3.BucketEncryption.S3_MANAGED,
            });
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketRegionalDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        if (props.geoRestriction) {
            distributionConfig = {
                ...distributionConfig,
                restrictions: {
                    geoRestriction: {
                        restrictionType: props.geoRestriction.restrictionType,
                        locations: props.geoRestriction.locations,
                    },
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Adds an IAM policy statement associated with this distribution to an IAM
     * principal's policy.
     *
     * @param identity The principal
     * @param actions The set of actions to allow (i.e. "cloudfront:ListInvalidations")
     */
    grant(identity, ...actions) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grant);
            }
            throw error;
        }
        return iam.Grant.addToPrincipal({ grantee: identity, actions, resourceArns: [(0, utils_1.formatDistributionArn)(this)] });
    }
    /**
     * Grant to create invalidations for this bucket to an IAM principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantCreateInvalidation(identity) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantCreateInvalidation);
            }
            throw error;
        }
        return this.grant(identity, 'cloudfront:CreateInvalidation');
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedKeyGroups: input.trustedKeyGroups?.map(key => key.keyGroupId),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: input.viewerProtocolPolicy || protoPolicy || distribution_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.functionAssociations) {
            toReturn = Object.assign(toReturn, {
                functionAssociations: input.functionAssociations.map(association => ({
                    functionArn: association.function.functionArn,
                    eventType: association.eventType.toString(),
                })),
            });
        }
        if (input.lambdaFunctionAssociations) {
            const includeBodyEventTypes = [distribution_1.LambdaEdgeEventType.ORIGIN_REQUEST, distribution_1.LambdaEdgeEventType.VIEWER_REQUEST];
            if (input.lambdaFunctionAssociations.some(fna => fna.includeBody && !includeBodyEventTypes.includes(fna.eventType))) {
                throw new Error('\'includeBody\' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types.');
            }
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.edgeArn,
                    includeBody: fna.includeBody,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && iam.Role.isRole(a.lambdaFunction.role) && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
    toOriginProperty(originConfig, originId) {
        if (!originConfig.s3OriginSource &&
            !originConfig.customOriginSource) {
            throw new Error('There must be at least one origin source - either an s3OriginSource, a customOriginSource');
        }
        if (originConfig.customOriginSource && originConfig.s3OriginSource) {
            throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
        }
        if ([
            originConfig.originHeaders,
            originConfig.s3OriginSource?.originHeaders,
            originConfig.customOriginSource?.originHeaders,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originHeaders field allowed across origin and failover origins');
        }
        if ([
            originConfig.originPath,
            originConfig.s3OriginSource?.originPath,
            originConfig.customOriginSource?.originPath,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originPath field allowed across origin and failover origins');
        }
        if ([
            originConfig.originShieldRegion,
            originConfig.s3OriginSource?.originShieldRegion,
            originConfig.customOriginSource?.originShieldRegion,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originShieldRegion field allowed across origin and failover origins');
        }
        const headers = originConfig.originHeaders ?? originConfig.s3OriginSource?.originHeaders ?? originConfig.customOriginSource?.originHeaders;
        const originHeaders = [];
        if (headers) {
            Object.keys(headers).forEach((key) => {
                const oHeader = {
                    headerName: key,
                    headerValue: headers[key],
                };
                originHeaders.push(oHeader);
            });
        }
        let s3OriginConfig;
        if (originConfig.s3OriginSource) {
            // first case for backwards compatibility
            if (originConfig.s3OriginSource.originAccessIdentity) {
                // grant CloudFront OriginAccessIdentity read access to S3 bucket
                // Used rather than `grantRead` because `grantRead` will grant overly-permissive policies.
                // Only GetObject is needed to retrieve objects for the distribution.
                // This also excludes KMS permissions; currently, OAI only supports SSE-S3 for buckets.
                // Source: https://aws.amazon.com/blogs/networking-and-content-delivery/serving-sse-kms-encrypted-content-from-s3-using-cloudfront/
                originConfig.s3OriginSource.s3BucketSource.addToResourcePolicy(new iam.PolicyStatement({
                    resources: [originConfig.s3OriginSource.s3BucketSource.arnForObjects('*')],
                    actions: ['s3:GetObject'],
                    principals: [originConfig.s3OriginSource.originAccessIdentity.grantPrincipal],
                }));
                s3OriginConfig = {
                    originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityId}`,
                };
            }
            else {
                s3OriginConfig = {};
            }
        }
        const connectionAttempts = originConfig.connectionAttempts ?? 3;
        if (connectionAttempts < 1 || 3 < connectionAttempts || !Number.isInteger(connectionAttempts)) {
            throw new Error('connectionAttempts: You can specify 1, 2, or 3 as the number of attempts.');
        }
        const connectionTimeout = (originConfig.connectionTimeout || cdk.Duration.seconds(10)).toSeconds();
        if (connectionTimeout < 1 || 10 < connectionTimeout || !Number.isInteger(connectionTimeout)) {
            throw new Error('connectionTimeout: You can specify a number of seconds between 1 and 10 (inclusive).');
        }
        const originProperty = {
            id: originId,
            domainName: originConfig.s3OriginSource
                ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                : originConfig.customOriginSource.domainName,
            originPath: originConfig.originPath ?? originConfig.customOriginSource?.originPath ?? originConfig.s3OriginSource?.originPath,
            originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
            s3OriginConfig,
            originShield: this.toOriginShieldProperty(originConfig),
            customOriginConfig: originConfig.customOriginSource
                ? {
                    httpPort: originConfig.customOriginSource.httpPort || 80,
                    httpsPort: originConfig.customOriginSource.httpsPort || 443,
                    originKeepaliveTimeout: (originConfig.customOriginSource.originKeepaliveTimeout &&
                        originConfig.customOriginSource.originKeepaliveTimeout.toSeconds()) ||
                        5,
                    originReadTimeout: (originConfig.customOriginSource.originReadTimeout &&
                        originConfig.customOriginSource.originReadTimeout.toSeconds()) ||
                        30,
                    originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy ||
                        distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
                    originSslProtocols: originConfig.customOriginSource
                        .allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                }
                : undefined,
            connectionAttempts,
            connectionTimeout,
        };
        return originProperty;
    }
    /**
     * Takes origin shield region from props and converts to CfnDistribution.OriginShieldProperty
     */
    toOriginShieldProperty(originConfig) {
        const originShieldRegion = originConfig.originShieldRegion ??
            originConfig.customOriginSource?.originShieldRegion ??
            originConfig.s3OriginSource?.originShieldRegion;
        return originShieldRegion
            ? { enabled: true, originShieldRegion }
            : undefined;
    }
}
_b = JSII_RTTI_SYMBOL_1;
CloudFrontWebDistribution[_b] = { fqn: "aws-cdk-lib.aws_cloudfront.CloudFrontWebDistribution", version: "2.74.0" };
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,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