"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataResourceType = exports.ManagementEventSources = exports.Trail = exports.InsightType = exports.ReadWriteType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const events = require("../../aws-events");
const iam = require("../../aws-iam");
const logs = require("../../aws-logs");
const s3 = require("../../aws-s3");
const core_1 = require("../../core");
const cloudtrail_generated_1 = require("./cloudtrail.generated");
/**
 * Types of events that CloudTrail can log
 */
var ReadWriteType;
(function (ReadWriteType) {
    /**
     * Read-only events include API operations that read your resources,
     * but don't make changes.
     * For example, read-only events include the Amazon EC2 DescribeSecurityGroups
     * and DescribeSubnets API operations.
     */
    ReadWriteType["READ_ONLY"] = "ReadOnly";
    /**
     * Write-only events include API operations that modify (or might modify)
     * your resources.
     * For example, the Amazon EC2 RunInstances and TerminateInstances API
     * operations modify your instances.
     */
    ReadWriteType["WRITE_ONLY"] = "WriteOnly";
    /**
     * All events
     */
    ReadWriteType["ALL"] = "All";
    /**
     * No events
     */
    ReadWriteType["NONE"] = "None";
})(ReadWriteType = exports.ReadWriteType || (exports.ReadWriteType = {}));
/**
 * Util element for InsightSelector
 */
class InsightType {
    constructor(value) {
        this.value = value;
    }
}
_a = JSII_RTTI_SYMBOL_1;
InsightType[_a] = { fqn: "aws-cdk-lib.aws_cloudtrail.InsightType", version: "2.74.0" };
/**
 * The type of insights to log on a trail. (API Call Rate)
 */
InsightType.API_CALL_RATE = new InsightType('ApiCallRateInsight');
/**
 * The type of insights to log on a trail. (API Error Rate)
 */
InsightType.API_ERROR_RATE = new InsightType('ApiErrorRateInsight');
exports.InsightType = InsightType;
/**
 * Cloud trail allows you to log events that happen in your AWS account
 * For example:
 *
 * import { CloudTrail } from '@aws-cdk/aws-cloudtrail'
 *
 * const cloudTrail = new CloudTrail(this, 'MyTrail');
 *
 */
class Trail extends core_1.Resource {
    /**
     * Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     */
    static onEvent(scope, id, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_events_OnEventOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.onEvent);
            }
            throw error;
        }
        const rule = new events.Rule(scope, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            detailType: ['AWS API Call via CloudTrail'],
        });
        return rule;
    }
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.trailName,
        });
        this.eventSelectors = [];
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudtrail_TrailProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Trail);
            }
            throw error;
        }
        const cloudTrailPrincipal = new iam.ServicePrincipal('cloudtrail.amazonaws.com');
        this.s3bucket = props.bucket || new s3.Bucket(this, 'S3', { enforceSSL: true });
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.bucketArn],
            actions: ['s3:GetBucketAcl'],
            principals: [cloudTrailPrincipal],
        }));
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.arnForObjects(`${props.s3KeyPrefix ? `${props.s3KeyPrefix}/` : ''}AWSLogs/${core_1.Stack.of(this).account}/*`)],
            actions: ['s3:PutObject'],
            principals: [cloudTrailPrincipal],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        this.topic = props.snsTopic;
        if (this.topic) {
            this.topic.grantPublish(cloudTrailPrincipal);
        }
        let logsRole;
        if (props.sendToCloudWatchLogs) {
            if (props.cloudWatchLogGroup) {
                this.logGroup = props.cloudWatchLogGroup;
            }
            else {
                this.logGroup = new logs.LogGroup(this, 'LogGroup', {
                    retention: props.cloudWatchLogsRetention ?? logs.RetentionDays.ONE_YEAR,
                });
            }
            logsRole = new iam.Role(this, 'LogsRole', { assumedBy: cloudTrailPrincipal });
            logsRole.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: [this.logGroup.logGroupArn],
            }));
        }
        this.managementEvents = props.managementEvents;
        if (this.managementEvents && this.managementEvents !== ReadWriteType.NONE) {
            this.eventSelectors.push({
                includeManagementEvents: true,
                readWriteType: props.managementEvents,
            });
        }
        this.node.addValidation({ validate: () => this.validateEventSelectors() });
        if (props.kmsKey && props.encryptionKey) {
            throw new Error('Both kmsKey and encryptionKey must not be specified. Use only encryptionKey');
        }
        if (props.insightTypes) {
            this.insightTypeValues = props.insightTypes.map(function (t) {
                return { insightType: t.value };
            });
        }
        // TODO: not all regions support validation. Use service configuration data to fail gracefully
        const trail = new cloudtrail_generated_1.CfnTrail(this, 'Resource', {
            isLogging: true,
            enableLogFileValidation: props.enableFileValidation == null ? true : props.enableFileValidation,
            isMultiRegionTrail: props.isMultiRegionTrail == null ? true : props.isMultiRegionTrail,
            includeGlobalServiceEvents: props.includeGlobalServiceEvents == null ? true : props.includeGlobalServiceEvents,
            trailName: this.physicalName,
            kmsKeyId: props.encryptionKey?.keyArn ?? props.kmsKey?.keyArn,
            s3BucketName: this.s3bucket.bucketName,
            s3KeyPrefix: props.s3KeyPrefix,
            cloudWatchLogsLogGroupArn: this.logGroup?.logGroupArn,
            cloudWatchLogsRoleArn: logsRole?.roleArn,
            snsTopicName: this.topic?.topicName,
            eventSelectors: this.eventSelectors,
            isOrganizationTrail: props.isOrganizationTrail,
            insightSelectors: this.insightTypeValues,
        });
        this.trailArn = this.getResourceArnAttribute(trail.attrArn, {
            service: 'cloudtrail',
            resource: 'trail',
            resourceName: this.physicalName,
        });
        this.trailSnsTopicArn = trail.attrSnsTopicArn;
        // Add a dependency on the bucket policy being updated, CloudTrail will test this upon creation.
        if (this.s3bucket.policy) {
            trail.node.addDependency(this.s3bucket.policy);
        }
        // If props.sendToCloudWatchLogs is set to true then the trail needs to depend on the created logsRole
        // so that it can create the log stream for the log group. This ensures the logsRole is created and propagated
        // before the trail tries to create the log stream.
        if (logsRole !== undefined) {
            trail.node.addDependency(logsRole);
        }
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an Event Selector for filtering events that match either S3 or Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param dataResourceValues the list of data resource ARNs to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addEventSelector(dataResourceType, dataResourceValues, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudtrail_DataResourceType(dataResourceType);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudtrail_AddEventSelectorOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addEventSelector);
            }
            throw error;
        }
        if (dataResourceValues.length > 250) {
            throw new Error('A maximum of 250 data elements can be in one event selector');
        }
        if (this.eventSelectors.length > 5) {
            throw new Error('A maximum of 5 event selectors are supported per trail.');
        }
        let includeAllManagementEvents;
        if (this.managementEvents === ReadWriteType.NONE) {
            includeAllManagementEvents = false;
        }
        this.eventSelectors.push({
            dataResources: [{
                    type: dataResourceType,
                    values: dataResourceValues,
                }],
            includeManagementEvents: options.includeManagementEvents ?? includeAllManagementEvents,
            excludeManagementEventSources: options.excludeManagementEventSources,
            readWriteType: options.readWriteType,
        });
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds a Lambda Data Event Selector for filtering events that match Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param handlers the list of lambda function handlers whose data events should be logged (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addLambdaEventSelector(handlers, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudtrail_AddEventSelectorOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLambdaEventSelector);
            }
            throw error;
        }
        if (handlers.length === 0) {
            return;
        }
        const dataResourceValues = handlers.map((h) => h.functionArn);
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, dataResourceValues, options);
    }
    /**
     * Log all Lambda data events for all lambda functions the account.
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @default false
     */
    logAllLambdaDataEvents(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudtrail_AddEventSelectorOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.logAllLambdaDataEvents);
            }
            throw error;
        }
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, [`arn:${this.stack.partition}:lambda`], options);
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an S3 Data Event Selector for filtering events that match S3 operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param s3Selector the list of S3 bucket with optional prefix to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addS3EventSelector(s3Selector, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudtrail_AddEventSelectorOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addS3EventSelector);
            }
            throw error;
        }
        if (s3Selector.length === 0) {
            return;
        }
        const dataResourceValues = s3Selector.map((sel) => `${sel.bucket.bucketArn}/${sel.objectPrefix ?? ''}`);
        return this.addEventSelector(DataResourceType.S3_OBJECT, dataResourceValues, options);
    }
    /**
     * Log all S3 data events for all objects for all buckets in the account.
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @default false
     */
    logAllS3DataEvents(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudtrail_AddEventSelectorOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.logAllS3DataEvents);
            }
            throw error;
        }
        return this.addEventSelector(DataResourceType.S3_OBJECT, [`arn:${this.stack.partition}:s3:::`], options);
    }
    /**
     * Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     *
     * @deprecated - use Trail.onEvent()
     */
    onCloudTrailEvent(id, options = {}) {
        return Trail.onEvent(this, id, options);
    }
    validateEventSelectors() {
        const errors = [];
        // Ensure that there is at least one event selector when management events are set to None
        if (this.managementEvents === ReadWriteType.NONE && this.eventSelectors.length === 0) {
            errors.push('At least one event selector must be added when management event recording is set to None');
        }
        return errors;
    }
}
_b = JSII_RTTI_SYMBOL_1;
Trail[_b] = { fqn: "aws-cdk-lib.aws_cloudtrail.Trail", version: "2.74.0" };
exports.Trail = Trail;
/**
 * Types of management event sources that can be excluded
 */
var ManagementEventSources;
(function (ManagementEventSources) {
    /**
     * AWS Key Management Service (AWS KMS) events
     */
    ManagementEventSources["KMS"] = "kms.amazonaws.com";
    /**
     * Data API events
     */
    ManagementEventSources["RDS_DATA_API"] = "rdsdata.amazonaws.com";
})(ManagementEventSources = exports.ManagementEventSources || (exports.ManagementEventSources = {}));
/**
 * Resource type for a data event
 */
var DataResourceType;
(function (DataResourceType) {
    /**
     * Data resource type for Lambda function
     */
    DataResourceType["LAMBDA_FUNCTION"] = "AWS::Lambda::Function";
    /**
     * Data resource type for S3 objects
     */
    DataResourceType["S3_OBJECT"] = "AWS::S3::Object";
})(DataResourceType = exports.DataResourceType || (exports.DataResourceType = {}));
//# sourceMappingURL=data:application/json;base64,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