"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.normalizeRawStringStatistic = exports.normalizeStatistic = exports.parseStatistic = exports.pairStatisticToString = exports.singleStatisticToString = void 0;
const stats_1 = require("../stats");
function parseSingleStatistic(statistic, prefix) {
    const prefixLower = prefix.toLowerCase();
    // Allow `P99` uppercase
    statistic = statistic.toLowerCase();
    if (!statistic.startsWith(prefixLower)) {
        return undefined;
    }
    // A decimal positive number regex (1, 1.2, 99.999, etc)
    const reDecimal = '\\d+(?:\\.\\d+)?';
    // p99.99
    // /^p(\d+(?:\.\d+)?)$/
    const r = new RegExp(`^${prefixLower}(${reDecimal})$`).exec(statistic);
    if (!r) {
        return undefined;
    }
    const value = parseFloat(r[1]);
    if (value < 0 || value > 100) {
        return undefined;
    }
    return {
        type: 'single',
        rawStatistic: statistic,
        statPrefix: prefixLower,
        value,
    };
}
/**
 * Parse a statistic that looks like `tm( LOWER : UPPER )`.
 */
function parsePairStatistic(statistic, prefix) {
    const r = new RegExp(`^${prefix}\\(([^)]+)\\)$`, 'i').exec(statistic);
    if (!r) {
        return undefined;
    }
    const common = {
        type: 'pair',
        canBeSingleStat: false,
        rawStatistic: statistic,
        statPrefix: prefix.toUpperCase(),
    };
    const [lhs, rhs] = r[1].split(':');
    if (rhs === undefined) {
        // Doesn't have 2 parts
        return undefined;
    }
    const parseNumberAndPercent = (x) => {
        x = x.trim();
        if (!x) {
            return [undefined, false];
        }
        const value = parseFloat(x.replace(/%$/, ''));
        const percent = x.endsWith('%');
        if (isNaN(value) || value < 0 || (percent && value > 100)) {
            return ['fail', false];
        }
        return [value, percent];
    };
    const [lower, lhsPercent] = parseNumberAndPercent(lhs);
    const [upper, rhsPercent] = parseNumberAndPercent(rhs);
    if (lower === 'fail' || upper === 'fail' || (lower === undefined && upper === undefined)) {
        return undefined;
    }
    if (lower !== undefined && upper !== undefined && lhsPercent !== rhsPercent) {
        // If one value is a percentage, the other one must be too
        return undefined;
    }
    const isPercent = lhsPercent || rhsPercent;
    const canBeSingleStat = lower === undefined && isPercent;
    const asSingleStatStr = canBeSingleStat ? `${prefix.toLowerCase()}${upper}` : undefined;
    return { ...common, lower, upper, isPercent, canBeSingleStat, asSingleStatStr };
}
function singleStatisticToString(parsed) {
    return `${parsed.statPrefix}${parsed.value}`;
}
exports.singleStatisticToString = singleStatisticToString;
function pairStatisticToString(parsed) {
    const percent = parsed.isPercent ? '%' : '';
    const lower = parsed.lower ? `${parsed.lower}${percent}` : '';
    const upper = parsed.upper ? `${parsed.upper}${percent}` : '';
    return `${parsed.statPrefix}(${lower}:${upper})`;
}
exports.pairStatisticToString = pairStatisticToString;
/**
 * Parse a statistic, returning the type of metric that was used
 */
function parseStatistic(stat) {
    const lowerStat = stat.toLowerCase();
    // Simple statistics
    const statMap = {
        average: stats_1.Stats.AVERAGE,
        avg: stats_1.Stats.AVERAGE,
        minimum: stats_1.Stats.MINIMUM,
        min: stats_1.Stats.MINIMUM,
        maximum: stats_1.Stats.MAXIMUM,
        max: stats_1.Stats.MAXIMUM,
        samplecount: stats_1.Stats.SAMPLE_COUNT,
        n: stats_1.Stats.SAMPLE_COUNT,
        sum: stats_1.Stats.SUM,
        iqm: stats_1.Stats.IQM,
    };
    if (lowerStat in statMap) {
        return {
            type: 'simple',
            statistic: statMap[lowerStat],
        };
    }
    let m = undefined;
    // Percentile statistics
    m = parseSingleStatistic(stat, 'p');
    if (m)
        return { ...m, statName: 'percentile' };
    // Trimmed mean statistics
    m = parseSingleStatistic(stat, 'tm') || parsePairStatistic(stat, 'tm');
    if (m)
        return { ...m, statName: 'trimmedMean' };
    // Winsorized mean statistics
    m = parseSingleStatistic(stat, 'wm') || parsePairStatistic(stat, 'wm');
    if (m)
        return { ...m, statName: 'winsorizedMean' };
    // Trimmed count statistics
    m = parseSingleStatistic(stat, 'tc') || parsePairStatistic(stat, 'tc');
    if (m)
        return { ...m, statName: 'trimmedCount' };
    // Trimmed sum statistics
    m = parseSingleStatistic(stat, 'ts') || parsePairStatistic(stat, 'ts');
    if (m)
        return { ...m, statName: 'trimmedSum' };
    return {
        type: 'generic',
        statistic: stat,
    };
}
exports.parseStatistic = parseStatistic;
function normalizeStatistic(parsed) {
    if (parsed.type === 'simple' || parsed.type === 'generic') {
        return parsed.statistic;
    }
    else if (parsed.type === 'single') {
        // Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the stat prefix.
        return parsed.rawStatistic.toLowerCase();
    }
    else if (parsed.type === 'pair') {
        // Avoid parsing because we might get into
        // floating point rounding issues, return as-is but uppercase the stat prefix.
        return parsed.rawStatistic.toUpperCase();
    }
    return '';
}
exports.normalizeStatistic = normalizeStatistic;
function normalizeRawStringStatistic(stat) {
    const parsed = parseStatistic(stat);
    return normalizeStatistic(parsed);
}
exports.normalizeRawStringStatistic = normalizeRawStringStatistic;
//# sourceMappingURL=data:application/json;base64,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