"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxArmBuildImage = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const project_1 = require("./project");
/**
 * A CodeBuild image running aarch64 Linux.
 *
 * This class has a bunch of public constants that represent the CodeBuild ARM images.
 *
 * You can also specify a custom image using the static method:
 *
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 *
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class LinuxArmBuildImage {
    /**
     * Returns an ARM image running Linux from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tagOrDigest Image tag or digest (default "latest", digests must start with `sha256:`)
     * @returns An aarch64 Linux build image from an ECR repository.
     */
    static fromEcrRepository(repository, tagOrDigest = 'latest') {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecr_IRepository(repository);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromEcrRepository);
            }
            throw error;
        }
        return new LinuxArmBuildImage({
            imageId: repository.repositoryUriForTagOrDigest(tagOrDigest),
            imagePullPrincipalType: project_1.ImagePullPrincipalType.SERVICE_ROLE,
            repository,
        });
    }
    /**
     * Uses a Docker image provided by CodeBuild.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     *
     * @param id The image identifier
     * @example 'aws/codebuild/amazonlinux2-aarch64-standard:1.0'
     * @returns A Docker image provided by CodeBuild.
     */
    static fromCodeBuildImageId(id) {
        return new LinuxArmBuildImage({
            imageId: id,
            imagePullPrincipalType: project_1.ImagePullPrincipalType.CODEBUILD,
        });
    }
    constructor(props) {
        this.type = 'ARM_CONTAINER';
        this.defaultComputeType = project_1.ComputeType.LARGE;
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * Validates by checking the BuildEnvironment computeType as aarch64 images only support ComputeType.SMALL and
     * ComputeType.LARGE
     * @param buildEnvironment BuildEnvironment
     */
    validate(buildEnvironment) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_BuildEnvironment(buildEnvironment);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.validate);
            }
            throw error;
        }
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== project_1.ComputeType.SMALL &&
            buildEnvironment.computeType !== project_1.ComputeType.LARGE) {
            ret.push(`ARM images only support ComputeTypes '${project_1.ComputeType.SMALL}' and '${project_1.ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return (0, run_script_linux_build_spec_1.runScriptLinuxBuildSpec)(entrypoint);
    }
}
_a = JSII_RTTI_SYMBOL_1;
LinuxArmBuildImage[_a] = { fqn: "aws-cdk-lib.aws_codebuild.LinuxArmBuildImage", version: "2.74.0" };
/** Image "aws/codebuild/amazonlinux2-aarch64-standard:1.0". */
LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_1_0 = LinuxArmBuildImage.fromCodeBuildImageId('aws/codebuild/amazonlinux2-aarch64-standard:1.0');
/** Image "aws/codebuild/amazonlinux2-aarch64-standard:2.0". */
LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_2_0 = LinuxArmBuildImage.fromCodeBuildImageId('aws/codebuild/amazonlinux2-aarch64-standard:2.0');
exports.LinuxArmBuildImage = LinuxArmBuildImage;
//# sourceMappingURL=data:application/json;base64,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