"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxGpuBuildImage = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ecr = require("../../aws-ecr");
const core = require("../../core");
const region_info_1 = require("../../region-info");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const project_1 = require("./project");
/**
 * A CodeBuild GPU image running Linux.
 *
 * This class has public constants that represent the most popular GPU images from AWS Deep Learning Containers.
 *
 * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
 */
class LinuxGpuBuildImage {
    /**
     * Returns a Linux GPU build image from AWS Deep Learning Containers.
     *
     * @param repositoryName the name of the repository,
     *   for example "pytorch-inference"
     * @param tag the tag of the image, for example "1.5.0-gpu-py36-cu101-ubuntu16.04"
     * @param account the AWS account ID where the DLC repository for this region is hosted in.
     *   In many cases, the CDK can infer that for you, but for some newer region our information
     *   might be out of date; in that case, you can specify the region explicitly using this optional parameter
     * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
     */
    static awsDeepLearningContainersImage(repositoryName, tag, account) {
        return new LinuxGpuBuildImage(repositoryName, tag, account);
    }
    /**
     * Returns a GPU image running Linux from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest') {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecr_IRepository(repository);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromEcrRepository);
            }
            throw error;
        }
        return new LinuxGpuBuildImage(repository.repositoryName, tag, repository.env.account);
    }
    constructor(repositoryName, tag, account) {
        this.repositoryName = repositoryName;
        this.account = account;
        this.type = 'LINUX_GPU_CONTAINER';
        this.defaultComputeType = project_1.ComputeType.LARGE;
        this.imagePullPrincipalType = project_1.ImagePullPrincipalType.SERVICE_ROLE;
        const imageAccount = account ?? core.Lazy.string({
            produce: () => {
                if (this._imageAccount === undefined) {
                    throw new Error('Make sure this \'LinuxGpuBuildImage\' is used in a CodeBuild Project construct');
                }
                return this._imageAccount;
            },
        });
        // The value of imageId below *should* have been `Lazy.stringValue(() => repository.repositoryUriForTag(this.tag))`,
        // but we can't change that anymore because someone somewhere might at this point have written code
        // to do `image.imageId.includes('pytorch')` and changing this to a full-on token would break them.
        this.imageId = `${imageAccount}.dkr.ecr.${core.Aws.REGION}.${core.Aws.URL_SUFFIX}/${repositoryName}:${tag}`;
    }
    bind(scope, project, _options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_IProject(project);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_BuildImageBindOptions(_options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        const account = this.account ?? core.Stack.of(scope).regionalFact(region_info_1.FactName.DLC_REPOSITORY_ACCOUNT);
        const repository = ecr.Repository.fromRepositoryAttributes(scope, 'AwsDlcRepositoryCodeBuild', {
            repositoryName: this.repositoryName,
            repositoryArn: ecr.Repository.arnForLocalRepository(this.repositoryName, scope, account),
        });
        repository.grantPull(project);
        this._imageAccount = account;
        return {};
    }
    validate(buildEnvironment) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_BuildEnvironment(buildEnvironment);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.validate);
            }
            throw error;
        }
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== project_1.ComputeType.LARGE) {
            ret.push(`GPU images only support ComputeType '${project_1.ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return (0, run_script_linux_build_spec_1.runScriptLinuxBuildSpec)(entrypoint);
    }
}
_a = JSII_RTTI_SYMBOL_1;
LinuxGpuBuildImage[_a] = { fqn: "aws-cdk-lib.aws_codebuild.LinuxGpuBuildImage", version: "2.74.0" };
/** Tensorflow 1.14.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_14_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.14.0-gpu-py36-cu100-ubuntu16.04');
/** Tensorflow 1.15.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.0-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 1.15.2 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.2-gpu-py37-cu100-ubuntu18.04');
/** Tensorflow 1.15.2 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '1.15.2-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 2.0.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.0-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 2.0.1 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.1-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 2.1.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/** Tensorflow 2.1.0 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/** Tensorflow 2.2.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_2_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.2.0-gpu-py37-cu101-ubuntu18.04');
/** PyTorch 1.2.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_2_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.2.0-gpu-py36-cu100-ubuntu16.04');
/** PyTorch 1.3.1 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_3_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.3.1-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.4.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.4.0 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.5.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.5.0 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/** MXNet 1.4.1 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_MXNET_1_4_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.4.1-gpu-py36-cu100-ubuntu16.04');
/** MXNet 1.6.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_MXNET_1_6_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.6.0-gpu-py36-cu101-ubuntu16.04');
exports.LinuxGpuBuildImage = LinuxGpuBuildImage;
//# sourceMappingURL=data:application/json;base64,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