"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FilterGroup = exports.EventAction = exports.Source = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const source_types_1 = require("./source-types");
/**
 * Source provider definition for a CodeBuild Project.
 */
class Source {
    static s3(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_S3SourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.s3);
            }
            throw error;
        }
        return new S3Source(props);
    }
    static codeCommit(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_CodeCommitSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.codeCommit);
            }
            throw error;
        }
        return new CodeCommitSource(props);
    }
    static gitHub(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_GitHubSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.gitHub);
            }
            throw error;
        }
        return new GitHubSource(props);
    }
    static gitHubEnterprise(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_GitHubEnterpriseSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.gitHubEnterprise);
            }
            throw error;
        }
        return new GitHubEnterpriseSource(props);
    }
    static bitBucket(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_BitBucketSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bitBucket);
            }
            throw error;
        }
        return new BitBucketSource(props);
    }
    constructor(props) {
        this.badgeSupported = false;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_SourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Source);
            }
            throw error;
        }
        this.identifier = props.identifier;
    }
    /**
     * Called by the project when the source is added so that the source can perform
     * binding operations on the source. For example, it can grant permissions to the
     * code build project to read from the S3 bucket.
     */
    bind(_scope, _project) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_IProject(_project);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        return {
            sourceProperty: {
                sourceIdentifier: this.identifier,
                type: this.type,
            },
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
Source[_a] = { fqn: "aws-cdk-lib.aws_codebuild.Source", version: "2.74.0" };
exports.Source = Source;
/**
 * A common superclass of all build sources that are backed by Git.
 */
class GitSource extends Source {
    constructor(props) {
        super(props);
        this.cloneDepth = props.cloneDepth;
        this.branchOrRef = props.branchOrRef;
        this.fetchSubmodules = props.fetchSubmodules;
    }
    bind(_scope, _project) {
        const superConfig = super.bind(_scope, _project);
        return {
            sourceVersion: this.branchOrRef,
            sourceProperty: {
                ...superConfig.sourceProperty,
                gitCloneDepth: this.cloneDepth,
                gitSubmodulesConfig: this.fetchSubmodules ? {
                    fetchSubmodules: this.fetchSubmodules,
                } : undefined,
            },
        };
    }
}
/**
 * The types of webhook event actions.
 */
var EventAction;
(function (EventAction) {
    /**
     * A push (of a branch, or a tag) to the repository.
     */
    EventAction["PUSH"] = "PUSH";
    /**
     * Creating a Pull Request.
     */
    EventAction["PULL_REQUEST_CREATED"] = "PULL_REQUEST_CREATED";
    /**
     * Updating a Pull Request.
     */
    EventAction["PULL_REQUEST_UPDATED"] = "PULL_REQUEST_UPDATED";
    /**
     * Merging a Pull Request.
     */
    EventAction["PULL_REQUEST_MERGED"] = "PULL_REQUEST_MERGED";
    /**
     * Re-opening a previously closed Pull Request.
     * Note that this event is only supported for GitHub and GitHubEnterprise sources.
     */
    EventAction["PULL_REQUEST_REOPENED"] = "PULL_REQUEST_REOPENED";
})(EventAction = exports.EventAction || (exports.EventAction = {}));
var WebhookFilterTypes;
(function (WebhookFilterTypes) {
    WebhookFilterTypes["FILE_PATH"] = "FILE_PATH";
    WebhookFilterTypes["COMMIT_MESSAGE"] = "COMMIT_MESSAGE";
    WebhookFilterTypes["HEAD_REF"] = "HEAD_REF";
    WebhookFilterTypes["ACTOR_ACCOUNT_ID"] = "ACTOR_ACCOUNT_ID";
    WebhookFilterTypes["BASE_REF"] = "BASE_REF";
})(WebhookFilterTypes || (WebhookFilterTypes = {}));
/**
 * An object that represents a group of filter conditions for a webhook.
 * Every condition in a given FilterGroup must be true in order for the whole group to be true.
 * You construct instances of it by calling the `#inEventOf` static factory method,
 * and then calling various `andXyz` instance methods to create modified instances of it
 * (this class is immutable).
 *
 * You pass instances of this class to the `webhookFilters` property when constructing a source.
 */
class FilterGroup {
    /**
     * Creates a new event FilterGroup that triggers on any of the provided actions.
     *
     * @param actions the actions to trigger the webhook on
     */
    static inEventOf(...actions) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codebuild_EventAction(actions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.inEventOf);
            }
            throw error;
        }
        return new FilterGroup(new Set(actions), []);
    }
    constructor(actions, filters) {
        if (actions.size === 0) {
            throw new Error('A filter group must contain at least one event action');
        }
        this.actions = actions;
        this.filters = filters;
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIs(branchName) {
        return this.addHeadBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIsNot(branchName) {
        return this.addHeadBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect a head commit with the given message.
     *
     * @param commitMessage the commit message (can be a regular expression)
     */
    andCommitMessageIs(commitMessage) {
        return this.addCommitMessageFilter(commitMessage, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect a head commit with the given message.
     *
     * @param commitMessage the commit message (can be a regular expression)
     */
    andCommitMessageIsNot(commitMessage) {
        return this.addCommitMessageFilter(commitMessage, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIs(tagName) {
        return this.addHeadTagFilter(tagName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIsNot(tagName) {
        return this.addHeadTagFilter(tagName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIs(pattern) {
        return this.addHeadRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIsNot(pattern) {
        return this.addHeadRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIs(pattern) {
        return this.addActorAccountId(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must not match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIsNot(pattern) {
        return this.addActorAccountId(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIs(branchName) {
        return this.addBaseBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIsNot(branchName) {
        return this.addBaseBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIs(pattern) {
        return this.addBaseRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIsNot(pattern) {
        return this.addBaseRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub, Bitbucket and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIs(pattern) {
        return this.addFilePathFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must not affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub, Bitbucket and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIsNot(pattern) {
        return this.addFilePathFilter(pattern, false);
    }
    /** @internal */
    get _actions() {
        return set2Array(this.actions);
    }
    /** @internal */
    get _filters() {
        return this.filters.slice();
    }
    /** @internal */
    _toJson() {
        const eventFilter = {
            type: 'EVENT',
            pattern: set2Array(this.actions).join(', '),
        };
        return [eventFilter].concat(this.filters);
    }
    addCommitMessageFilter(commitMessage, include) {
        return this.addFilter(WebhookFilterTypes.COMMIT_MESSAGE, commitMessage, include);
    }
    addHeadBranchFilter(branchName, include) {
        return this.addHeadRefFilter(`refs/heads/${branchName}`, include);
    }
    addHeadTagFilter(tagName, include) {
        return this.addHeadRefFilter(`refs/tags/${tagName}`, include);
    }
    addHeadRefFilter(refName, include) {
        return this.addFilter(WebhookFilterTypes.HEAD_REF, refName, include);
    }
    addActorAccountId(accountId, include) {
        return this.addFilter(WebhookFilterTypes.ACTOR_ACCOUNT_ID, accountId, include);
    }
    addBaseBranchFilter(branchName, include) {
        return this.addBaseRefFilter(`refs/heads/${branchName}`, include);
    }
    addBaseRefFilter(refName, include) {
        if (this.actions.has(EventAction.PUSH)) {
            throw new Error('A base reference condition cannot be added if a Group contains a PUSH event action');
        }
        return this.addFilter(WebhookFilterTypes.BASE_REF, refName, include);
    }
    addFilePathFilter(pattern, include) {
        return this.addFilter(WebhookFilterTypes.FILE_PATH, pattern, include);
    }
    addFilter(type, pattern, include) {
        return new FilterGroup(this.actions, this.filters.concat([{
                type,
                pattern,
                excludeMatchedPattern: include ? undefined : true,
            }]));
    }
}
_b = JSII_RTTI_SYMBOL_1;
FilterGroup[_b] = { fqn: "aws-cdk-lib.aws_codebuild.FilterGroup", version: "2.74.0" };
exports.FilterGroup = FilterGroup;
/**
 * A common superclass of all third-party build sources that are backed by Git.
 */
class ThirdPartyGitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.webhook = props.webhook;
        this.reportBuildStatus = props.reportBuildStatus ?? true;
        this.webhookFilters = props.webhookFilters || [];
        this.webhookTriggersBatchBuild = props.webhookTriggersBatchBuild;
        this.buildStatusUrl = props.buildStatusUrl;
    }
    bind(_scope, project) {
        const anyFilterGroupsProvided = this.webhookFilters.length > 0;
        const webhook = this.webhook ?? (anyFilterGroupsProvided ? true : undefined);
        if (!webhook && anyFilterGroupsProvided) {
            throw new Error('`webhookFilters` cannot be used when `webhook` is `false`');
        }
        if (!webhook && this.webhookTriggersBatchBuild) {
            throw new Error('`webhookTriggersBatchBuild` cannot be used when `webhook` is `false`');
        }
        const superConfig = super.bind(_scope, project);
        if (this.webhookTriggersBatchBuild) {
            project.enableBatchBuilds();
        }
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                reportBuildStatus: this.reportBuildStatus,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: webhook === undefined ? undefined : {
                webhook,
                buildType: this.webhookTriggersBatchBuild ? 'BUILD_BATCH' : undefined,
                filterGroups: anyFilterGroupsProvided ? this.webhookFilters.map(fg => fg._toJson()) : undefined,
            },
        };
    }
}
/**
 * CodeCommit Source definition for a CodeBuild project.
 */
class CodeCommitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.type = source_types_1.CODECOMMIT_SOURCE_TYPE;
        this.repo = props.repository;
    }
    bind(_scope, project) {
        // https://docs.aws.amazon.com/codebuild/latest/userguide/setting-up.html
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['codecommit:GitPull'],
            resources: [this.repo.repositoryArn],
        }));
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.repo.repositoryCloneUrlHttp,
            },
            sourceVersion: superConfig.sourceVersion,
        };
    }
}
/**
 * S3 bucket definition for a CodeBuild project.
 */
class S3Source extends Source {
    constructor(props) {
        super(props);
        this.type = source_types_1.S3_SOURCE_TYPE;
        this.bucket = props.bucket;
        this.path = props.path;
        this.version = props.version;
    }
    bind(_scope, project) {
        this.bucket.grantRead(project, this.path);
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: `${this.bucket.bucketName}/${this.path}`,
            },
            sourceVersion: this.version,
        };
    }
}
class CommonGithubSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.buildStatusContext = props.buildStatusContext;
    }
    bind(scope, project) {
        const superConfig = super.bind(scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                buildStatusConfig: this.buildStatusContext !== undefined || this.buildStatusUrl !== undefined
                    ? {
                        context: this.buildStatusContext,
                        targetUrl: this.buildStatusUrl,
                    }
                    : undefined,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * GitHub Source definition for a CodeBuild project.
 */
class GitHubSource extends CommonGithubSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_SOURCE_TYPE;
        this.httpsCloneUrl = `https://github.com/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, project) {
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * GitHub Enterprise Source definition for a CodeBuild project.
 */
class GitHubEnterpriseSource extends CommonGithubSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_ENTERPRISE_SOURCE_TYPE;
        this.httpsCloneUrl = props.httpsCloneUrl;
        this.ignoreSslErrors = props.ignoreSslErrors;
    }
    bind(_scope, _project) {
        if (this.hasCommitMessageFilterAndPrEvent()) {
            throw new Error('COMMIT_MESSAGE filters cannot be used with GitHub Enterprise Server pull request events');
        }
        if (this.hasFilePathFilterAndPrEvent()) {
            throw new Error('FILE_PATH filters cannot be used with GitHub Enterprise Server pull request events');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
                insecureSsl: this.ignoreSslErrors,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    hasCommitMessageFilterAndPrEvent() {
        return this.webhookFilters.some(fg => (fg._filters.some(fp => fp.type === WebhookFilterTypes.COMMIT_MESSAGE) &&
            this.hasPrEvent(fg._actions)));
    }
    hasFilePathFilterAndPrEvent() {
        return this.webhookFilters.some(fg => (fg._filters.some(fp => fp.type === WebhookFilterTypes.FILE_PATH) &&
            this.hasPrEvent(fg._actions)));
    }
    hasPrEvent(actions) {
        return actions.includes(EventAction.PULL_REQUEST_CREATED ||
            EventAction.PULL_REQUEST_MERGED ||
            EventAction.PULL_REQUEST_REOPENED ||
            EventAction.PULL_REQUEST_UPDATED);
    }
}
/**
 * BitBucket Source definition for a CodeBuild project.
 */
class BitBucketSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.BITBUCKET_SOURCE_TYPE;
        this.httpsCloneUrl = `https://bitbucket.org/${props.owner}/${props.repo}.git`;
        this.buildStatusName = props.buildStatusName;
    }
    bind(_scope, _project) {
        // BitBucket sources don't support the PULL_REQUEST_REOPENED event action
        if (this.anyWebhookFilterContainsPrReopenedEventAction()) {
            throw new Error('BitBucket sources do not support the PULL_REQUEST_REOPENED webhook event action');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
                buildStatusConfig: this.buildStatusName !== undefined || this.buildStatusUrl !== undefined
                    ? {
                        context: this.buildStatusName,
                        targetUrl: this.buildStatusUrl,
                    }
                    : undefined,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    anyWebhookFilterContainsPrReopenedEventAction() {
        return this.webhookFilters.findIndex(fg => {
            return fg._actions.findIndex(a => a === EventAction.PULL_REQUEST_REOPENED) !== -1;
        }) !== -1;
    }
}
function set2Array(set) {
    const ret = [];
    set.forEach(el => ret.push(el));
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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