"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsDeploymentGroup = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ecs = require("../../../aws-ecs");
const iam = require("../../../aws-iam");
const cdk = require("../../../core");
const cx_api_1 = require("../../../cx-api");
const constructs_1 = require("constructs");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
const codedeploy_generated_1 = require("../codedeploy.generated");
const base_deployment_group_1 = require("../private/base-deployment-group");
const utils_1 = require("../private/utils");
/**
 * A CodeDeploy deployment group that orchestrates ECS blue-green deployments.
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class EcsDeploymentGroup extends base_deployment_group_1.DeploymentGroupBase {
    /**
     * Reference an ECS Deployment Group defined outside the CDK app.
     *
     * Account and region for the DeploymentGroup are taken from the application.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromEcsDeploymentGroupAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codedeploy_EcsDeploymentGroupAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromEcsDeploymentGroupAttributes);
            }
            throw error;
        }
        return new ImportedEcsDeploymentGroup(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            deploymentGroupName: props.deploymentGroupName,
            role: props.role,
            roleConstructId: 'ServiceRole',
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codedeploy_EcsDeploymentGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EcsDeploymentGroup);
            }
            throw error;
        }
        this.role = this._role;
        this.application = props.application || new application_1.EcsApplication(this, 'Application');
        this.alarms = props.alarms || [];
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AWSCodeDeployRoleForECS'));
        this.deploymentConfig = this._bindDeploymentConfig(props.deploymentConfig || deployment_config_1.EcsDeploymentConfig.ALL_AT_ONCE);
        if (cdk.Resource.isOwnedResource(props.service)) {
            const cfnSvc = props.service.node.defaultChild;
            if (cfnSvc.deploymentController === undefined ||
                cfnSvc.deploymentController.type !== ecs.DeploymentControllerType.CODE_DEPLOY) {
                throw new Error('The ECS service associated with the deployment group must use the CODE_DEPLOY deployment controller type');
            }
            if (cfnSvc.taskDefinition !== props.service.taskDefinition.family) {
                throw new Error('The ECS service associated with the deployment group must specify the task definition using the task definition family name only. Otherwise, the task definition cannot be updated in the stack');
            }
        }
        const removeAlarmsFromDeploymentGroup = cdk.FeatureFlags.of(this).isEnabled(cx_api_1.CODEDEPLOY_REMOVE_ALARMS_FROM_DEPLOYMENT_GROUP);
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            serviceRoleArn: this.role.roleArn,
            deploymentGroupName: this.physicalName,
            deploymentConfigName: this.deploymentConfig.deploymentConfigName,
            deploymentStyle: {
                deploymentType: 'BLUE_GREEN',
                deploymentOption: 'WITH_TRAFFIC_CONTROL',
            },
            ecsServices: [{
                    clusterName: props.service.cluster.clusterName,
                    serviceName: props.service.serviceName,
                }],
            blueGreenDeploymentConfiguration: cdk.Lazy.any({
                produce: () => this.renderBlueGreenDeploymentConfiguration(props.blueGreenDeploymentConfig),
            }),
            loadBalancerInfo: cdk.Lazy.any({ produce: () => this.renderLoadBalancerInfo(props.blueGreenDeploymentConfig) }),
            alarmConfiguration: cdk.Lazy.any({
                produce: () => (0, utils_1.renderAlarmConfiguration)(this.alarms, props.ignorePollAlarmsFailure, removeAlarmsFromDeploymentGroup),
            }),
            autoRollbackConfiguration: cdk.Lazy.any({ produce: () => (0, utils_1.renderAutoRollbackConfiguration)(this.alarms, props.autoRollback) }),
        });
        this._setNameAndArn(resource, this.application);
        // If the deployment config is a construct, add a dependency to ensure the deployment config
        // is created before the deployment group is.
        if (constructs_1.Construct.isConstruct(this.deploymentConfig)) {
            this.node.addDependency(this.deploymentConfig);
        }
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_IAlarm(alarm);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAlarm);
            }
            throw error;
        }
        this.alarms.push(alarm);
    }
    renderBlueGreenDeploymentConfiguration(options) {
        return {
            deploymentReadyOption: {
                actionOnTimeout: options.deploymentApprovalWaitTime ? 'STOP_DEPLOYMENT' : 'CONTINUE_DEPLOYMENT',
                waitTimeInMinutes: options.deploymentApprovalWaitTime?.toMinutes() ?? 0,
            },
            terminateBlueInstancesOnDeploymentSuccess: {
                action: 'TERMINATE',
                terminationWaitTimeInMinutes: options.terminationWaitTime?.toMinutes() ?? 0,
            },
        };
    }
    renderLoadBalancerInfo(options) {
        return {
            targetGroupPairInfoList: [
                {
                    targetGroups: [
                        {
                            name: options.blueTargetGroup.targetGroupName,
                        },
                        {
                            name: options.greenTargetGroup.targetGroupName,
                        },
                    ],
                    prodTrafficRoute: {
                        listenerArns: [
                            options.listener.listenerArn,
                        ],
                    },
                    testTrafficRoute: options.testListener ? {
                        listenerArns: [
                            options.testListener.listenerArn,
                        ],
                    } : undefined,
                },
            ],
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
EcsDeploymentGroup[_a] = { fqn: "aws-cdk-lib.aws_codedeploy.EcsDeploymentGroup", version: "2.74.0" };
exports.EcsDeploymentGroup = EcsDeploymentGroup;
class ImportedEcsDeploymentGroup extends base_deployment_group_1.ImportedDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, {
            application: props.application,
            deploymentGroupName: props.deploymentGroupName,
        });
        this.application = props.application;
        this.deploymentConfig = this._bindDeploymentConfig(props.deploymentConfig || deployment_config_1.EcsDeploymentConfig.ALL_AT_ONCE);
    }
}
//# sourceMappingURL=data:application/json;base64,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