"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDeploymentGroup = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../../aws-iam");
const cdk = require("../../../core");
const cx_api_1 = require("../../../cx-api");
const constructs_1 = require("constructs");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
const codedeploy_generated_1 = require("../codedeploy.generated");
const base_deployment_group_1 = require("../private/base-deployment-group");
const utils_1 = require("../private/utils");
/**
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class LambdaDeploymentGroup extends base_deployment_group_1.DeploymentGroupBase {
    /**
     * Import an Lambda Deployment Group defined either outside the CDK app, or in a different AWS region.
     *
     * Account and region for the DeploymentGroup are taken from the application.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromLambdaDeploymentGroupAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codedeploy_LambdaDeploymentGroupAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromLambdaDeploymentGroupAttributes);
            }
            throw error;
        }
        return new ImportedLambdaDeploymentGroup(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            deploymentGroupName: props.deploymentGroupName,
            role: props.role,
            roleConstructId: 'ServiceRole',
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codedeploy_LambdaDeploymentGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LambdaDeploymentGroup);
            }
            throw error;
        }
        this.role = this._role;
        this.application = props.application || new application_1.LambdaApplication(this, 'Application');
        this.alarms = props.alarms || [];
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRoleForLambdaLimited'));
        this.deploymentConfig = this._bindDeploymentConfig(props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES);
        const removeAlarmsFromDeploymentGroup = cdk.FeatureFlags.of(this).isEnabled(cx_api_1.CODEDEPLOY_REMOVE_ALARMS_FROM_DEPLOYMENT_GROUP);
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            serviceRoleArn: this.role.roleArn,
            deploymentGroupName: this.physicalName,
            deploymentConfigName: this.deploymentConfig.deploymentConfigName,
            deploymentStyle: {
                deploymentType: 'BLUE_GREEN',
                deploymentOption: 'WITH_TRAFFIC_CONTROL',
            },
            alarmConfiguration: cdk.Lazy.any({
                produce: () => (0, utils_1.renderAlarmConfiguration)(this.alarms, props.ignorePollAlarmsFailure, removeAlarmsFromDeploymentGroup),
            }),
            autoRollbackConfiguration: cdk.Lazy.any({ produce: () => (0, utils_1.renderAutoRollbackConfiguration)(this.alarms, props.autoRollback) }),
        });
        this._setNameAndArn(resource, this.application);
        if (props.preHook) {
            this.addPreHook(props.preHook);
        }
        if (props.postHook) {
            this.addPostHook(props.postHook);
        }
        props.alias.node.defaultChild.cfnOptions.updatePolicy = {
            codeDeployLambdaAliasUpdate: {
                applicationName: this.application.applicationName,
                deploymentGroupName: resource.ref,
                beforeAllowTrafficHook: cdk.Lazy.string({ produce: () => this.preHook && this.preHook.functionName }),
                afterAllowTrafficHook: cdk.Lazy.string({ produce: () => this.postHook && this.postHook.functionName }),
            },
        };
        // If the deployment config is a construct, add a dependency to ensure the deployment config
        // is created before the deployment group is.
        if (this.deploymentConfig instanceof constructs_1.Construct) {
            this.node.addDependency(this.deploymentConfig);
        }
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_IAlarm(alarm);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAlarm);
            }
            throw error;
        }
        this.alarms.push(alarm);
    }
    /**
     * Associate a function to run before deployment begins.
     * @param preHook function to run before deployment beings
     * @throws an error if a pre-hook function is already configured
     */
    addPreHook(preHook) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_IFunction(preHook);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPreHook);
            }
            throw error;
        }
        if (this.preHook !== undefined) {
            throw new Error('A pre-hook function is already defined for this deployment group');
        }
        this.preHook = preHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.preHook);
        this.preHook.grantInvoke(this.role);
    }
    /**
     * Associate a function to run after deployment completes.
     * @param postHook function to run after deployment completes
     * @throws an error if a post-hook function is already configured
     */
    addPostHook(postHook) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_IFunction(postHook);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPostHook);
            }
            throw error;
        }
        if (this.postHook !== undefined) {
            throw new Error('A post-hook function is already defined for this deployment group');
        }
        this.postHook = postHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.postHook);
        this.postHook.grantInvoke(this.role);
    }
    /**
     * Grant a principal permission to codedeploy:PutLifecycleEventHookExecutionStatus
     * on this deployment group resource.
     * @param grantee to grant permission to
     */
    grantPutLifecycleEventHookExecutionStatus(grantee) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantPutLifecycleEventHookExecutionStatus);
            }
            throw error;
        }
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.deploymentGroupArn],
            actions: ['codedeploy:PutLifecycleEventHookExecutionStatus'],
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
LambdaDeploymentGroup[_a] = { fqn: "aws-cdk-lib.aws_codedeploy.LambdaDeploymentGroup", version: "2.74.0" };
exports.LambdaDeploymentGroup = LambdaDeploymentGroup;
class ImportedLambdaDeploymentGroup extends base_deployment_group_1.ImportedDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, {
            application: props.application,
            deploymentGroupName: props.deploymentGroupName,
        });
        this.application = props.application;
        this.deploymentConfig = this._bindDeploymentConfig(props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES);
    }
}
//# sourceMappingURL=data:application/json;base64,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