"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerDeploymentGroup = exports.InstanceTagSet = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../../aws-ec2");
const iam = require("../../../aws-iam");
const s3 = require("../../../aws-s3");
const cdk = require("../../../core");
const cx_api_1 = require("../../../cx-api");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
const load_balancer_1 = require("./load-balancer");
const codedeploy_generated_1 = require("../codedeploy.generated");
const base_deployment_group_1 = require("../private/base-deployment-group");
const utils_1 = require("../private/utils");
class ImportedServerDeploymentGroup extends base_deployment_group_1.ImportedDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, {
            application: props.application,
            deploymentGroupName: props.deploymentGroupName,
        });
        this.role = undefined;
        this.autoScalingGroups = undefined;
        this.application = props.application;
        this.deploymentConfig = this._bindDeploymentConfig(props.deploymentConfig || deployment_config_1.ServerDeploymentConfig.ONE_AT_A_TIME);
    }
}
/**
 * Represents a set of instance tag groups.
 * An instance will match a set if it matches all of the groups in the set -
 * in other words, sets follow 'and' semantics.
 * You can have a maximum of 3 tag groups inside a set.
 */
class InstanceTagSet {
    constructor(...instanceTagGroups) {
        if (instanceTagGroups.length > 3) {
            throw new Error('An instance tag set can have a maximum of 3 instance tag groups, ' +
                `but ${instanceTagGroups.length} were provided`);
        }
        this._instanceTagGroups = instanceTagGroups;
    }
    get instanceTagGroups() {
        return this._instanceTagGroups.slice();
    }
}
_a = JSII_RTTI_SYMBOL_1;
InstanceTagSet[_a] = { fqn: "aws-cdk-lib.aws_codedeploy.InstanceTagSet", version: "2.74.0" };
exports.InstanceTagSet = InstanceTagSet;
/**
 * A CodeDeploy Deployment Group that deploys to EC2/on-premise instances.
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class ServerDeploymentGroup extends base_deployment_group_1.DeploymentGroupBase {
    /**
     * Import an EC2/on-premise Deployment Group defined either outside the CDK app,
     * or in a different region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromServerDeploymentGroupAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codedeploy_ServerDeploymentGroupAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromServerDeploymentGroupAttributes);
            }
            throw error;
        }
        return new ImportedServerDeploymentGroup(scope, id, attrs);
    }
    constructor(scope, id, props = {}) {
        super(scope, id, {
            deploymentGroupName: props.deploymentGroupName,
            role: props.role,
            roleConstructId: 'Role',
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codedeploy_ServerDeploymentGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ServerDeploymentGroup);
            }
            throw error;
        }
        this.role = this._role;
        this.application = props.application || new application_1.ServerApplication(this, 'Application', {
            applicationName: props.deploymentGroupName === cdk.PhysicalName.GENERATE_IF_NEEDED ? cdk.PhysicalName.GENERATE_IF_NEEDED : undefined,
        });
        this.deploymentConfig = this._bindDeploymentConfig(props.deploymentConfig || deployment_config_1.ServerDeploymentConfig.ONE_AT_A_TIME);
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRole'));
        this._autoScalingGroups = props.autoScalingGroups || [];
        this.installAgent = props.installAgent ?? true;
        this.codeDeployBucket = s3.Bucket.fromBucketName(this, 'Bucket', `aws-codedeploy-${cdk.Stack.of(this).region}`);
        for (const asg of this._autoScalingGroups) {
            this.addCodeDeployAgentInstallUserData(asg);
        }
        this.alarms = props.alarms || [];
        const removeAlarmsFromDeploymentGroup = cdk.FeatureFlags.of(this).isEnabled(cx_api_1.CODEDEPLOY_REMOVE_ALARMS_FROM_DEPLOYMENT_GROUP);
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            deploymentGroupName: this.physicalName,
            serviceRoleArn: this.role.roleArn,
            deploymentConfigName: props.deploymentConfig &&
                props.deploymentConfig.deploymentConfigName,
            autoScalingGroups: cdk.Lazy.list({ produce: () => this._autoScalingGroups.map(asg => asg.autoScalingGroupName) }, { omitEmpty: true }),
            loadBalancerInfo: this.loadBalancerInfo(props.loadBalancer),
            deploymentStyle: props.loadBalancer === undefined
                ? undefined
                : {
                    deploymentOption: 'WITH_TRAFFIC_CONTROL',
                },
            ec2TagSet: this.ec2TagSet(props.ec2InstanceTags),
            onPremisesTagSet: this.onPremiseTagSet(props.onPremiseInstanceTags),
            alarmConfiguration: cdk.Lazy.any({
                produce: () => (0, utils_1.renderAlarmConfiguration)(this.alarms, props.ignorePollAlarmsFailure, removeAlarmsFromDeploymentGroup),
            }),
            autoRollbackConfiguration: cdk.Lazy.any({ produce: () => (0, utils_1.renderAutoRollbackConfiguration)(this.alarms, props.autoRollback) }),
        });
        this._setNameAndArn(resource, this.application);
    }
    /**
     * Adds an additional auto-scaling group to this Deployment Group.
     *
     * @param asg the auto-scaling group to add to this Deployment Group.
     * [disable-awslint:ref-via-interface] is needed in order to install the code
     * deploy agent by updating the ASGs user data.
     */
    addAutoScalingGroup(asg) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_autoscaling_AutoScalingGroup(asg);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAutoScalingGroup);
            }
            throw error;
        }
        this._autoScalingGroups.push(asg);
        this.addCodeDeployAgentInstallUserData(asg);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_IAlarm(alarm);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAlarm);
            }
            throw error;
        }
        this.alarms.push(alarm);
    }
    get autoScalingGroups() {
        return this._autoScalingGroups.slice();
    }
    addCodeDeployAgentInstallUserData(asg) {
        if (!this.installAgent) {
            return;
        }
        this.codeDeployBucket.grantRead(asg, 'latest/*');
        switch (asg.osType) {
            case ec2.OperatingSystemType.LINUX:
                asg.addUserData('set +e', // make sure we don't exit on the `which` failing
                'PKG_CMD=`which yum 2>/dev/null`', 'set -e', // continue with failing on error
                'if [ -z "$PKG_CMD" ]; then', 'PKG_CMD=apt-get', 'else', 'PKG_CMD=yum', 'fi', '$PKG_CMD update -y', 'set +e', // make sure we don't exit on the next command failing (we check its exit code below)
                '$PKG_CMD install -y ruby2.0', 'RUBY2_INSTALL=$?', 'set -e', // continue with failing on error
                'if [ $RUBY2_INSTALL -ne 0 ]; then', '$PKG_CMD install -y ruby', 'fi', 'AWS_CLI_PACKAGE_NAME=awscli', 'if [ "$PKG_CMD" = "yum" ]; then', 'AWS_CLI_PACKAGE_NAME=aws-cli', 'fi', '$PKG_CMD install -y $AWS_CLI_PACKAGE_NAME', 'TMP_DIR=`mktemp -d`', 'cd $TMP_DIR', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/install . --region ${cdk.Stack.of(this).region}`, 'chmod +x ./install', './install auto', 'rm -fr $TMP_DIR');
                break;
            case ec2.OperatingSystemType.WINDOWS:
                asg.addUserData('Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/codedeploy-agent.msi $TEMPDIR\\codedeploy-agent.msi`, 'cd $TEMPDIR', '.\\codedeploy-agent.msi /quiet /l c:\\temp\\host-agent-install-log.txt');
                break;
        }
    }
    loadBalancerInfo(loadBalancer) {
        if (!loadBalancer) {
            return undefined;
        }
        switch (loadBalancer.generation) {
            case load_balancer_1.LoadBalancerGeneration.FIRST:
                return {
                    elbInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
            case load_balancer_1.LoadBalancerGeneration.SECOND:
                return {
                    targetGroupInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
        }
    }
    ec2TagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            ec2TagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    ec2TagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    onPremiseTagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            onPremisesTagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    onPremisesTagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    tagGroup2TagsArray(tagGroup) {
        const tagsInGroup = new Array();
        for (const tagKey in tagGroup) {
            if (tagGroup.hasOwnProperty(tagKey)) {
                const tagValues = tagGroup[tagKey];
                if (tagKey.length > 0) {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                key: tagKey,
                                value: tagValue,
                                type: 'KEY_AND_VALUE',
                            });
                        }
                    }
                    else {
                        tagsInGroup.push({
                            key: tagKey,
                            type: 'KEY_ONLY',
                        });
                    }
                }
                else {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                value: tagValue,
                                type: 'VALUE_ONLY',
                            });
                        }
                    }
                    else {
                        throw new Error('Cannot specify both an empty key and no values for an instance tag filter');
                    }
                }
            }
        }
        return tagsInGroup;
    }
}
_b = JSII_RTTI_SYMBOL_1;
ServerDeploymentGroup[_b] = { fqn: "aws-cdk-lib.aws_codedeploy.ServerDeploymentGroup", version: "2.74.0" };
exports.ServerDeploymentGroup = ServerDeploymentGroup;
//# sourceMappingURL=data:application/json;base64,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