"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProfilingGroup = exports.ComputePlatform = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("../../aws-iam");
const core_1 = require("../../core");
const codeguruprofiler_generated_1 = require("./codeguruprofiler.generated");
/**
 * The compute platform of the profiling group.
 */
var ComputePlatform;
(function (ComputePlatform) {
    /**
     * Use AWS_LAMBDA if your application runs on AWS Lambda.
     */
    ComputePlatform["AWS_LAMBDA"] = "AWSLambda";
    /**
     * Use Default if your application runs on a compute platform that is not AWS Lambda,
     * such an Amazon EC2 instance, an on-premises server, or a different platform.
     */
    ComputePlatform["DEFAULT"] = "Default";
})(ComputePlatform = exports.ComputePlatform || (exports.ComputePlatform = {}));
class ProfilingGroupBase extends core_1.Resource {
    /**
     * Grant access to publish profiling information to the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:ConfigureAgent
     *  - codeguru-profiler:PostAgentProfile
     *
     * @param grantee Principal to grant publish rights to
     */
    grantPublish(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:ConfigureAgent', 'codeguru-profiler:PostAgentProfile'],
            resourceArns: [this.profilingGroupArn],
        });
    }
    /**
     * Grant access to read profiling information from the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:GetProfile
     *  - codeguru-profiler:DescribeProfilingGroup
     *
     * @param grantee Principal to grant read rights to
     */
    grantRead(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:GetProfile', 'codeguru-profiler:DescribeProfilingGroup'],
            resourceArns: [this.profilingGroupArn],
        });
    }
}
/**
 * A new Profiling Group.
 */
class ProfilingGroup extends ProfilingGroupBase {
    /**
     * Import an existing Profiling Group provided a Profiling Group Name.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupName Profiling Group Name
     */
    static fromProfilingGroupName(scope, id, profilingGroupName) {
        const stack = core_1.Stack.of(scope);
        return this.fromProfilingGroupArn(scope, id, stack.formatArn({
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: profilingGroupName,
        }));
    }
    /**
     * Import an existing Profiling Group provided an ARN.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupArn Profiling Group ARN
     */
    static fromProfilingGroupArn(scope, id, profilingGroupArn) {
        class Import extends ProfilingGroupBase {
            constructor() {
                super(...arguments);
                this.profilingGroupName = core_1.Stack.of(scope).splitArn(profilingGroupArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
                this.profilingGroupArn = profilingGroupArn;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: profilingGroupArn,
        });
    }
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.profilingGroupName ?? core_1.Lazy.string({ produce: () => this.generateUniqueId() }),
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codeguruprofiler_ProfilingGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ProfilingGroup);
            }
            throw error;
        }
        const profilingGroup = new codeguruprofiler_generated_1.CfnProfilingGroup(this, 'ProfilingGroup', {
            profilingGroupName: this.physicalName,
            computePlatform: props.computePlatform,
        });
        this.profilingGroupName = this.getResourceNameAttribute(profilingGroup.ref);
        this.profilingGroupArn = this.getResourceArnAttribute(profilingGroup.attrArn, {
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: this.physicalName,
        });
    }
    generateUniqueId() {
        const name = core_1.Names.uniqueId(this);
        if (name.length > 240) {
            return name.substring(0, 120) + name.substring(name.length - 120);
        }
        return name;
    }
}
_a = JSII_RTTI_SYMBOL_1;
ProfilingGroup[_a] = { fqn: "aws-cdk-lib.aws_codeguruprofiler.ProfilingGroup", version: "2.74.0" };
exports.ProfilingGroup = ProfilingGroup;
//# sourceMappingURL=data:application/json;base64,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