"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseCapabilities = exports.SingletonPolicy = void 0;
const iam = require("../../../../aws-iam");
const cdk = require("../../../../core");
const constructs_1 = require("constructs");
/**
 * Manages a bunch of singleton-y statements on the policy of an IAM Role.
 * Dedicated methods can be used to add specific permissions to the role policy
 * using as few statements as possible (adding resources to existing compatible
 * statements instead of adding new statements whenever possible).
 *
 * Statements created outside of this class are not considered when adding new
 * permissions.
 */
class SingletonPolicy extends constructs_1.Construct {
    /**
     * Obtain a SingletonPolicy for a given role.
     * @param role the Role this policy is bound to.
     * @returns the SingletonPolicy for this role.
     */
    static forRole(role) {
        const found = role.node.tryFindChild(SingletonPolicy.UUID);
        return found || new SingletonPolicy(role);
    }
    constructor(role) {
        super(role, SingletonPolicy.UUID);
        this.role = role;
        this.statements = {};
        this.grantPrincipal = role;
    }
    grantExecuteChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStacks',
                'cloudformation:DescribeChangeSet',
                'cloudformation:ExecuteChangeSet',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateReplaceChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:CreateChangeSet',
                'cloudformation:DeleteChangeSet',
                'cloudformation:DescribeChangeSet',
                'cloudformation:DescribeStacks',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStack(props) {
        const actions = [
            'cloudformation:DescribeStack*',
            'cloudformation:CreateStack',
            'cloudformation:UpdateStack',
            'cloudformation:GetTemplate*',
            'cloudformation:ValidateTemplate',
            'cloudformation:GetStackPolicy',
            'cloudformation:SetStackPolicy',
        ];
        if (props.replaceOnFailure) {
            actions.push('cloudformation:DeleteStack');
        }
        this.statementFor({ actions }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStackSet(props) {
        const actions = [
            'cloudformation:CreateStackSet',
            'cloudformation:UpdateStackSet',
            'cloudformation:DescribeStackSet',
            'cloudformation:DescribeStackSetOperation',
            'cloudformation:ListStackInstances',
            'cloudformation:CreateStackInstances',
        ];
        this.statementFor({ actions }).addResources(this.stackSetArnFromProps(props));
    }
    grantDeleteStack(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStack*',
                'cloudformation:DeleteStack',
            ],
        }).addResources(this.stackArnFromProps(props));
    }
    grantPassRole(role) {
        this.statementFor({ actions: ['iam:PassRole'] }).addResources(typeof role === 'string' ? role : role.roleArn);
    }
    statementFor(template) {
        const key = keyFor(template);
        if (!(key in this.statements)) {
            this.statements[key] = new iam.PolicyStatement({ actions: template.actions });
            if (template.conditions) {
                this.statements[key].addConditions(template.conditions);
            }
            this.role.addToPolicy(this.statements[key]);
        }
        return this.statements[key];
        function keyFor(props) {
            const actions = `${props.actions.sort().join('\x1F')}`;
            const conditions = formatConditions(props.conditions);
            return `${actions}\x1D${conditions}`;
            function formatConditions(cond) {
                if (cond == null) {
                    return '';
                }
                let result = '';
                for (const op of Object.keys(cond).sort()) {
                    result += `${op}\x1E`;
                    const condition = cond[op];
                    for (const attribute of Object.keys(condition).sort()) {
                        const value = condition[attribute];
                        result += `${value}\x1F`;
                    }
                }
                return result;
            }
        }
    }
    stackArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stack',
            resourceName: `${props.stackName}/*`,
        });
    }
    stackSetArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stackset',
            resourceName: `${props.stackSetName}:*`,
        });
    }
}
SingletonPolicy.UUID = '8389e75f-0810-4838-bf64-d6f85a95cf83';
exports.SingletonPolicy = SingletonPolicy;
function parseCapabilities(capabilities) {
    if (capabilities === undefined) {
        return undefined;
    }
    else if (capabilities.length === 1) {
        const capability = capabilities.toString();
        return (capability === '') ? undefined : capability;
    }
    else if (capabilities.length > 1) {
        return capabilities.join(',');
    }
    return undefined;
}
exports.parseCapabilities = parseCapabilities;
//# sourceMappingURL=data:application/json;base64,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