"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationDeployStackSetAction = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codepipeline = require("../../../aws-codepipeline");
const singleton_policy_1 = require("./private/singleton-policy");
const stackset_types_1 = require("./stackset-types");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * CodePipeline action to deploy a stackset.
 *
 * CodePipeline offers the ability to perform AWS CloudFormation StackSets
 * operations as part of your CI/CD process. You use a stack set to create
 * stacks in AWS accounts across AWS Regions by using a single AWS
 * CloudFormation template. All the resources included in each stack are defined
 * by the stack set’s AWS CloudFormation template. When you create the stack
 * set, you specify the template to use, as well as any parameters and
 * capabilities that the template requires.
 *
 * For more information about concepts for AWS CloudFormation StackSets, see
 * [StackSets
 * concepts](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html)
 * in the AWS CloudFormation User Guide.
 *
 * If you use this action to make an update that includes adding stack
 * instances, the new instances are deployed first and the update is completed
 * last. The new instances first receive the old version, and then the update is
 * applied to all instances.
 *
 * As a best practice, you should construct your pipeline so that the stack set
 * is created and initially deploys to a subset or a single instance. After you
 * test your deployment and view the generated stack set, then add the
 * CloudFormationStackInstances action so that the remaining instances are
 * created and updated.
 */
class CloudFormationDeployStackSetAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            region: props.stackSetRegion,
            provider: 'CloudFormationStackSet',
            category: codepipeline.ActionCategory.DEPLOY,
            artifactBounds: {
                minInputs: 1,
                maxInputs: 3,
                minOutputs: 0,
                maxOutputs: 0,
            },
            inputs: [
                ...props.template._artifactsReferenced ?? [],
                ...props.parameters?._artifactsReferenced ?? [],
                ...props.stackInstances?._artifactsReferenced ?? [],
            ],
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codepipeline_actions_CloudFormationDeployStackSetActionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudFormationDeployStackSetAction);
            }
            throw error;
        }
        this.props = props;
        this.deploymentModel = props.deploymentModel ?? stackset_types_1.StackSetDeploymentModel.selfManaged();
        (0, common_1.validatePercentage)('failureTolerancePercentage', props.failureTolerancePercentage);
        (0, common_1.validatePercentage)('maxAccountConcurrencyPercentage', props.maxAccountConcurrencyPercentage);
    }
    bound(scope, _stage, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codepipeline_IStage(_stage);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codepipeline_ActionBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bound);
            }
            throw error;
        }
        const singletonPolicy = singleton_policy_1.SingletonPolicy.forRole(options.role);
        singletonPolicy.grantCreateUpdateStackSet(this.props);
        const instancesResult = this.props.stackInstances?._bind(scope);
        const permissionModelBind = this.deploymentModel?._bind(scope);
        for (const role of permissionModelBind?.passedRoles ?? []) {
            singletonPolicy.grantPassRole(role);
        }
        if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(singletonPolicy);
        }
        return {
            configuration: {
                StackSetName: this.props.stackSetName,
                Description: this.props.description,
                TemplatePath: this.props.template._render(),
                Parameters: this.props.parameters?._render(),
                Capabilities: (0, singleton_policy_1.parseCapabilities)(this.props.cfnCapabilities),
                FailureTolerancePercentage: this.props.failureTolerancePercentage,
                MaxConcurrentPercentage: this.props.maxAccountConcurrencyPercentage,
                ...instancesResult?.stackSetConfiguration,
                ...permissionModelBind?.stackSetConfiguration,
            },
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
CloudFormationDeployStackSetAction[_a] = { fqn: "aws-cdk-lib.aws_codepipeline_actions.CloudFormationDeployStackSetAction", version: "2.74.0" };
exports.CloudFormationDeployStackSetAction = CloudFormationDeployStackSetAction;
//# sourceMappingURL=data:application/json;base64,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