"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3SourceAction = exports.S3Trigger = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codepipeline = require("../../../aws-codepipeline");
const targets = require("../../../aws-events-targets");
const core_1 = require("../../../core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the S3 Action detect changes.
 * This is the type of the `S3SourceAction.trigger` property.
 */
var S3Trigger;
(function (S3Trigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    S3Trigger["NONE"] = "None";
    /**
     * CodePipeline will poll S3 to detect changes.
     * This is the default method of detecting changes.
     */
    S3Trigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * Note that the Bucket that the Action uses needs to be part of a CloudTrail Trail
     * for the events to be delivered.
     */
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 */
class S3SourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: (0, common_1.sourceArtifactBounds)(),
            outputs: [props.output],
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codepipeline_actions_S3SourceActionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, S3SourceAction);
            }
            throw error;
        }
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    bound(_scope, stage, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codepipeline_IStage(stage);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codepipeline_ActionBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bound);
            }
            throw error;
        }
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = this.generateEventId(stage);
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey],
                crossStackScope: stage.pipeline,
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role, this.props.bucketKey);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        let ret;
        const baseId = core_1.Names.nodeUniqueId(stage.pipeline.node) + 'SourceEventRule';
        if (core_1.Token.isUnresolved(this.props.bucketKey)) {
            // If bucketKey is a Token, don't include it in the ID.
            // Instead, use numbers to differentiate if multiple actions observe the same bucket
            let candidate = baseId;
            let counter = 0;
            while (this.props.bucket.node.tryFindChild(candidate) !== undefined) {
                counter += 1;
                candidate = baseId + counter;
            }
            ret = candidate;
        }
        else {
            // we can't use Tokens in construct IDs,
            // however, if bucketKey is not a Token,
            // we want it to differentiate between multiple actions
            // observing the same Bucket with different keys
            ret = baseId + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(ret)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
        }
        return ret;
    }
}
_a = JSII_RTTI_SYMBOL_1;
S3SourceAction[_a] = { fqn: "aws-cdk-lib.aws_codepipeline_actions.S3SourceAction", version: "2.74.0" };
exports.S3SourceAction = S3SourceAction;
//# sourceMappingURL=data:application/json;base64,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