"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossRegionSupportStack = exports.CrossRegionSupportConstruct = void 0;
const kms = require("../../../aws-kms");
const s3 = require("../../../aws-s3");
const cdk = require("../../../core");
const constructs_1 = require("constructs");
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends constructs_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        const createKmsKey = props.createKmsKey ?? true;
        let encryptionAlias;
        if (createKmsKey) {
            const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
                removalPolicy: cdk.RemovalPolicy.DESTROY,
                enableKeyRotation: props.enableKeyRotation,
            });
            encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
                targetKey: encryptionKey,
                aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
        }
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryption: encryptionAlias ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
            encryptionKey: encryptionAlias,
            enforceSSL: true,
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
            synthesizer: props.synthesizer,
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default', {
            createKmsKey: props.createKmsKey,
            enableKeyRotation: props.enableKeyRotation,
        });
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,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