"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const custom_resources_1 = require("../../custom-resources");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     * The format is 'resource-server-identifier/scope'.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
    /**
     * Adds a custom scope that's tied to a resource server in your stack
     */
    static resourceServer(server, scope) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cognito_IUserPoolResourceServer(server);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cognito_ResourceServerScope(scope);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.resourceServer);
            }
            throw error;
        }
        return new OAuthScope(`${server.userPoolResourceServerId}/${scope.scopeName}`);
    }
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
}
_a = JSII_RTTI_SYMBOL_1;
OAuthScope[_a] = { fqn: "aws-cdk-lib.aws_cognito.OAuthScope", version: "2.74.0" };
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens,
 * such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
exports.OAuthScope = OAuthScope;
/**
 * Identity providers supported by the UserPoolClient
 */
class UserPoolClientIdentityProvider {
    /**
     * Specify a provider not yet supported by the CDK.
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
    constructor(name) {
        this.name = name;
    }
}
_b = JSII_RTTI_SYMBOL_1;
UserPoolClientIdentityProvider[_b] = { fqn: "aws-cdk-lib.aws_cognito.UserPoolClientIdentityProvider", version: "2.74.0" };
/**
 * Allow users to sign in using 'Sign In With Apple'.
 * A `UserPoolIdentityProviderApple` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.APPLE = new UserPoolClientIdentityProvider('SignInWithApple');
/**
 * Allow users to sign in using 'Facebook Login'.
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Google Login'.
 * A `UserPoolIdentityProviderGoogle` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.GOOGLE = new UserPoolClientIdentityProvider('Google');
/**
 * Allow users to sign in using 'Login With Amazon'.
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
/**
 * Define a UserPool App Client
 */
class UserPoolClient extends core_1.Resource {
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
            get userPoolClientSecret() {
                throw new Error('UserPool Client Secret is not available for imported Clients');
            }
        }
        return new Import(scope, id);
    }
    /*
     * Note to implementers: Two CloudFormation return values Name and ClientSecret are part of the spec.
     * However, they have been explicity not implemented here. They are not documented in CloudFormation, and
     * CloudFormation returns the following the string when these two attributes are 'GetAtt' - "attribute not supported
     * at this time, please use the CLI or Console to retrieve this value".
     * Awaiting updates from CloudFormation.
     */
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cognito_UserPoolClientProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, UserPoolClient);
            }
            throw error;
        }
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = props.oAuth?.flows ?? {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = props.oAuth?.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        this._generateSecret = props.generateSecret;
        this.userPool = props.userPool;
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 && !props.disableOAuth ? callbackUrls : undefined,
            logoutUrLs: props.oAuth?.logoutUrls,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
            readAttributes: props.readAttributes?.attributes(),
            writeAttributes: props.writeAttributes?.attributes(),
            enableTokenRevocation: props.enableTokenRevocation,
        });
        this.configureAuthSessionValidity(resource, props);
        this.configureTokenValidity(resource, props);
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization,
     * throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    get userPoolClientSecret() {
        if (!this._generateSecret) {
            throw new Error('userPoolClientSecret is available only if generateSecret is set to true.');
        }
        // Create the Custom Resource that assists in resolving the User Pool Client secret
        // just once, no matter how many times this method is called
        if (!this._userPoolClientSecret) {
            this._userPoolClientSecret = core_1.SecretValue.resourceAttribute(new custom_resources_1.AwsCustomResource(this, 'DescribeCognitoUserPoolClient', {
                resourceType: 'Custom::DescribeCognitoUserPoolClient',
                onUpdate: {
                    region: core_1.Stack.of(this).region,
                    service: 'CognitoIdentityServiceProvider',
                    action: 'describeUserPoolClient',
                    parameters: {
                        UserPoolId: this.userPool.userPoolId,
                        ClientId: this.userPoolClientId,
                    },
                    physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.userPoolClientId),
                },
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.userPool.userPoolArn],
                }),
                // APIs are available in 2.1055.0
                installLatestAwsSdk: false,
            }).getResponseField('UserPoolClient.ClientSecret'));
        }
        return this._userPoolClientSecret;
    }
    configureAuthFlows(props) {
        if (!props.authFlows || Object.keys(props.authFlows).length === 0)
            return undefined;
        const authFlows = [];
        if (props.authFlows.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if (props.authFlows.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        // refreshToken should always be allowed if authFlows are present
        authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        const scopes = oAuth?.scopes ?? [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
    configureAuthSessionValidity(resource, props) {
        this.validateDuration('authSessionValidity', core_1.Duration.minutes(3), core_1.Duration.minutes(15), props.authSessionValidity);
        resource.authSessionValidity = props.authSessionValidity ? props.authSessionValidity.toMinutes() : undefined;
    }
    configureTokenValidity(resource, props) {
        this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.idTokenValidity);
        this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.accessTokenValidity);
        this.validateDuration('refreshTokenValidity', core_1.Duration.minutes(60), core_1.Duration.days(10 * 365), props.refreshTokenValidity);
        if (props.refreshTokenValidity) {
            this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.idTokenValidity);
            this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.accessTokenValidity);
        }
        if (props.accessTokenValidity || props.idTokenValidity || props.refreshTokenValidity) {
            resource.tokenValidityUnits = {
                idToken: props.idTokenValidity ? 'minutes' : undefined,
                accessToken: props.accessTokenValidity ? 'minutes' : undefined,
                refreshToken: props.refreshTokenValidity ? 'minutes' : undefined,
            };
        }
        ;
        resource.idTokenValidity = props.idTokenValidity ? props.idTokenValidity.toMinutes() : undefined;
        resource.refreshTokenValidity = props.refreshTokenValidity ? props.refreshTokenValidity.toMinutes() : undefined;
        resource.accessTokenValidity = props.accessTokenValidity ? props.accessTokenValidity.toMinutes() : undefined;
    }
    validateDuration(name, min, max, value) {
        if (value === undefined) {
            return;
        }
        if (value.toMilliseconds() < min.toMilliseconds() || value.toMilliseconds() > max.toMilliseconds()) {
            throw new Error(`${name}: Must be a duration between ${min.toHumanString()} and ${max.toHumanString()} (inclusive); received ${value.toHumanString()}.`);
        }
    }
}
_c = JSII_RTTI_SYMBOL_1;
UserPoolClient[_c] = { fqn: "aws-cdk-lib.aws_cognito.UserPoolClient", version: "2.74.0" };
exports.UserPoolClient = UserPoolClient;
//# sourceMappingURL=data:application/json;base64,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