"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPool = exports.AdvancedSecurityMode = exports.AccountRecovery = exports.Mfa = exports.VerificationEmailStyle = exports.UserPoolOperation = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("../../aws-iam");
const core_1 = require("../../core");
const punycode_1 = require("punycode/");
const cognito_generated_1 = require("./cognito.generated");
const attr_names_1 = require("./private/attr-names");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
const user_pool_resource_server_1 = require("./user-pool-resource-server");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    /** A custom user pool operation */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
    constructor(operationName) {
        this.operationName = operationName;
    }
}
_a = JSII_RTTI_SYMBOL_1;
UserPoolOperation[_a] = { fqn: "aws-cdk-lib.aws_cognito.UserPoolOperation", version: "2.74.0" };
/**
 * Creates a challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * Amazon Cognito invokes this trigger to send email notifications to users.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-email-sender.html
 */
UserPoolOperation.CUSTOM_EMAIL_SENDER = new UserPoolOperation('customEmailSender');
/**
 * Amazon Cognito invokes this trigger to send email notifications to users.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-sms-sender.html
 */
UserPoolOperation.CUSTOM_SMS_SENDER = new UserPoolOperation('customSmsSender');
exports.UserPoolOperation = UserPoolOperation;
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
/**
 * How will a user be able to recover their account?
 *
 * When a user forgets their password, they can have a code sent to their verified email or verified phone to recover their account.
 * You can choose the preferred way to send codes below.
 * We recommend not allowing phone to be used for both password resets and multi-factor authentication (MFA).
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html
 */
var AccountRecovery;
(function (AccountRecovery) {
    /**
     * Email if available, otherwise phone, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["EMAIL_AND_PHONE_WITHOUT_MFA"] = 0] = "EMAIL_AND_PHONE_WITHOUT_MFA";
    /**
     * Phone if available, otherwise email, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_WITHOUT_MFA_AND_EMAIL"] = 1] = "PHONE_WITHOUT_MFA_AND_EMAIL";
    /**
     * Email only
     */
    AccountRecovery[AccountRecovery["EMAIL_ONLY"] = 2] = "EMAIL_ONLY";
    /**
     * Phone only, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_ONLY_WITHOUT_MFA"] = 3] = "PHONE_ONLY_WITHOUT_MFA";
    /**
     * (Not Recommended) Phone if available, otherwise email, and do allow a user to reset their password via phone if they are also using it for MFA.
     */
    AccountRecovery[AccountRecovery["PHONE_AND_EMAIL"] = 4] = "PHONE_AND_EMAIL";
    /**
     * None – users will have to contact an administrator to reset their passwords
     */
    AccountRecovery[AccountRecovery["NONE"] = 5] = "NONE";
})(AccountRecovery = exports.AccountRecovery || (exports.AccountRecovery = {}));
/**
 * The different ways in which a user pool's Advanced Security Mode can be configured.
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-userpooladdons.html#cfn-cognito-userpool-userpooladdons-advancedsecuritymode
 */
var AdvancedSecurityMode;
(function (AdvancedSecurityMode) {
    /** Enable advanced security mode */
    AdvancedSecurityMode["ENFORCED"] = "ENFORCED";
    /** gather metrics on detected risks without taking action. Metrics are published to Amazon CloudWatch */
    AdvancedSecurityMode["AUDIT"] = "AUDIT";
    /** Advanced security mode is disabled */
    AdvancedSecurityMode["OFF"] = "OFF";
})(AdvancedSecurityMode = exports.AdvancedSecurityMode || (exports.AdvancedSecurityMode = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.identityProviders = [];
    }
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    addResourceServer(id, options) {
        return new user_pool_resource_server_1.UserPoolResourceServer(this, id, {
            userPool: this,
            ...options,
        });
    }
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
    grant(grantee, ...actions) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.userPoolArn],
            scope: this,
        });
    }
}
/**
 * Define a Cognito User Pool
 */
class UserPool extends UserPoolBase {
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        let userPoolArn = core_1.Stack.of(scope).formatArn({
            service: 'cognito-idp',
            resource: 'userpool',
            resourceName: userPoolId,
        });
        return UserPool.fromUserPoolArn(scope, id, userPoolArn);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        const arnParts = core_1.Stack.of(scope).splitArn(userPoolArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!arnParts.resourceName) {
            throw new Error('invalid user pool ARN');
        }
        const userPoolId = arnParts.resourceName;
        class ImportedUserPool extends UserPoolBase {
            constructor() {
                super(scope, id, {
                    account: arnParts.account,
                    region: arnParts.region,
                });
                this.userPoolArn = userPoolArn;
                this.userPoolId = userPoolId;
            }
        }
        return new ImportedUserPool();
    }
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.triggers = {};
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cognito_UserPoolProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, UserPool);
            }
            throw error;
        }
        const signIn = this.signInConfiguration(props);
        if (props.customSenderKmsKey) {
            const kmsKey = props.customSenderKmsKey;
            this.triggers.kmsKeyId = kmsKey.keyArn;
        }
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                let trigger;
                switch (t) {
                    case 'customSmsSender':
                    case 'customEmailSender':
                        if (!this.triggers.kmsKeyId) {
                            throw new Error('you must specify a KMS key if you are using customSmsSender or customEmailSender.');
                        }
                        trigger = props.lambdaTriggers[t];
                        const version = 'V1_0';
                        if (trigger !== undefined) {
                            this.addLambdaPermission(trigger, t);
                            this.triggers[t] = {
                                lambdaArn: trigger.functionArn,
                                lambdaVersion: version,
                            };
                        }
                        break;
                    default:
                        trigger = props.lambdaTriggers[t];
                        if (trigger !== undefined) {
                            this.addLambdaPermission(trigger, t);
                            this.triggers[t] = trigger.functionArn;
                        }
                        break;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: props.userInvitation?.emailBody,
            emailSubject: props.userInvitation?.emailSubject,
            smsMessage: props.userInvitation?.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled ?? false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        if (props.email && props.emailSettings) {
            throw new Error('you must either provide "email" or "emailSettings", but not both');
        }
        const emailConfiguration = props.email ? props.email._bind(this) : undefinedIfNoKeys({
            from: encodePuny(props.emailSettings?.from),
            replyToEmailAddress: encodePuny(props.emailSettings?.replyTo),
        });
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.any({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsAuthenticationMessage: this.mfaMessage(props),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            userPoolAddOns: undefinedIfNoKeys({
                advancedSecurityMode: props.advancedSecurityMode,
            }),
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration,
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
            accountRecoverySetting: this.accountRecovery(props),
            deviceConfiguration: props.deviceTracking,
            userAttributeUpdateSettings: this.configureUserAttributeChanges(props),
            deletionProtection: defaultDeletionProtection(props.deletionProtection),
        });
        userPool.applyRemovalPolicy(props.removalPolicy);
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Add a lambda trigger to a user pool operation
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cognito_UserPoolOperation(operation);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_IFunction(fn);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addTrigger);
            }
            throw error;
        }
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation.operationName} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        switch (operation.operationName) {
            case 'customEmailSender':
            case 'customSmsSender':
                if (!this.triggers.kmsKeyId) {
                    throw new Error('you must specify a KMS key if you are using customSmsSender or customEmailSender.');
                }
                this.triggers[operation.operationName] = {
                    lambdaArn: fn.functionArn,
                    lambdaVersion: 'V1_0',
                };
                break;
            default:
                this.triggers[operation.operationName] = fn.functionArn;
        }
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: core_1.Lazy.string({ produce: () => this.userPoolArn }),
            scope: this,
        });
    }
    mfaMessage(props) {
        const CODE_TEMPLATE = '{####}';
        const MAX_LENGTH = 140;
        const message = props.mfaMessage;
        if (message && !core_1.Token.isUnresolved(message)) {
            if (!message.includes(CODE_TEMPLATE)) {
                throw new Error(`MFA message must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (message.length > MAX_LENGTH) {
                throw new Error(`MFA message must be between ${CODE_TEMPLATE.length} and ${MAX_LENGTH} characters`);
            }
        }
        return message;
    }
    verificationMessageConfiguration(props) {
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = props.userVerification?.emailStyle ?? VerificationEmailStyle.CODE;
        const emailSubject = props.userVerification?.emailSubject ?? 'Verify your new account';
        const smsMessage = props.userVerification?.smsMessage ?? `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = props.userVerification?.emailBody ?? `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (!core_1.Token.isUnresolved(smsMessage) && smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = props.userVerification?.emailBody ??
                `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = props.signInAliases ?? { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.preferredUsername);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        if (props.enableSmsRole === false && props.smsRole) {
            throw new Error('enableSmsRole cannot be disabled when smsRole is specified');
        }
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
                snsRegion: props.snsRegion,
            };
        }
        if (props.enableSmsRole === false) {
            return undefined;
        }
        const mfaConfiguration = this.mfaConfiguration(props);
        const phoneVerification = props.signInAliases?.phone === true || props.autoVerify?.phone === true;
        const roleRequired = mfaConfiguration?.includes('SMS_MFA') || phoneVerification;
        if (!roleRequired && props.enableSmsRole === undefined) {
            return undefined;
        }
        const smsRoleExternalId = core_1.Names.uniqueId(this).slice(0, 1223); // sts:ExternalId max length of 1224
        const smsRole = props.smsRole ?? new aws_iam_1.Role(this, 'smsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                conditions: {
                    StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                },
            }),
            inlinePolicies: {
                /*
                  * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                  * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                  * Seems like a case of being excessively strict.
                  */
                'sns-publish': new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: ['sns:Publish'],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        return {
            externalId: smsRoleExternalId,
            snsCallerArn: smsRole.roleArn,
            snsRegion: props.snsRegion,
        };
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        const tempPasswordValidity = props.passwordPolicy?.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? props.passwordPolicy.minLength ?? 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity?.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: props.passwordPolicy?.requireLowercase,
            requireUppercase: props.passwordPolicy?.requireUppercase,
            requireNumbers: props.passwordPolicy?.requireDigits,
            requireSymbols: props.passwordPolicy?.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.standardAttributes) {
            const stdAttributes = Object.entries(props.standardAttributes)
                .filter(([, attr]) => !!attr)
                .map(([attrName, attr]) => ({
                name: attr_names_1.StandardAttributeNames[attrName],
                mutable: attr.mutable ?? true,
                required: attr.required ?? false,
            }));
            schema.push(...stdAttributes);
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: attrConfig.numberConstraints?.min?.toString(),
                    maxValue: attrConfig.numberConstraints?.max?.toString(),
                };
                const stringConstraints = {
                    minLength: attrConfig.stringConstraints?.minLen?.toString(),
                    maxLength: attrConfig.stringConstraints?.maxLen?.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: attrConfig.numberConstraints
                        ? numberConstraints
                        : undefined,
                    stringAttributeConstraints: attrConfig.stringConstraints
                        ? stringConstraints
                        : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
    accountRecovery(props) {
        const accountRecovery = props.accountRecovery ?? AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL;
        switch (accountRecovery) {
            case AccountRecovery.EMAIL_AND_PHONE_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_email', priority: 1 },
                        { name: 'verified_phone_number', priority: 2 },
                    ],
                };
            case AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_phone_number', priority: 1 },
                        { name: 'verified_email', priority: 2 },
                    ],
                };
            case AccountRecovery.EMAIL_ONLY:
                return {
                    recoveryMechanisms: [{ name: 'verified_email', priority: 1 }],
                };
            case AccountRecovery.PHONE_ONLY_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [{ name: 'verified_phone_number', priority: 1 }],
                };
            case AccountRecovery.NONE:
                return {
                    recoveryMechanisms: [{ name: 'admin_only', priority: 1 }],
                };
            case AccountRecovery.PHONE_AND_EMAIL:
                return undefined;
            default:
                throw new Error(`Unsupported AccountRecovery type - ${accountRecovery}`);
        }
    }
    configureUserAttributeChanges(props) {
        if (!props.keepOriginal) {
            return undefined;
        }
        const attributesRequireVerificationBeforeUpdate = [];
        if (props.keepOriginal.email) {
            attributesRequireVerificationBeforeUpdate.push(attr_names_1.StandardAttributeNames.email);
        }
        if (props.keepOriginal.phone) {
            attributesRequireVerificationBeforeUpdate.push(attr_names_1.StandardAttributeNames.phoneNumber);
        }
        return {
            attributesRequireVerificationBeforeUpdate,
        };
    }
}
_b = JSII_RTTI_SYMBOL_1;
UserPool[_b] = { fqn: "aws-cdk-lib.aws_cognito.UserPool", version: "2.74.0" };
exports.UserPool = UserPool;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).every(val => val === undefined);
    return allUndefined ? undefined : struct;
}
function encodePuny(input) {
    return input !== undefined ? (0, punycode_1.toASCII)(input) : input;
}
function defaultDeletionProtection(deletionProtection) {
    if (deletionProtection === true) {
        return 'ACTIVE';
    }
    if (deletionProtection === false) {
        return 'INACTIVE';
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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