"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceType = exports.ManagedRuleIdentifiers = exports.CustomPolicy = exports.CustomRule = exports.ManagedRule = exports.MaximumExecutionFrequency = exports.RuleScope = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto_1 = require("crypto");
const events = require("../../aws-events");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * Defines an EventBridge event rule which triggers for rule events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName],
            },
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines an EventBridge event rule which triggers for rule compliance events.
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * Defines an EventBridge event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * Imports an existing rule.
     *
     * @param configRuleName the name of the rule
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
}
/**
 * Determines which resources trigger an evaluation of an AWS Config rule.
 */
class RuleScope {
    /** restricts scope of changes to a specific resource type or resource identifier */
    static fromResource(resourceType, resourceId) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_config_ResourceType(resourceType);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromResource);
            }
            throw error;
        }
        return new RuleScope(resourceId, [resourceType]);
    }
    /** restricts scope of changes to specific resource types */
    static fromResources(resourceTypes) {
        return new RuleScope(undefined, resourceTypes);
    }
    /** restricts scope of changes to a specific tag */
    static fromTag(key, value) {
        return new RuleScope(undefined, undefined, key, value);
    }
    constructor(resourceId, resourceTypes, tagKey, tagValue) {
        this.resourceTypes = resourceTypes;
        this.resourceId = resourceId;
        this.key = tagKey;
        this.value = tagValue;
    }
}
_a = JSII_RTTI_SYMBOL_1;
RuleScope[_a] = { fqn: "aws-cdk-lib.aws_config.RuleScope", version: "2.74.0" };
exports.RuleScope = RuleScope;
/**
 * The maximum frequency at which the AWS Config rule runs evaluations.
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    /**
     * 1 hour.
     */
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    /**
     * 3 hours.
     */
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    /**
     * 6 hours.
     */
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    /**
     * 12 hours.
     */
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    /**
     * 24 hours.
     */
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * A new managed rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_config_ManagedRuleProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ManagedRule);
            }
            throw error;
        }
        this.ruleScope = props.ruleScope;
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.any({ produce: () => renderScope(this.ruleScope) }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
_b = JSII_RTTI_SYMBOL_1;
ManagedRule[_b] = { fqn: "aws-cdk-lib.aws_config.ManagedRule", version: "2.74.0" };
exports.ManagedRule = ManagedRule;
/**
 * The source of the event, such as an AWS service,
 * that triggers AWS Config to evaluate your AWS resources.
 */
var EventSource;
(function (EventSource) {
    /* from aws.config */
    EventSource["AWS_CONFIG"] = "aws.config";
})(EventSource || (EventSource = {}));
/**
 * The type of notification that triggers AWS Config to run an evaluation for a rule.
 */
var MessageType;
(function (MessageType) {
    /**
     * Triggers an evaluation when AWS Config delivers a configuration item as a result of a resource change.
     */
    MessageType["CONFIGURATION_ITEM_CHANGE_NOTIFICATION"] = "ConfigurationItemChangeNotification";
    /**
     * Triggers an evaluation when AWS Config delivers an oversized configuration item.
     */
    MessageType["OVERSIZED_CONFIGURATION_ITEM_CHANGE_NOTIFICATION"] = "OversizedConfigurationItemChangeNotification";
    /**
     * Triggers a periodic evaluation at the frequency specified for MaximumExecutionFrequency.
     */
    MessageType["SCHEDULED_NOTIFICATION"] = "ScheduledNotification";
    /**
     * Triggers a periodic evaluation when AWS Config delivers a configuration snapshot.
     */
    MessageType["CONFIGURATION_SNAPSHOT_DELIVERY_COMPLETED"] = "ConfigurationSnapshotDeliveryCompleted";
})(MessageType || (MessageType = {}));
/**
 * A new custom rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_config_CustomRuleProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CustomRule);
            }
            throw error;
        }
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        this.ruleScope = props.ruleScope;
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: EventSource.AWS_CONFIG,
                messageType: MessageType.CONFIGURATION_ITEM_CHANGE_NOTIFICATION,
            });
            sourceDetails.push({
                eventSource: EventSource.AWS_CONFIG,
                messageType: MessageType.OVERSIZED_CONFIGURATION_ITEM_CHANGE_NOTIFICATION,
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: EventSource.AWS_CONFIG,
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: MessageType.SCHEDULED_NOTIFICATION,
            });
        }
        const hash = (0, crypto_1.createHash)('sha256')
            .update(JSON.stringify({
            fnName: props.lambdaFunction.functionName.toString,
            accountId: core_1.Stack.of(this).resolve(this.env.account),
            region: core_1.Stack.of(this).resolve(this.env.region),
        }), 'utf8')
            .digest('base64');
        const customRulePermissionId = `CustomRulePermission${hash}`;
        if (!props.lambdaFunction.permissionsNode.tryFindChild(customRulePermissionId)) {
            props.lambdaFunction.addPermission(customRulePermissionId, {
                principal: new iam.ServicePrincipal('config.amazonaws.com'),
                sourceAccount: this.env.account,
            });
        }
        ;
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.any({ produce: () => renderScope(this.ruleScope) }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
_c = JSII_RTTI_SYMBOL_1;
CustomRule[_c] = { fqn: "aws-cdk-lib.aws_config.CustomRule", version: "2.74.0" };
exports.CustomRule = CustomRule;
/**
 * A new custom policy.
 *
 * @resource AWS::Config::ConfigRule
 */
class CustomPolicy extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_config_CustomPolicyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CustomPolicy);
            }
            throw error;
        }
        if (!props.policyText || [...props.policyText].length === 0) {
            throw new Error('Policy Text cannot be empty.');
        }
        if ([...props.policyText].length > 10000) {
            throw new Error('Policy Text is limited to 10,000 characters or less.');
        }
        const sourceDetails = [];
        this.ruleScope = props.ruleScope;
        sourceDetails.push({
            eventSource: EventSource.AWS_CONFIG,
            messageType: MessageType.CONFIGURATION_ITEM_CHANGE_NOTIFICATION,
        });
        sourceDetails.push({
            eventSource: EventSource.AWS_CONFIG,
            messageType: MessageType.OVERSIZED_CONFIGURATION_ITEM_CHANGE_NOTIFICATION,
        });
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            scope: core_1.Lazy.any({ produce: () => renderScope(this.ruleScope) }),
            source: {
                owner: 'CUSTOM_POLICY',
                sourceDetails,
                customPolicyDetails: {
                    enableDebugLogDelivery: props.enableDebugLog,
                    policyRuntime: 'guard-2.x.x',
                    policyText: props.policyText,
                },
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isCustomWithChanges = true;
    }
}
_d = JSII_RTTI_SYMBOL_1;
CustomPolicy[_d] = { fqn: "aws-cdk-lib.aws_config.CustomPolicy", version: "2.74.0" };
exports.CustomPolicy = CustomPolicy;
/**
 * Managed rules that are supported by AWS Config.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/managed-rules-by-aws-config.html
 */
class ManagedRuleIdentifiers {
    // utility class
    constructor() { }
}
_e = JSII_RTTI_SYMBOL_1;
ManagedRuleIdentifiers[_e] = { fqn: "aws-cdk-lib.aws_config.ManagedRuleIdentifiers", version: "2.74.0" };
/**
 * Checks that the inline policies attached to your AWS Identity and Access Management users,
 * roles, and groups do not allow blocked actions on all AWS Key Management Service keys.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-inline-policy-blocked-kms-actions.html
 */
ManagedRuleIdentifiers.IAM_INLINE_POLICY_BLOCKED_KMS_ACTIONS = 'IAM_INLINE_POLICY_BLOCKED_KMS_ACTIONS';
/**
 * Checks that the managed AWS Identity and Access Management policies that you create do not
 * allow blocked actions on all AWS AWS KMS keys.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-customer-policy-blocked-kms-actions.html
 */
ManagedRuleIdentifiers.IAM_CUSTOMER_POLICY_BLOCKED_KMS_ACTIONS = 'IAM_CUSTOMER_POLICY_BLOCKED_KMS_ACTIONS';
/**
 * Checks whether the active access keys are rotated within the number of days specified in maxAccessKeyAge.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 */
ManagedRuleIdentifiers.ACCESS_KEYS_ROTATED = 'ACCESS_KEYS_ROTATED';
/**
 * Checks whether AWS account is part of AWS Organizations.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/account-part-of-organizations.html
 */
ManagedRuleIdentifiers.ACCOUNT_PART_OF_ORGANIZATIONS = 'ACCOUNT_PART_OF_ORGANIZATIONS';
/**
 * Checks whether ACM Certificates in your account are marked for expiration within the specified number of days.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/acm-certificate-expiration-check.html
 */
ManagedRuleIdentifiers.ACM_CERTIFICATE_EXPIRATION_CHECK = 'ACM_CERTIFICATE_EXPIRATION_CHECK';
/**
 * Checks if an Application Load Balancer (ALB) is configured with a user defined desync mitigation mode.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/alb-desync-mode-check.html
 */
ManagedRuleIdentifiers.ALB_DESYNC_MODE_CHECK = 'ALB_DESYNC_MODE_CHECK';
/**
 * Checks if rule evaluates Application Load Balancers (ALBs) to ensure they are configured to drop http headers.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/alb-http-drop-invalid-header-enabled.html
 */
ManagedRuleIdentifiers.ALB_HTTP_DROP_INVALID_HEADER_ENABLED = 'ALB_HTTP_DROP_INVALID_HEADER_ENABLED';
/**
 * Checks whether HTTP to HTTPS redirection is configured on all HTTP listeners of Application Load Balancer.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/alb-http-to-https-redirection-check.html
 */
ManagedRuleIdentifiers.ALB_HTTP_TO_HTTPS_REDIRECTION_CHECK = 'ALB_HTTP_TO_HTTPS_REDIRECTION_CHECK';
/**
 * Checks if Web Application Firewall (WAF) is enabled on Application Load Balancers (ALBs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/alb-waf-enabled.html
 */
ManagedRuleIdentifiers.ALB_WAF_ENABLED = 'ALB_WAF_ENABLED';
/**
 * Checks if Amazon API Gateway V2 stages have access logging enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gwv2-access-logs-enabled.html
 */
ManagedRuleIdentifiers.API_GWV2_ACCESS_LOGS_ENABLED = 'API_GWV2_ACCESS_LOGS_ENABLED';
/**
 * Checks if Amazon API Gatewayv2 API routes have an authorization type set.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gwv2-authorization-type-configured.html
 */
ManagedRuleIdentifiers.API_GWV2_AUTHORIZATION_TYPE_CONFIGURED = 'API_GWV2_AUTHORIZATION_TYPE_CONFIGURED';
/**
 * Checks if an Amazon API Gateway API stage is using an AWS WAF Web ACL.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-associated-with-waf.html
 */
ManagedRuleIdentifiers.API_GW_ASSOCIATED_WITH_WAF = 'API_GW_ASSOCIATED_WITH_WAF';
/**
 * Checks that all methods in Amazon API Gateway stages have caching enabled and encrypted.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-cache-enabled-and-encrypted.html
 */
ManagedRuleIdentifiers.API_GW_CACHE_ENABLED_AND_ENCRYPTED = 'API_GW_CACHE_ENABLED_AND_ENCRYPTED';
/**
 * Checks that Amazon API Gateway APIs are of the type specified in the rule parameter endpointConfigurationType.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-endpoint-type-check.html
 */
ManagedRuleIdentifiers.API_GW_ENDPOINT_TYPE_CHECK = 'API_GW_ENDPOINT_TYPE_CHECK';
/**
 * Checks that all methods in Amazon API Gateway stage has logging enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-execution-logging-enabled.html
 */
ManagedRuleIdentifiers.API_GW_EXECUTION_LOGGING_ENABLED = 'API_GW_EXECUTION_LOGGING_ENABLED';
/**
 * Checks if a REST API stage uses an Secure Sockets Layer (SSL) certificate.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-ssl-enabled.html
 */
ManagedRuleIdentifiers.API_GW_SSL_ENABLED = 'API_GW_SSL_ENABLED';
/**
 * Checks if AWS X-Ray tracing is enabled on Amazon API Gateway REST APIs.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-xray-enabled.html
 */
ManagedRuleIdentifiers.API_GW_XRAY_ENABLED = 'API_GW_XRAY_ENABLED';
/**
 * Checks whether running instances are using specified AMIs.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/approved-amis-by-id.html
 */
ManagedRuleIdentifiers.APPROVED_AMIS_BY_ID = 'APPROVED_AMIS_BY_ID';
/**
 * Checks whether running instances are using specified AMIs.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/approved-amis-by-tag.html
 */
ManagedRuleIdentifiers.APPROVED_AMIS_BY_TAG = 'APPROVED_AMIS_BY_TAG';
/**
 * Checks if a recovery point was created for Amazon Aurora DB clusters.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/aurora-last-backup-recovery-point-created.html
 */
ManagedRuleIdentifiers.AURORA_LAST_BACKUP_RECOVERY_POINT_CREATED = 'AURORA_LAST_BACKUP_RECOVERY_POINT_CREATED';
/**
 * Checks if an Amazon Aurora MySQL cluster has backtracking enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/aurora-mysql-backtracking-enabled.html
 */
ManagedRuleIdentifiers.AURORA_MYSQL_BACKTRACKING_ENABLED = 'AURORA_MYSQL_BACKTRACKING_ENABLED';
/**
 * Checks if Amazon Aurora DB clusters are protected by a backup plan.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/aurora-resources-protected-by-backup-plan.html
 */
ManagedRuleIdentifiers.AURORA_RESOURCES_PROTECTED_BY_BACKUP_PLAN = 'AURORA_RESOURCES_PROTECTED_BY_BACKUP_PLAN';
/**
 * Checks if Capacity Rebalancing is enabled for Amazon EC2 Auto Scaling groups that use multiple instance types.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-capacity-rebalancing.html
 */
ManagedRuleIdentifiers.AUTOSCALING_CAPACITY_REBALANCING = 'AUTOSCALING_CAPACITY_REBALANCING';
/**
 * Checks whether your Auto Scaling groups that are associated with a load balancer are using
 * Elastic Load Balancing health checks.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-group-elb-healthcheck-required.html
 */
ManagedRuleIdentifiers.AUTOSCALING_GROUP_ELB_HEALTHCHECK_REQUIRED = 'AUTOSCALING_GROUP_ELB_HEALTHCHECK_REQUIRED';
/**
 * Checks whether only IMDSv2 is enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-launchconfig-requires-imdsv2.html
 */
ManagedRuleIdentifiers.AUTOSCALING_LAUNCHCONFIG_REQUIRES_IMDSV2 = 'AUTOSCALING_LAUNCHCONFIG_REQUIRES_IMDSV2';
/**
 * Checks the number of network hops that the metadata token can travel.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-launch-config-hop-limit.html
 */
ManagedRuleIdentifiers.AUTOSCALING_LAUNCH_CONFIG_HOP_LIMIT = 'AUTOSCALING_LAUNCH_CONFIG_HOP_LIMIT';
/**
 * Checks if Amazon EC2 Auto Scaling groups have public IP addresses enabled through Launch Configurations.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-launch-config-public-ip-disabled.html
 */
ManagedRuleIdentifiers.AUTOSCALING_LAUNCH_CONFIG_PUBLIC_IP_DISABLED = 'AUTOSCALING_LAUNCH_CONFIG_PUBLIC_IP_DISABLED';
/**
 * Checks if an Amazon Elastic Compute Cloud (EC2) Auto Scaling group is created from an EC2 launch template.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-launch-template.html
 */
ManagedRuleIdentifiers.AUTOSCALING_LAUNCH_TEMPLATE = 'AUTOSCALING_LAUNCH_TEMPLATE';
/**
 * Checks if the Auto Scaling group spans multiple Availability Zones.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-multiple-az.html
 */
ManagedRuleIdentifiers.AUTOSCALING_MULTIPLE_AZ = 'AUTOSCALING_MULTIPLE_AZ';
/**
 * Checks if an Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling group uses multiple instance types.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-multiple-instance-types.html
 */
ManagedRuleIdentifiers.AUTOSCALING_MULTIPLE_INSTANCE_TYPES = 'AUTOSCALING_MULTIPLE_INSTANCE_TYPES';
/**
 * Checks if a backup plan has a backup rule that satisfies the required frequency and retention period.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/backup-plan-min-frequency-and-min-retention-check.html
 */
ManagedRuleIdentifiers.BACKUP_PLAN_MIN_FREQUENCY_AND_MIN_RETENTION_CHECK = 'BACKUP_PLAN_MIN_FREQUENCY_AND_MIN_RETENTION_CHECK';
/**
 * Checks if a recovery point is encrypted.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/backup-recovery-point-encrypted.html
 */
ManagedRuleIdentifiers.BACKUP_RECOVERY_POINT_ENCRYPTED = 'BACKUP_RECOVERY_POINT_ENCRYPTED';
/**
 * Checks if a backup vault has an attached resource-based policy which prevents deletion of recovery points.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/backup-recovery-point-manual-deletion-disabled.html
 */
ManagedRuleIdentifiers.BACKUP_RECOVERY_POINT_MANUAL_DELETION_DISABLED = 'BACKUP_RECOVERY_POINT_MANUAL_DELETION_DISABLED';
/**
 * Checks if a recovery point expires no earlier than after the specified period.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/backup-recovery-point-minimum-retention-check.html
 */
ManagedRuleIdentifiers.BACKUP_RECOVERY_POINT_MINIMUM_RETENTION_CHECK = 'BACKUP_RECOVERY_POINT_MINIMUM_RETENTION_CHECK';
/**
 * Checks if an AWS Elastic Beanstalk environment is configured for enhanced health reporting.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/beanstalk-enhanced-health-reporting-enabled.html
 */
ManagedRuleIdentifiers.BEANSTALK_ENHANCED_HEALTH_REPORTING_ENABLED = 'BEANSTALK_ENHANCED_HEALTH_REPORTING_ENABLED';
/**
 * Checks if Classic Load Balancers (CLB) are configured with a user defined Desync mitigation mode.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/clb-desync-mode-check.html
 */
ManagedRuleIdentifiers.CLB_DESYNC_MODE_CHECK = 'CLB_DESYNC_MODE_CHECK';
/**
 * Checks if a Classic Load Balancer spans multiple Availability Zones (AZs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/clb-multiple-az.html
 */
ManagedRuleIdentifiers.CLB_MULTIPLE_AZ = 'CLB_MULTIPLE_AZ';
/**
 * Checks whether an AWS CloudFormation stack's actual configuration differs, or has drifted,
 * from it's expected configuration.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-drift-detection-check.html
 */
ManagedRuleIdentifiers.CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK = 'CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK';
/**
 * Checks whether your CloudFormation stacks are sending event notifications to an SNS topic.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-notification-check.html
 */
ManagedRuleIdentifiers.CLOUDFORMATION_STACK_NOTIFICATION_CHECK = 'CLOUDFORMATION_STACK_NOTIFICATION_CHECK';
/**
 * Checks if Amazon CloudFront distributions are configured to capture information from
 * Amazon Simple Storage Service (Amazon S3) server access logs.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-accesslogs-enabled.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_ACCESSLOGS_ENABLED = 'CLOUDFRONT_ACCESSLOGS_ENABLED';
/**
 * Checks if Amazon CloudFront distributions are associated with either WAF or WAFv2 web access control lists (ACLs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-associated-with-waf.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_ASSOCIATED_WITH_WAF = 'CLOUDFRONT_ASSOCIATED_WITH_WAF';
/**
 * Checks if the certificate associated with an Amazon CloudFront distribution is the default Secure Sockets Layer (SSL) certificate.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-custom-ssl-certificate.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_CUSTOM_SSL_CERTIFICATE = 'CLOUDFRONT_CUSTOM_SSL_CERTIFICATE';
/**
 * Checks if an Amazon CloudFront distribution is configured to return a specific object that is the default root object.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-default-root-object-configured.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_DEFAULT_ROOT_OBJECT_CONFIGURED = 'CLOUDFRONT_DEFAULT_ROOT_OBJECT_CONFIGURED';
/**
 * Checks if CloudFront distributions are using deprecated SSL protocols for HTTPS communication between
 * CloudFront edge locations and custom origins.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-no-deprecated-ssl-protocols.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_NO_DEPRECATED_SSL_PROTOCOLS = 'CLOUDFRONT_NO_DEPRECATED_SSL_PROTOCOLS';
/**
 * Checks that Amazon CloudFront distribution with Amazon S3 Origin type has Origin Access Identity (OAI) configured.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-origin-access-identity-enabled.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_ORIGIN_ACCESS_IDENTITY_ENABLED = 'CLOUDFRONT_ORIGIN_ACCESS_IDENTITY_ENABLED';
/**
 * Checks whether an origin group is configured for the distribution of at least 2 origins in the
 * origin group for Amazon CloudFront.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-origin-failover-enabled.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_ORIGIN_FAILOVER_ENABLED = 'CLOUDFRONT_ORIGIN_FAILOVER_ENABLED';
/**
 * Checks if Amazon CloudFront distributions are using a minimum security policy and cipher suite of TLSv1.2 or
 * greater for viewer connections.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-security-policy-check.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_SECURITY_POLICY_CHECK = 'CLOUDFRONT_SECURITY_POLICY_CHECK';
/**
 * Checks if Amazon CloudFront distributions are using a custom SSL certificate and are configured
 * to use SNI to serve HTTPS requests.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-sni-enabled.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_SNI_ENABLED = 'CLOUDFRONT_SNI_ENABLED';
/**
 * Checks if Amazon CloudFront distributions are encrypting traffic to custom origins.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-traffic-to-origin-encrypted.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_TRAFFIC_TO_ORIGIN_ENCRYPTED = 'CLOUDFRONT_TRAFFIC_TO_ORIGIN_ENCRYPTED';
/**
 * Checks whether your Amazon CloudFront distributions use HTTPS (directly or via a redirection).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-viewer-policy-https.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_VIEWER_POLICY_HTTPS = 'CLOUDFRONT_VIEWER_POLICY_HTTPS';
/**
 * Checks whether AWS CloudTrail trails are configured to send logs to Amazon CloudWatch Logs.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-cloud-watch-logs-enabled.html
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_CLOUD_WATCH_LOGS_ENABLED = 'CLOUD_TRAIL_CLOUD_WATCH_LOGS_ENABLED';
/**
 * Checks whether AWS CloudTrail is enabled in your AWS account.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-enabled.html
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_ENABLED = 'CLOUD_TRAIL_ENABLED';
/**
 * Checks whether AWS CloudTrail is configured to use the server side encryption (SSE)
 * AWS Key Management Service (AWS KMS) customer master key (CMK) encryption.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-encryption-enabled.html
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_ENCRYPTION_ENABLED = 'CLOUD_TRAIL_ENCRYPTION_ENABLED';
/**
 * Checks whether AWS CloudTrail creates a signed digest file with logs.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-log-file-validation-enabled.html
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_LOG_FILE_VALIDATION_ENABLED = 'CLOUD_TRAIL_LOG_FILE_VALIDATION_ENABLED';
/**
 * Checks whether at least one AWS CloudTrail trail is logging Amazon S3 data events for all S3 buckets.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-s3-dataevents-enabled.html
 */
ManagedRuleIdentifiers.CLOUDTRAIL_S3_DATAEVENTS_ENABLED = 'CLOUDTRAIL_S3_DATAEVENTS_ENABLED';
/**
 * Checks that there is at least one AWS CloudTrail trail defined with security best practices.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-security-trail-enabled.html
 */
ManagedRuleIdentifiers.CLOUDTRAIL_SECURITY_TRAIL_ENABLED = 'CLOUDTRAIL_SECURITY_TRAIL_ENABLED';
/**
 * Checks whether CloudWatch alarms have at least one alarm action, one INSUFFICIENT_DATA action,
 * or one OK action enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-action-check.html
 */
ManagedRuleIdentifiers.CLOUDWATCH_ALARM_ACTION_CHECK = 'CLOUDWATCH_ALARM_ACTION_CHECK';
/**
 * Checks if Amazon CloudWatch alarms actions are in enabled state.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-action-enabled-check.html
 */
ManagedRuleIdentifiers.CLOUDWATCH_ALARM_ACTION_ENABLED_CHECK = 'CLOUDWATCH_ALARM_ACTION_ENABLED_CHECK';
/**
 * Checks whether the specified resource type has a CloudWatch alarm for the specified metric.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-resource-check.html
 */
ManagedRuleIdentifiers.CLOUDWATCH_ALARM_RESOURCE_CHECK = 'CLOUDWATCH_ALARM_RESOURCE_CHECK';
/**
 * Checks whether CloudWatch alarms with the given metric name have the specified settings.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-settings-check.html
 */
ManagedRuleIdentifiers.CLOUDWATCH_ALARM_SETTINGS_CHECK = 'CLOUDWATCH_ALARM_SETTINGS_CHECK';
/**
 * Checks whether a log group in Amazon CloudWatch Logs is encrypted with
 * a AWS Key Management Service (KMS) managed Customer Master Keys (CMK).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-log-group-encrypted.html
 */
ManagedRuleIdentifiers.CLOUDWATCH_LOG_GROUP_ENCRYPTED = 'CLOUDWATCH_LOG_GROUP_ENCRYPTED';
/**
 * Checks that key rotation is enabled for each key and matches to the key ID of the
 * customer created customer master key (CMK).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cmk-backing-key-rotation-enabled.html
 */
ManagedRuleIdentifiers.CMK_BACKING_KEY_ROTATION_ENABLED = 'CMK_BACKING_KEY_ROTATION_ENABLED';
/**
 * Checks if an AWS CodeBuild project has encryption enabled for all of its artifacts.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-artifact-encryption.html
 */
ManagedRuleIdentifiers.CODEBUILD_PROJECT_ARTIFACT_ENCRYPTION = 'CODEBUILD_PROJECT_ARTIFACT_ENCRYPTION';
/**
 * Checks if an AWS CodeBuild project environment has privileged mode enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-environment-privileged-check.html
 */
ManagedRuleIdentifiers.CODEBUILD_PROJECT_ENVIRONMENT_PRIVILEGED_CHECK = 'CODEBUILD_PROJECT_ENVIRONMENT_PRIVILEGED_CHECK';
/**
 * Checks whether the project contains environment variables AWS_ACCESS_KEY_ID and AWS_SECRET_ACCESS_KEY.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-envvar-awscred-check.html
 */
ManagedRuleIdentifiers.CODEBUILD_PROJECT_ENVVAR_AWSCRED_CHECK = 'CODEBUILD_PROJECT_ENVVAR_AWSCRED_CHECK';
/**
 * Checks if an AWS CodeBuild project environment has at least one log option enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-logging-enabled.html
 */
ManagedRuleIdentifiers.CODEBUILD_PROJECT_LOGGING_ENABLED = 'CODEBUILD_PROJECT_LOGGING_ENABLED';
/**
 * Checks if a AWS CodeBuild project configured with Amazon S3 Logs has encryption enabled for its logs.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-s3-logs-encrypted.html
 */
ManagedRuleIdentifiers.CODEBUILD_PROJECT_S3_LOGS_ENCRYPTED = 'CODEBUILD_PROJECT_S3_LOGS_ENCRYPTED';
/**
 * Checks whether the GitHub or Bitbucket source repository URL contains either personal access tokens
 * or user name and password.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-source-repo-url-check.html
 */
ManagedRuleIdentifiers.CODEBUILD_PROJECT_SOURCE_REPO_URL_CHECK = 'CODEBUILD_PROJECT_SOURCE_REPO_URL_CHECK';
/**
 * Checks if the deployment group is configured with automatic deployment rollback and
 * deployment monitoring with alarms attached.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codedeploy-auto-rollback-monitor-enabled.html
 */
ManagedRuleIdentifiers.CODEDEPLOY_AUTO_ROLLBACK_MONITOR_ENABLED = 'CODEDEPLOY_AUTO_ROLLBACK_MONITOR_ENABLED';
/**
 * Checks if the deployment group for EC2/On-Premises Compute Platform is configured with
 * a minimum healthy hosts fleet percentage or host count greater than or equal to the input threshold.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codedeploy-ec2-minimum-healthy-hosts-configured.html
 */
ManagedRuleIdentifiers.CODEDEPLOY_EC2_MINIMUM_HEALTHY_HOSTS_CONFIGURED = 'CODEDEPLOY_EC2_MINIMUM_HEALTHY_HOSTS_CONFIGURED';
/**
 * Checks if the deployment group for Lambda Compute Platform is not using the default deployment configuration.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codedeploy-lambda-allatonce-traffic-shift-disabled.html
 */
ManagedRuleIdentifiers.CODEDEPLOY_LAMBDA_ALLATONCE_TRAFFIC_SHIFT_DISABLED = 'CODEDEPLOY_LAMBDA_ALLATONCE_TRAFFIC_SHIFT_DISABLED';
/**
 * Checks whether the first deployment stage of the AWS CodePipeline performs more than one deployment.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codepipeline-deployment-count-check.html
 */
ManagedRuleIdentifiers.CODEPIPELINE_DEPLOYMENT_COUNT_CHECK = 'CODEPIPELINE_DEPLOYMENT_COUNT_CHECK';
/**
 * Checks whether each stage in the AWS CodePipeline deploys to more than N times the number of
 * the regions the AWS CodePipeline has deployed in all the previous combined stages,
 * where N is the region fanout number.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codepipeline-region-fanout-check.html
 */
ManagedRuleIdentifiers.CODEPIPELINE_REGION_FANOUT_CHECK = 'CODEPIPELINE_REGION_FANOUT_CHECK';
/**
 * Checks whether Amazon CloudWatch LogGroup retention period is set to specific number of days.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cw-loggroup-retention-period-check.html
 */
ManagedRuleIdentifiers.CW_LOGGROUP_RETENTION_PERIOD_CHECK = 'CW_LOGGROUP_RETENTION_PERIOD_CHECK';
/**
 * Checks that DynamoDB Accelerator (DAX) clusters are encrypted.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dax-encryption-enabled.html
 */
ManagedRuleIdentifiers.DAX_ENCRYPTION_ENABLED = 'DAX_ENCRYPTION_ENABLED';
/**
 * Checks whether RDS DB instances have backups enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/db-instance-backup-enabled.html
 */
ManagedRuleIdentifiers.RDS_DB_INSTANCE_BACKUP_ENABLED = 'DB_INSTANCE_BACKUP_ENABLED';
/**
 * Checks instances for specified tenancy.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/desired-instance-tenancy.html
 */
ManagedRuleIdentifiers.EC2_DESIRED_INSTANCE_TENANCY = 'DESIRED_INSTANCE_TENANCY';
/**
 * Checks whether your EC2 instances are of the specified instance types.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/desired-instance-type.html
 */
ManagedRuleIdentifiers.EC2_DESIRED_INSTANCE_TYPE = 'DESIRED_INSTANCE_TYPE';
/**
 * Checks whether AWS Database Migration Service replication instances are public.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dms-replication-not-public.html
 */
ManagedRuleIdentifiers.DMS_REPLICATION_NOT_PUBLIC = 'DMS_REPLICATION_NOT_PUBLIC';
/**
 * Checks whether Auto Scaling or On-Demand is enabled on your DynamoDB tables and/or global secondary indexes.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-autoscaling-enabled.html
 */
ManagedRuleIdentifiers.DYNAMODB_AUTOSCALING_ENABLED = 'DYNAMODB_AUTOSCALING_ENABLED';
/**
 * Checks whether Amazon DynamoDB table is present in AWS Backup plans.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-in-backup-plan.html
 */
ManagedRuleIdentifiers.DYNAMODB_IN_BACKUP_PLAN = 'DYNAMODB_IN_BACKUP_PLAN';
/**
 * Checks if a recovery point was created for Amazon DynamoDB Tables within the specified period.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-last-backup-recovery-point-created.html
 */
ManagedRuleIdentifiers.DYNAMODB_LAST_BACKUP_RECOVERY_POINT_CREATED = 'DYNAMODB_LAST_BACKUP_RECOVERY_POINT_CREATED';
/**
 * Checks that point in time recovery (PITR) is enabled for Amazon DynamoDB tables.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-pitr-enabled.html
 */
ManagedRuleIdentifiers.DYNAMODB_PITR_ENABLED = 'DYNAMODB_PITR_ENABLED';
/**
 * Checks if Amazon DynamoDB tables are protected by a backup plan.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-resources-protected-by-backup-plan.html
 */
ManagedRuleIdentifiers.DYNAMODB_RESOURCES_PROTECTED_BY_BACKUP_PLAN = 'DYNAMODB_RESOURCES_PROTECTED_BY_BACKUP_PLAN';
/**
 * Checks whether Amazon DynamoDB table is encrypted with AWS Key Management Service (KMS).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-table-encrypted-kms.html
 */
ManagedRuleIdentifiers.DYNAMODB_TABLE_ENCRYPTED_KMS = 'DYNAMODB_TABLE_ENCRYPTED_KMS';
/**
 * Checks whether the Amazon DynamoDB tables are encrypted and checks their status.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-table-encryption-enabled.html
 */
ManagedRuleIdentifiers.DYNAMODB_TABLE_ENCRYPTION_ENABLED = 'DYNAMODB_TABLE_ENCRYPTION_ENABLED';
/**
 * Checks whether provisioned DynamoDB throughput is approaching the maximum limit for your account.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-throughput-limit-check.html
 */
ManagedRuleIdentifiers.DYNAMODB_THROUGHPUT_LIMIT_CHECK = 'DYNAMODB_THROUGHPUT_LIMIT_CHECK';
/**
 * Checks if Amazon Elastic Block Store (Amazon EBS) volumes are added in backup plans of AWS Backup.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ebs-in-backup-plan.html
 */
ManagedRuleIdentifiers.EBS_IN_BACKUP_PLAN = 'EBS_IN_BACKUP_PLAN';
/**
 * Checks whether Amazon Elastic File System (Amazon EFS) file systems are added
 * in the backup plans of AWS Backup.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/efs-in-backup-plan.html
 */
ManagedRuleIdentifiers.EFS_IN_BACKUP_PLAN = 'EFS_IN_BACKUP_PLAN';
/**
 * Check that Amazon Elastic Block Store (EBS) encryption is enabled by default.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-ebs-encryption-by-default.html
 */
ManagedRuleIdentifiers.EC2_EBS_ENCRYPTION_BY_DEFAULT = 'EC2_EBS_ENCRYPTION_BY_DEFAULT';
/**
 * Checks whether EBS optimization is enabled for your EC2 instances that can be EBS-optimized.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ebs-optimized-instance.html
 */
ManagedRuleIdentifiers.EBS_OPTIMIZED_INSTANCE = 'EBS_OPTIMIZED_INSTANCE';
/**
 * Checks if Amazon Elastic Block Store (Amazon EBS) volumes are protected by a backup plan.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ebs-resources-protected-by-backup-plan.html
 */
ManagedRuleIdentifiers.EBS_RESOURCES_PROTECTED_BY_BACKUP_PLAN = 'EBS_RESOURCES_PROTECTED_BY_BACKUP_PLAN';
/**
 * Checks whether Amazon Elastic Block Store snapshots are not publicly restorable.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ebs-snapshot-public-restorable-check.html
 */
ManagedRuleIdentifiers.EBS_SNAPSHOT_PUBLIC_RESTORABLE_CHECK = 'EBS_SNAPSHOT_PUBLIC_RESTORABLE_CHECK';
/**
 * Checks whether detailed monitoring is enabled for EC2 instances.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-detailed-monitoring-enabled.html
 */
ManagedRuleIdentifiers.EC2_INSTANCE_DETAILED_MONITORING_ENABLED = 'EC2_INSTANCE_DETAILED_MONITORING_ENABLED';
/**
 * Checks whether the Amazon EC2 instances in your account are managed by AWS Systems Manager.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-managed-by-systems-manager.html
 */
ManagedRuleIdentifiers.EC2_INSTANCE_MANAGED_BY_SSM = 'EC2_INSTANCE_MANAGED_BY_SSM';
/**
 * Checks if an Amazon Elastic Compute Cloud (Amazon EC2) instance has an Identity and Access
 * Management (IAM) profile attached to it. This rule is NON_COMPLIANT if no IAM profile is
 * attached to the Amazon EC2 instance.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-profile-attached.html
 */
ManagedRuleIdentifiers.EC2_INSTANCE_PROFILE_ATTACHED = 'EC2_INSTANCE_PROFILE_ATTACHED';
/**
 * Checks if Amazon Elastic Compute Cloud (Amazon EC2) uses multiple ENIs (Elastic Network Interfaces)
 * or Elastic Fabric Adapters (EFAs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-multiple-eni-check.html
 */
ManagedRuleIdentifiers.EC2_INSTANCE_MULTIPLE_ENI_CHECK = 'EC2_INSTANCE_MULTIPLE_ENI_CHECK';
/**
 * Checks whether Amazon Elastic Compute Cloud (Amazon EC2) instances have a public IP association.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-no-public-ip.html
 */
ManagedRuleIdentifiers.EC2_INSTANCE_NO_PUBLIC_IP = 'EC2_INSTANCE_NO_PUBLIC_IP';
/**
 * Checks if a recovery point was created for Amazon Elastic Compute Cloud (Amazon EC2) instances.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-last-backup-recovery-point-created.html
 */
ManagedRuleIdentifiers.EC2_LAST_BACKUP_RECOVERY_POINT_CREATED = 'EC2_LAST_BACKUP_RECOVERY_POINT_CREATED';
/**
 * Checks whether your EC2 instances belong to a virtual private cloud (VPC).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instances-in-vpc.html
 */
ManagedRuleIdentifiers.EC2_INSTANCES_IN_VPC = 'INSTANCES_IN_VPC';
/**
 * Checks that none of the specified applications are installed on the instance.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-applications-blacklisted.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_APPLICATIONS_BLOCKED = 'EC2_MANAGEDINSTANCE_APPLICATIONS_BLACKLISTED';
/**
 * Checks whether all of the specified applications are installed on the instance.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-applications-required.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_APPLICATIONS_REQUIRED = 'EC2_MANAGEDINSTANCE_APPLICATIONS_REQUIRED';
/**
 * Checks whether the compliance status of AWS Systems Manager association compliance is COMPLIANT
 * or NON_COMPLIANT after the association execution on the instance.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-association-compliance-status-check.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_ASSOCIATION_COMPLIANCE_STATUS_CHECK = 'EC2_MANAGEDINSTANCE_ASSOCIATION_COMPLIANCE_STATUS_CHECK';
/**
 * Checks whether instances managed by AWS Systems Manager are configured to collect blocked inventory types.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-inventory-blacklisted.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_INVENTORY_BLOCKED = 'EC2_MANAGEDINSTANCE_INVENTORY_BLACKLISTED';
/**
 * Checks whether the compliance status of the Amazon EC2 Systems Manager patch compliance is
 * COMPLIANT or NON_COMPLIANT after the patch installation on the instance.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-patch-compliance-status-check.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_PATCH_COMPLIANCE_STATUS_CHECK = 'EC2_MANAGEDINSTANCE_PATCH_COMPLIANCE_STATUS_CHECK';
/**
 * Checks whether EC2 managed instances have the desired configurations.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-platform-check.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_PLATFORM_CHECK = 'EC2_MANAGEDINSTANCE_PLATFORM_CHECK';
/**
 * Checks if running Amazon Elastic Compute Cloud (EC2) instances are launched using amazon key pairs.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-no-amazon-key-pair.html
 */
ManagedRuleIdentifiers.EC2_NO_AMAZON_KEY_PAIR = 'EC2_NO_AMAZON_KEY_PAIR';
/**
 * Checks if the virtualization type of an EC2 instance is paravirtual.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-paravirtual-instance-check.html
 */
ManagedRuleIdentifiers.EC2_PARAVIRTUAL_INSTANCE_CHECK = 'EC2_PARAVIRTUAL_INSTANCE_CHECK';
/**
 * Checks if Amazon Elastic Compute Cloud (Amazon EC2) instances are protected by a backup plan.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-resources-protected-by-backup-plan.html
 */
ManagedRuleIdentifiers.EC2_RESOURCES_PROTECTED_BY_BACKUP_PLAN = 'EC2_RESOURCES_PROTECTED_BY_BACKUP_PLAN';
/**
 * Checks that security groups are attached to Amazon Elastic Compute Cloud (Amazon EC2) instances
 * or to an elastic network interface.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-security-group-attached-to-eni.html
 */
ManagedRuleIdentifiers.EC2_SECURITY_GROUP_ATTACHED_TO_ENI = 'EC2_SECURITY_GROUP_ATTACHED_TO_ENI';
/**
 * Checks if non-default security groups are attached to Elastic network interfaces (ENIs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-security-group-attached-to-eni-periodic.html
 */
ManagedRuleIdentifiers.EC2_SECURITY_GROUP_ATTACHED_TO_ENI_PERIODIC = 'EC2_SECURITY_GROUP_ATTACHED_TO_ENI_PERIODIC';
/**
 * Checks whether there are instances stopped for more than the allowed number of days.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-stopped-instance.html
 */
ManagedRuleIdentifiers.EC2_STOPPED_INSTANCE = 'EC2_STOPPED_INSTANCE';
/**
 * Checks if an Amazon Elastic Compute Cloud (EC2) instance metadata
 * has a specified token hop limit that is below the desired limit.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-token-hop-limit-check.html
 */
ManagedRuleIdentifiers.EC2_TOKEN_HOP_LIMIT_CHECK = 'EC2_TOKEN_HOP_LIMIT_CHECK';
/**
 * Checks if Amazon Elastic Compute Cloud (Amazon EC2) Transit Gateways have 'AutoAcceptSharedAttachments' enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-transit-gateway-auto-vpc-attach-disabled.html
 */
ManagedRuleIdentifiers.EC2_TRANSIT_GATEWAY_AUTO_VPC_ATTACH_DISABLED = 'EC2_TRANSIT_GATEWAY_AUTO_VPC_ATTACH_DISABLED';
/**
 * Checks whether EBS volumes are attached to EC2 instances.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-volume-inuse-check.html
 */
ManagedRuleIdentifiers.EC2_VOLUME_INUSE_CHECK = 'EC2_VOLUME_INUSE_CHECK';
/**
 * Checks if a private Amazon Elastic Container Registry (ECR) repository has image scanning enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ecr-private-image-scanning-enabled.html
 */
ManagedRuleIdentifiers.ECR_PRIVATE_IMAGE_SCANNING_ENABLED = 'ECR_PRIVATE_IMAGE_SCANNING_ENABLED';
/**
 * Checks if a private Amazon Elastic Container Registry (ECR) repository has at least one lifecycle policy configured.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ecr-private-lifecycle-policy-configured.html
 */
ManagedRuleIdentifiers.ECR_PRIVATE_LIFECYCLE_POLICY_CONFIGURED = 'ECR_PRIVATE_LIFECYCLE_POLICY_CONFIGURED';
/**
 * Checks if a private Amazon Elastic Container Registry (ECR) repository has tag immutability enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ecr-private-tag-immutability-enabled.html
 */
ManagedRuleIdentifiers.ECR_PRIVATE_TAG_IMMUTABILITY_ENABLED = 'ECR_PRIVATE_TAG_IMMUTABILITY_ENABLED';
/**
 * Checks if the networking mode for active ECSTaskDefinitions is set to ‘awsvpc’.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ecs-awsvpc-networking-enabled.html
 */
ManagedRuleIdentifiers.ECS_AWSVPC_NETWORKING_ENABLED = 'ECS_AWSVPC_NETWORKING_ENABLED';
/**
 * Checks if the privileged parameter in the container definition of ECSTaskDefinitions is set to ‘true’.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instances-in-vpc.html
 */
ManagedRuleIdentifiers.ECS_CONTAINERS_NONPRIVILEGED = 'ECS_CONTAINERS_NONPRIVILEGED';
/**
 * Checks if Amazon Elastic Container Service (Amazon ECS) Containers only have read-only access to its root filesystems.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ecs-containers-readonly-access.html
 */
ManagedRuleIdentifiers.ECS_CONTAINERS_READONLY_ACCESS = 'ECS_CONTAINERS_READONLY_ACCESS';
/**
 * Checks if Amazon Elastic Container Service clusters have container insights enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ecs-container-insights-enabled.html
 */
ManagedRuleIdentifiers.ECS_CONTAINER_INSIGHTS_ENABLED = 'ECS_CONTAINER_INSIGHTS_ENABLED';
/**
 * Checks if Amazon Elastic Container Service (ECS) Fargate Services
 * is running on the latest Fargate platform version.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ecs-fargate-latest-platform-version.html
 */
ManagedRuleIdentifiers.ECS_FARGATE_LATEST_PLATFORM_VERSION = 'ECS_FARGATE_LATEST_PLATFORM_VERSION';
/**
 * Checks if secrets are passed as container environment variables.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ecs-no-environment-secrets.html
 */
ManagedRuleIdentifiers.ECS_NO_ENVIRONMENT_SECRETS = 'ECS_NO_ENVIRONMENT_SECRETS';
/**
 * Checks if logConfiguration is set on active ECS Task Definitions.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ecs-task-definition-log-configuration.html
 */
ManagedRuleIdentifiers.ECS_TASK_DEFINITION_LOG_CONFIGURATION = 'ECS_TASK_DEFINITION_LOG_CONFIGURATION';
/**
 * Checks if Amazon Elastic Container Service (ECS) task definitions have a set memory limit for its container definitions.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ecs-task-definition-memory-hard-limit.html
 */
ManagedRuleIdentifiers.ECS_TASK_DEFINITION_MEMORY_HARD_LIMIT = 'ECS_TASK_DEFINITION_MEMORY_HARD_LIMIT';
/**
 * Checks if ECSTaskDefinitions specify a user
 * for Amazon Elastic Container Service (Amazon ECS) EC2 launch type containers to run on.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ecs-task-definition-nonroot-user.html
 */
ManagedRuleIdentifiers.ECS_TASK_DEFINITION_NONROOT_USER = 'ECS_TASK_DEFINITION_NONROOT_USER';
/**
 * Checks if ECSTaskDefinitions are configured to share a host’s process namespace
 * with its Amazon Elastic Container Service (Amazon ECS) containers.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-stopped-instance.html
 */
ManagedRuleIdentifiers.ECS_TASK_DEFINITION_PID_MODE_CHECK = 'ECS_TASK_DEFINITION_PID_MODE_CHECK';
/**
 * Checks if an Amazon Elastic Container Service (Amazon ECS) task definition
 * with host networking mode has 'privileged' or 'user' container definitions.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-volume-inuse-check.html
 */
ManagedRuleIdentifiers.EC2_VOLUME_IECS_TASK_DEFINITION_USER_FOR_HOST_MODE_CHECKNUSE_CHECK = 'ECS_TASK_DEFINITION_USER_FOR_HOST_MODE_CHECK';
/**
 * Checks if Amazon Elastic File System (Amazon EFS) access points are configured to enforce a root directory.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/efs-access-point-enforce-root-directory.html
 */
ManagedRuleIdentifiers.EFS_ACCESS_POINT_ENFORCE_ROOT_DIRECTORY = 'EFS_ACCESS_POINT_ENFORCE_ROOT_DIRECTORY';
/**
 * Checks if Amazon Elastic File System (Amazon EFS) access points are configured to enforce a user identity.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-volume-inuse-check.html
 */
ManagedRuleIdentifiers.EFS_ACCESS_POINT_ENFORCE_USER_IDENTITY = 'EFS_ACCESS_POINT_ENFORCE_USER_IDENTITY';
/**
 * hecks whether Amazon Elastic File System (Amazon EFS) is configured to encrypt the file data
 * using AWS Key Management Service (AWS KMS).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/efs-encrypted-check.html
 */
ManagedRuleIdentifiers.EFS_ENCRYPTED_CHECK = 'EFS_ENCRYPTED_CHECK';
/**
 * Checks if a recovery point was created for Amazon Elastic File System (Amazon EFS) File Systems.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/efs-last-backup-recovery-point-created.html
 */
ManagedRuleIdentifiers.EFS_LAST_BACKUP_RECOVERY_POINT_CREATED = 'EFS_LAST_BACKUP_RECOVERY_POINT_CREATED';
/**
 * Checks if Amazon Elastic File System (Amazon EFS) File Systems are protected by a backup plan.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/efs-resources-protected-by-backup-plan.html
 */
ManagedRuleIdentifiers.EFS_RESOURCES_PROTECTED_BY_BACKUP_PLAN = 'EFS_RESOURCES_PROTECTED_BY_BACKUP_PLAN';
/**
 * Checks whether all Elastic IP addresses that are allocated to a VPC are attached to
 * EC2 instances or in-use elastic network interfaces (ENIs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/eip-attached.html
 */
ManagedRuleIdentifiers.EIP_ATTACHED = 'EIP_ATTACHED';
/**
 * Checks whether Amazon Elasticsearch Service (Amazon ES) domains have encryption
 * at rest configuration enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-encrypted-at-rest.html
 */
ManagedRuleIdentifiers.ELASTICSEARCH_ENCRYPTED_AT_REST = 'ELASTICSEARCH_ENCRYPTED_AT_REST';
/**
 * Checks whether Amazon Elasticsearch Service (Amazon ES) domains are in
 * Amazon Virtual Private Cloud (Amazon VPC).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-in-vpc-only.html
 */
ManagedRuleIdentifiers.ELASTICSEARCH_IN_VPC_ONLY = 'ELASTICSEARCH_IN_VPC_ONLY';
/**
 * Check if the Amazon ElastiCache Redis clusters have automatic backup turned on.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticache-redis-cluster-automatic-backup-check.html
 */
ManagedRuleIdentifiers.ELASTICACHE_REDIS_CLUSTER_AUTOMATIC_BACKUP_CHECK = 'ELASTICACHE_REDIS_CLUSTER_AUTOMATIC_BACKUP_CHECK';
/**
 * Checks whether your Amazon Elastic Compute Cloud (Amazon EC2) instance metadata version
 * is configured with Instance Metadata Service Version 2 (IMDSv2).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-imdsv2-check.html
 */
ManagedRuleIdentifiers.EC2_IMDSV2_CHECK = 'EC2_IMDSV2_CHECK';
/**
 * Checks if an Amazon Elastic Kubernetes Service (EKS) cluster is running the oldest supported version.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/eks-cluster-oldest-supported-version.html
 */
ManagedRuleIdentifiers.EKS_CLUSTER_OLDEST_SUPPORTED_VERSION = 'EKS_CLUSTER_OLDEST_SUPPORTED_VERSION';
/**
  * Checks if an Amazon Elastic Kubernetes Service (EKS) cluster is running a supported Kubernetes version.
  * @see https://docs.aws.amazon.com/config/latest/developerguide/eks-cluster-supported-version.html
  */
ManagedRuleIdentifiers.EKS_CLUSTER_SUPPORTED_VERSION = 'EKS_CLUSTER_SUPPORTED_VERSION';
/**
 * Checks whether Amazon Elastic Kubernetes Service (Amazon EKS) endpoint is not publicly accessible.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/eks-endpoint-no-public-access.html
 */
ManagedRuleIdentifiers.EKS_ENDPOINT_NO_PUBLIC_ACCESS = 'EKS_ENDPOINT_NO_PUBLIC_ACCESS';
/**
 * Checks whether Amazon Elastic Kubernetes Service clusters are configured to have Kubernetes
 * secrets encrypted using AWS Key Management Service (KMS) keys.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/eks-secrets-encrypted.html
 */
ManagedRuleIdentifiers.EKS_SECRETS_ENCRYPTED = 'EKS_SECRETS_ENCRYPTED';
/**
 * Check that Amazon ElasticSearch Service nodes are encrypted end to end.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-node-to-node-encryption-check.html
 */
ManagedRuleIdentifiers.ELASTICSEARCH_NODE_TO_NODE_ENCRYPTION_CHECK = 'ELASTICSEARCH_NODE_TO_NODE_ENCRYPTION_CHECK';
/**
 * Checks if managed platform updates in an AWS Elastic Beanstalk environment is enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elastic-beanstalk-managed-updates-enabled.html
 */
ManagedRuleIdentifiers.ELASTIC_BEANSTALK_MANAGED_UPDATES_ENABLED = 'ELASTIC_BEANSTALK_MANAGED_UPDATES_ENABLED';
/**
 * Checks if Application Load Balancers and Network Load Balancers
 * have listeners that are configured to use certificates from AWS Certificate Manager (ACM).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elbv2-acm-certificate-required.html
 */
ManagedRuleIdentifiers.ELBV2_ACM_CERTIFICATE_REQUIRED = 'ELBV2_ACM_CERTIFICATE_REQUIRED';
/**
 * Checks if an Elastic Load Balancer V2 (Application, Network, or Gateway Load Balancer)
 * has registered instances from multiple Availability Zones (AZ's).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elbv2-multiple-az.html
 */
ManagedRuleIdentifiers.ELBV2_MULTIPLE_AZ = 'ELBV2_MULTIPLE_AZ';
/**
 * Checks if cross-zone load balancing is enabled for the Classic Load Balancers (CLBs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-cross-zone-load-balancing-enabled.html
 */
ManagedRuleIdentifiers.ELB_CROSS_ZONE_LOAD_BALANCING_ENABLED = 'ELB_CROSS_ZONE_LOAD_BALANCING_ENABLED';
/**
 * Checks whether your Classic Load Balancer is configured with SSL or HTTPS listeners.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-tls-https-listeners-only.html
 */
ManagedRuleIdentifiers.ELB_TLS_HTTPS_LISTENERS_ONLY = 'ELB_TLS_HTTPS_LISTENERS_ONLY';
/**
 * Checks whether the Classic Load Balancers use SSL certificates provided by AWS Certificate Manager.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-acm-certificate-required.html
 */
ManagedRuleIdentifiers.ELB_ACM_CERTIFICATE_REQUIRED = 'ELB_ACM_CERTIFICATE_REQUIRED';
/**
 * Checks whether your Classic Load Balancer SSL listeners are using a custom policy.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-custom-security-policy-ssl-check.html
 */
ManagedRuleIdentifiers.ELB_CUSTOM_SECURITY_POLICY_SSL_CHECK = 'ELB_CUSTOM_SECURITY_POLICY_SSL_CHECK';
/**
 * Checks whether Elastic Load Balancing has deletion protection enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-deletion-protection-enabled.html
 */
ManagedRuleIdentifiers.ELB_DELETION_PROTECTION_ENABLED = 'ELB_DELETION_PROTECTION_ENABLED';
/**
 * Checks whether the Application Load Balancer and the Classic Load Balancer have logging enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-logging-enabled.html
 */
ManagedRuleIdentifiers.ELB_LOGGING_ENABLED = 'ELB_LOGGING_ENABLED';
/**
 * Checks whether your Classic Load Balancer SSL listeners are using a predefined policy.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-predefined-security-policy-ssl-check.html
 */
ManagedRuleIdentifiers.ELB_PREDEFINED_SECURITY_POLICY_SSL_CHECK = 'ELB_PREDEFINED_SECURITY_POLICY_SSL_CHECK';
/**
 * Checks that Amazon EMR clusters have Kerberos enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/emr-kerberos-enabled.html
 */
ManagedRuleIdentifiers.EMR_KERBEROS_ENABLED = 'EMR_KERBEROS_ENABLED';
/**
 * Checks whether Amazon Elastic MapReduce (EMR) clusters' master nodes have public IPs.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/emr-master-no-public-ip.html
 */
ManagedRuleIdentifiers.EMR_MASTER_NO_PUBLIC_IP = 'EMR_MASTER_NO_PUBLIC_IP';
/**
 * Checks whether the EBS volumes that are in an attached state are encrypted.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/encrypted-volumes.html
 */
ManagedRuleIdentifiers.EBS_ENCRYPTED_VOLUMES = 'ENCRYPTED_VOLUMES';
/**
 * Checks whether the security groups associated inScope resources are compliant with the
 * master security groups at each rule level based on allowSecurityGroup and denySecurityGroup flag.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-security-group-audit-policy-check.html
 *
 * @deprecated Inactive managed rule
 *
 */
ManagedRuleIdentifiers.FMS_SECURITY_GROUP_AUDIT_POLICY_CHECK = 'FMS_SECURITY_GROUP_AUDIT_POLICY_CHECK';
/**
 * Checks whether AWS Firewall Manager created security groups content is the same as the master security groups.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-security-group-content-check.html
 *
 * @deprecated Inactive managed rule
 *
 */
ManagedRuleIdentifiers.FMS_SECURITY_GROUP_CONTENT_CHECK = 'FMS_SECURITY_GROUP_CONTENT_CHECK';
/**
 * Checks whether Amazon EC2 or an elastic network interface is associated with AWS Firewall Manager security groups.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-security-group-resource-association-check.html
 *
 * @deprecated Inactive managed rule
 *
 */
ManagedRuleIdentifiers.FMS_SECURITY_GROUP_RESOURCE_ASSOCIATION_CHECK = 'FMS_SECURITY_GROUP_RESOURCE_ASSOCIATION_CHECK';
/**
 * Checks whether an Application Load Balancer, Amazon CloudFront distributions,
 * Elastic Load Balancer or Elastic IP has AWS Shield protection.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-shield-resource-policy-check.html
 */
ManagedRuleIdentifiers.FMS_SHIELD_RESOURCE_POLICY_CHECK = 'FMS_SHIELD_RESOURCE_POLICY_CHECK';
/**
 * Checks whether the web ACL is associated with an Application Load Balancer, API Gateway stage,
 * or Amazon CloudFront distributions.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-webacl-resource-policy-check.html
 */
ManagedRuleIdentifiers.FMS_WEBACL_RESOURCE_POLICY_CHECK = 'FMS_WEBACL_RESOURCE_POLICY_CHECK';
/**
 * Checks that the rule groups associate with the web ACL at the correct priority.
 * The correct priority is decided by the rank of the rule groups in the ruleGroups parameter.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-webacl-rulegroup-association-check.html
 */
ManagedRuleIdentifiers.FMS_WEBACL_RULEGROUP_ASSOCIATION_CHECK = 'FMS_WEBACL_RULEGROUP_ASSOCIATION_CHECK';
/**
 * Checks if a recovery point was created for Amazon FSx File Systems.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fsx-last-backup-recovery-point-created.html
 */
ManagedRuleIdentifiers.FSX_LAST_BACKUP_RECOVERY_POINT_CREATED = 'FSX_LAST_BACKUP_RECOVERY_POINT_CREATED';
/**
 * Checks if Amazon FSx File Systems are protected by a backup plan.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fsx-resources-protected-by-backup-plan.html
 */
ManagedRuleIdentifiers.FSX_RESOURCES_PROTECTED_BY_BACKUP_PLAN = 'FSX_RESOURCES_PROTECTED_BY_BACKUP_PLAN';
/**
 * Checks whether Amazon GuardDuty is enabled in your AWS account and region. If you provide an AWS account for centralization,
 * the rule evaluates the Amazon GuardDuty results in the centralized account.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/guardduty-enabled-centralized.html
 */
ManagedRuleIdentifiers.GUARDDUTY_ENABLED_CENTRALIZED = 'GUARDDUTY_ENABLED_CENTRALIZED';
/**
 * Checks whether the Amazon GuardDuty has findings that are non archived.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/guardduty-non-archived-findings.html
 */
ManagedRuleIdentifiers.GUARDDUTY_NON_ARCHIVED_FINDINGS = 'GUARDDUTY_NON_ARCHIVED_FINDINGS';
/**
 * Checks that inline policy feature is not in use.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-no-inline-policy-check.html
 */
ManagedRuleIdentifiers.IAM_NO_INLINE_POLICY_CHECK = 'IAM_NO_INLINE_POLICY_CHECK';
/**
 * Checks whether IAM groups have at least one IAM user.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-group-has-users-check.html
 */
ManagedRuleIdentifiers.IAM_GROUP_HAS_USERS_CHECK = 'IAM_GROUP_HAS_USERS_CHECK';
/**
 * Checks whether the account password policy for IAM users meets the specified requirements
 * indicated in the parameters.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-password-policy.html
 */
ManagedRuleIdentifiers.IAM_PASSWORD_POLICY = 'IAM_PASSWORD_POLICY';
/**
 * Checks whether for each IAM resource, a policy ARN in the input parameter is attached to the IAM resource.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-blacklisted-check.html
 */
ManagedRuleIdentifiers.IAM_POLICY_BLOCKED_CHECK = 'IAM_POLICY_BLACKLISTED_CHECK';
/**
 * Checks whether the IAM policy ARN is attached to an IAM user, or an IAM group with one or more IAM users,
 * or an IAM role with one or more trusted entity.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-in-use.html
 */
ManagedRuleIdentifiers.IAM_POLICY_IN_USE = 'IAM_POLICY_IN_USE';
/**
 * Checks the IAM policies that you create for Allow statements that grant permissions to all actions on all resources.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-no-statements-with-admin-access.html
 */
ManagedRuleIdentifiers.IAM_POLICY_NO_STATEMENTS_WITH_ADMIN_ACCESS = 'IAM_POLICY_NO_STATEMENTS_WITH_ADMIN_ACCESS';
/**
 * Checks if AWS Identity and Access Management (IAM) policies that you create grant permissions to all actions on individual AWS resources.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-no-statements-with-full-access.html
 */
ManagedRuleIdentifiers.IAM_POLICY_NO_STATEMENTS_WITH_FULL_ACCESS = 'IAM_POLICY_NO_STATEMENTS_WITH_FULL_ACCESS';
/**
 * Checks that AWS Identity and Access Management (IAM) policies in a list of policies are attached to all AWS roles.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-role-managed-policy-check.html
 */
ManagedRuleIdentifiers.IAM_ROLE_MANAGED_POLICY_CHECK = 'IAM_ROLE_MANAGED_POLICY_CHECK';
/**
 * Checks whether the root user access key is available.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-root-access-key-check.html
 */
ManagedRuleIdentifiers.IAM_ROOT_ACCESS_KEY_CHECK = 'IAM_ROOT_ACCESS_KEY_CHECK';
/**
 * Checks whether IAM users are members of at least one IAM group.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-group-membership-check.html
 */
ManagedRuleIdentifiers.IAM_USER_GROUP_MEMBERSHIP_CHECK = 'IAM_USER_GROUP_MEMBERSHIP_CHECK';
/**
 * Checks whether the AWS Identity and Access Management users have multi-factor authentication (MFA) enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-mfa-enabled.html
 */
ManagedRuleIdentifiers.IAM_USER_MFA_ENABLED = 'IAM_USER_MFA_ENABLED';
/**
 * Checks that none of your IAM users have policies attached. IAM users must inherit permissions from IAM groups or roles.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-no-policies-check.html
 */
ManagedRuleIdentifiers.IAM_USER_NO_POLICIES_CHECK = 'IAM_USER_NO_POLICIES_CHECK';
/**
 * Checks whether your AWS Identity and Access Management (IAM) users have passwords or
 * active access keys that have not been used within the specified number of days you provided.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-unused-credentials-check.html
 */
ManagedRuleIdentifiers.IAM_USER_UNUSED_CREDENTIALS_CHECK = 'IAM_USER_UNUSED_CREDENTIALS_CHECK';
/**
 * Checks that Internet gateways (IGWs) are only attached to an authorized Amazon Virtual Private Cloud (VPCs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/internet-gateway-authorized-vpc-only.html
 */
ManagedRuleIdentifiers.INTERNET_GATEWAY_AUTHORIZED_VPC_ONLY = 'INTERNET_GATEWAY_AUTHORIZED_VPC_ONLY';
/**
 * Checks if Amazon Kinesis streams are encrypted at rest with server-side encryption.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/kinesis-stream-encrypted.html
 */
ManagedRuleIdentifiers.KINESIS_STREAM_ENCRYPTED = 'KINESIS_STREAM_ENCRYPTED';
/**
 * Checks whether customer master keys (CMKs) are not scheduled for deletion in AWS Key Management Service (KMS).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/kms-cmk-not-scheduled-for-deletion.html
 */
ManagedRuleIdentifiers.KMS_CMK_NOT_SCHEDULED_FOR_DELETION = 'KMS_CMK_NOT_SCHEDULED_FOR_DELETION';
/**
 * Checks whether the AWS Lambda function is configured with function-level concurrent execution limit.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-concurrency-check.html
 */
ManagedRuleIdentifiers.LAMBDA_CONCURRENCY_CHECK = 'LAMBDA_CONCURRENCY_CHECK';
/**
 * Checks whether an AWS Lambda function is configured with a dead-letter queue.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-dlq-check.html
 */
ManagedRuleIdentifiers.LAMBDA_DLQ_CHECK = 'LAMBDA_DLQ_CHECK';
/**
 * Checks whether the AWS Lambda function policy attached to the Lambda resource prohibits public access.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-function-public-access-prohibited.html
 */
ManagedRuleIdentifiers.LAMBDA_FUNCTION_PUBLIC_ACCESS_PROHIBITED = 'LAMBDA_FUNCTION_PUBLIC_ACCESS_PROHIBITED';
/**
 * Checks that the lambda function settings for runtime, role, timeout, and memory size match the expected values.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-function-settings-check.html
 */
ManagedRuleIdentifiers.LAMBDA_FUNCTION_SETTINGS_CHECK = 'LAMBDA_FUNCTION_SETTINGS_CHECK';
/**
 * Checks whether an AWS Lambda function is in an Amazon Virtual Private Cloud.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-inside-vpc.html
 */
ManagedRuleIdentifiers.LAMBDA_INSIDE_VPC = 'LAMBDA_INSIDE_VPC';
/**
 * Checks if Lambda has more than 1 availability zone associated.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-vpc-multi-az-check.html
 */
ManagedRuleIdentifiers.LAMBDA_VPC_MULTI_AZ_CHECK = 'LAMBDA_VPC_MULTI_AZ_CHECK';
/**
 * Checks whether AWS Multi-Factor Authentication (MFA) is enabled for all IAM users that use a console password.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/mfa-enabled-for-iam-console-access.html
 */
ManagedRuleIdentifiers.MFA_ENABLED_FOR_IAM_CONSOLE_ACCESS = 'MFA_ENABLED_FOR_IAM_CONSOLE_ACCESS';
/**
 * Checks that there is at least one multi-region AWS CloudTrail.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/multi-region-cloudtrail-enabled.html
 */
ManagedRuleIdentifiers.CLOUDTRAIL_MULTI_REGION_ENABLED = 'MULTI_REGION_CLOUD_TRAIL_ENABLED';
/**
 * Checks if default ports for SSH/RDP ingress traffic for network access control lists (NACLs) is unrestricted.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/nacl-no-unrestricted-ssh-rdp.html
 */
ManagedRuleIdentifiers.NACL_NO_UNRESTRICTED_SSH_RDP = 'NACL_NO_UNRESTRICTED_SSH_RDP';
/**
 * Checks if an AWS Network Firewall policy is configured with a user defined stateless default action for fragmented packets.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/netfw-policy-default-action-fragment-packets.html
 */
ManagedRuleIdentifiers.NETFW_POLICY_DEFAULT_ACTION_FRAGMENT_PACKETS = 'NETFW_POLICY_DEFAULT_ACTION_FRAGMENT_PACKETS';
/**
 * Checks if an AWS Network Firewall policy is configured with a user defined default stateless action for full packets.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/netfw-policy-default-action-full-packets.html
 */
ManagedRuleIdentifiers.NETFW_POLICY_DEFAULT_ACTION_FULL_PACKETS = 'NETFW_POLICY_DEFAULT_ACTION_FULL_PACKETS';
/**
 * Check AWS Network Firewall policy is associated with stateful OR stateless rule groups.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/netfw-policy-rule-group-associated.html
 */
ManagedRuleIdentifiers.NETFW_POLICY_RULE_GROUP_ASSOCIATED = 'NETFW_POLICY_RULE_GROUP_ASSOCIATED';
/**
 * Checks if a Stateless Network Firewall Rule Group contains rules.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/netfw-stateless-rule-group-not-empty.html
 */
ManagedRuleIdentifiers.NETFW_STATELESS_RULE_GROUP_NOT_EMPTY = 'NETFW_STATELESS_RULE_GROUP_NOT_EMPTY';
/**
 * Checks if cross-zone load balancing is enabled on Network Load Balancers (NLBs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/nlb-cross-zone-load-balancing-enabled.html
 */
ManagedRuleIdentifiers.NLB_CROSS_ZONE_LOAD_BALANCING_ENABLED = 'NLB_CROSS_ZONE_LOAD_BALANCING_ENABLED';
/**
 * Checks if Amazon OpenSearch Service domains have fine-grained access control enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/opensearch-access-control-enabled.html
 */
ManagedRuleIdentifiers.OPENSEARCH_ACCESS_CONTROL_ENABLED = 'OPENSEARCH_ACCESS_CONTROL_ENABLED';
/**
 * Checks if Amazon OpenSearch Service domains have audit logging enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/opensearch-audit-logging-enabled.html
 */
ManagedRuleIdentifiers.OPENSEARCH_AUDIT_LOGGING_ENABLED = 'OPENSEARCH_AUDIT_LOGGING_ENABLED';
/**
 * Checks if Amazon OpenSearch Service domains are configured with at least three data nodes and zoneAwarenessEnabled is true.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/opensearch-data-node-fault-tolerance.html
 */
ManagedRuleIdentifiers.OPENSEARCH_DATA_NODE_FAULT_TOLERANCE = 'OPENSEARCH_DATA_NODE_FAULT_TOLERANCE';
/**
 * Checks if Amazon OpenSearch Service domains have encryption at rest configuration enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/opensearch-encrypted-at-rest.html
 */
ManagedRuleIdentifiers.OPENSEARCH_ENCRYPTED_AT_REST = 'OPENSEARCH_ENCRYPTED_AT_REST';
/**
 * Checks whether connections to OpenSearch domains are using HTTPS.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/opensearch-https-required.html
 */
ManagedRuleIdentifiers.OPENSEARCH_HTTPS_REQUIRED = 'OPENSEARCH_HTTPS_REQUIRED';
/**
 * Checks if Amazon OpenSearch Service domains are in an Amazon Virtual Private Cloud (VPC).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/opensearch-in-vpc-only.html
 */
ManagedRuleIdentifiers.OPENSEARCH_IN_VPC_ONLY = 'OPENSEARCH_IN_VPC_ONLY';
/**
 * Checks if Amazon OpenSearch Service domains are configured to send logs to Amazon CloudWatch Logs.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/opensearch-logs-to-cloudwatch.html
 */
ManagedRuleIdentifiers.OPENSEARCH_LOGS_TO_CLOUDWATCH = 'OPENSEARCH_LOGS_TO_CLOUDWATCH';
/**
 * Check if Amazon OpenSearch Service nodes are encrypted end to end.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/opensearch-node-to-node-encryption-check.html
 */
ManagedRuleIdentifiers.OPENSEARCH_NODE_TO_NODE_ENCRYPTION_CHECK = 'OPENSEARCH_NODE_TO_NODE_ENCRYPTION_CHECK';
/**
 * Checks if Amazon Relational Database Service (RDS) database instances are configured for automatic minor version upgrades.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-automatic-minor-version-upgrade-enabled.html
 */
ManagedRuleIdentifiers.RDS_AUTOMATIC_MINOR_VERSION_UPGRADE_ENABLED = 'RDS_AUTOMATIC_MINOR_VERSION_UPGRADE_ENABLED';
/**
 * Checks if an Amazon Relational Database Service (Amazon RDS) database cluster has changed the admin username from its default value.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-cluster-default-admin-check.html
 */
ManagedRuleIdentifiers.RDS_CLUSTER_DEFAULT_ADMIN_CHECK = 'RDS_CLUSTER_DEFAULT_ADMIN_CHECK';
/**
 * Checks if an Amazon Relational Database Service (Amazon RDS) cluster has deletion protection enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-cluster-deletion-protection-enabled.html
 */
ManagedRuleIdentifiers.RDS_CLUSTER_DELETION_PROTECTION_ENABLED = 'RDS_CLUSTER_DELETION_PROTECTION_ENABLED';
/**
 * Checks if an Amazon RDS Cluster has AWS Identity and Access Management (IAM) authentication enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-cluster-iam-authentication-enabled.html
 */
ManagedRuleIdentifiers.RDS_CLUSTER_IAM_AUTHENTICATION_ENABLED = 'RDS_CLUSTER_IAM_AUTHENTICATION_ENABLED';
/**
 * Checks if Multi-AZ replication is enabled on Amazon Aurora and Hermes clusters managed by Amazon Relational Database Service (Amazon RDS).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-cluster-multi-az-enabled.html
 */
ManagedRuleIdentifiers.RDS_CLUSTER_MULTI_AZ_ENABLED = 'RDS_CLUSTER_MULTI_AZ_ENABLED';
/**
 * Checks if an Amazon Relational Database Service (Amazon RDS) database has changed the admin username from its default value.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-default-admin-check.html
 */
ManagedRuleIdentifiers.RDS_INSTANCE_DEFAULT_ADMIN_CHECK = 'RDS_INSTANCE_DEFAULT_ADMIN_CHECK';
/**
 *Checks if there are any Amazon Relational Database Service (RDS) DB security groups that are not the default DB security group.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-db-security-group-not-allowed.html
 */
ManagedRuleIdentifiers.RDS_DB_SECURITY_GROUP_NOT_ALLOWED = 'RDS_DB_SECURITY_GROUP_NOT_ALLOWED';
/**
 * Checks if an Amazon Relational Database Service (Amazon RDS) instance has deletion protection enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-deletion-protection-enabled.html
 */
ManagedRuleIdentifiers.RDS_INSTANCE_DELETION_PROTECTION_ENABLED = 'RDS_INSTANCE_DELETION_PROTECTION_ENABLED';
/**
 * Checks if an Amazon RDS instance has AWS Identity and Access Management (IAM) authentication enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-iam-authentication-enabled.html
 */
ManagedRuleIdentifiers.RDS_INSTANCE_IAM_AUTHENTICATION_ENABLED = 'RDS_INSTANCE_IAM_AUTHENTICATION_ENABLED';
/**
 * Checks that respective logs of Amazon Relational Database Service (Amazon RDS) are enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-logging-enabled.html
 */
ManagedRuleIdentifiers.RDS_LOGGING_ENABLED = 'RDS_LOGGING_ENABLED';
/**
 * Checks that Amazon Redshift automated snapshots are enabled for clusters.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-backup-enabled.html
 */
ManagedRuleIdentifiers.REDSHIFT_BACKUP_ENABLED = 'REDSHIFT_BACKUP_ENABLED';
/**
 * Checks whether enhanced monitoring is enabled for Amazon Relational Database Service (Amazon RDS) instances.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-enhanced-monitoring-enabled.html
 */
ManagedRuleIdentifiers.RDS_ENHANCED_MONITORING_ENABLED = 'RDS_ENHANCED_MONITORING_ENABLED';
/**
 * Checks whether Amazon Relational Database Service (Amazon RDS) DB snapshots are encrypted.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshot-encrypted.html
 */
ManagedRuleIdentifiers.RDS_SNAPSHOT_ENCRYPTED = 'RDS_SNAPSHOT_ENCRYPTED';
/**
 * Checks whether Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-require-tls-ssl.html
 */
ManagedRuleIdentifiers.REDSHIFT_REQUIRE_TLS_SSL = 'REDSHIFT_REQUIRE_TLS_SSL';
/**
 * Checks whether Amazon RDS database is present in back plans of AWS Backup.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-in-backup-plan.html
 */
ManagedRuleIdentifiers.RDS_IN_BACKUP_PLAN = 'RDS_IN_BACKUP_PLAN';
/**
 * Checks if a recovery point was created for Amazon Relational Database Service (Amazon RDS).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-last-backup-recovery-point-created.html
 */
ManagedRuleIdentifiers.RDS_LAST_BACKUP_RECOVERY_POINT_CREATED = 'RDS_LAST_BACKUP_RECOVERY_POINT_CREATED';
/**
 * Check whether the Amazon Relational Database Service instances are not publicly accessible.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-public-access-check.html
 */
ManagedRuleIdentifiers.RDS_INSTANCE_PUBLIC_ACCESS_CHECK = 'RDS_INSTANCE_PUBLIC_ACCESS_CHECK';
/**
 * Checks whether high availability is enabled for your RDS DB instances.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-multi-az-support.html
 */
ManagedRuleIdentifiers.RDS_MULTI_AZ_SUPPORT = 'RDS_MULTI_AZ_SUPPORT';
/**
 * Checks if Amazon Relational Database Service (Amazon RDS) instances are protected by a backup plan.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-resources-protected-by-backup-plan.html
 */
ManagedRuleIdentifiers.RDS_RESOURCES_PROTECTED_BY_BACKUP_PLAN = 'RDS_RESOURCES_PROTECTED_BY_BACKUP_PLAN';
/**
 * Checks if Amazon Relational Database Service (Amazon RDS) snapshots are public.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshots-public-prohibited.html
 */
ManagedRuleIdentifiers.RDS_SNAPSHOTS_PUBLIC_PROHIBITED = 'RDS_SNAPSHOTS_PUBLIC_PROHIBITED';
/**
 * Checks whether storage encryption is enabled for your RDS DB instances.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-storage-encrypted.html
 */
ManagedRuleIdentifiers.RDS_STORAGE_ENCRYPTED = 'RDS_STORAGE_ENCRYPTED';
/**
 * Checks if Amazon Redshift clusters are logging audits to a specific bucket.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-audit-logging-enabled.html
 */
ManagedRuleIdentifiers.REDSHIFT_AUDIT_LOGGING_ENABLED = 'REDSHIFT_AUDIT_LOGGING_ENABLED';
/**
 * Checks whether Amazon Redshift clusters have the specified settings.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-configuration-check.html
 */
ManagedRuleIdentifiers.REDSHIFT_CLUSTER_CONFIGURATION_CHECK = 'REDSHIFT_CLUSTER_CONFIGURATION_CHECK';
/**
 * Checks if Amazon Redshift clusters are using a specified AWS Key Management Service (AWS KMS) key for encryption.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-kms-enabled.html
 */
ManagedRuleIdentifiers.REDSHIFT_CLUSTER_KMS_ENABLED = 'REDSHIFT_CLUSTER_KMS_ENABLED';
/**
 * Checks whether Amazon Redshift clusters have the specified maintenance settings.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-maintenancesettings-check.html
 */
ManagedRuleIdentifiers.REDSHIFT_CLUSTER_MAINTENANCE_SETTINGS_CHECK = 'REDSHIFT_CLUSTER_MAINTENANCESETTINGS_CHECK';
/**
 * Checks whether Amazon Redshift clusters are not publicly accessible.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-public-access-check.html
 */
ManagedRuleIdentifiers.REDSHIFT_CLUSTER_PUBLIC_ACCESS_CHECK = 'REDSHIFT_CLUSTER_PUBLIC_ACCESS_CHECK';
/**
 * Checks if an Amazon Redshift cluster has changed the admin username from its default value.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-default-admin-check.html
 */
ManagedRuleIdentifiers.REDSHIFT_DEFAULT_ADMIN_CHECK = 'REDSHIFT_DEFAULT_ADMIN_CHECK';
/**
 * Checks if a Redshift cluster has changed its database name from the default value.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-default-db-name-check.html
 */
ManagedRuleIdentifiers.REDSHIFT_DEFAULT_DB_NAME_CHECK = 'REDSHIFT_DEFAULT_DB_NAME_CHECK';
/**
 * Checks if Amazon Redshift cluster has 'enhancedVpcRouting' enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-enhanced-vpc-routing-enabled.html
 */
ManagedRuleIdentifiers.REDSHIFT_ENHANCED_VPC_ROUTING_ENABLED = 'REDSHIFT_ENHANCED_VPC_ROUTING_ENABLED';
/**
 * Checks whether your resources have the tags that you specify.
 * For example, you can check whether your Amazon EC2 instances have the CostCenter tag.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/required-tags.html
 */
ManagedRuleIdentifiers.REQUIRED_TAGS = 'REQUIRED_TAGS';
/**
 * Checks whether the security groups in use do not allow unrestricted incoming TCP traffic to the specified ports.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/restricted-common-ports.html
 */
ManagedRuleIdentifiers.EC2_SECURITY_GROUPS_RESTRICTED_INCOMING_TRAFFIC = 'RESTRICTED_INCOMING_TRAFFIC';
/**
 * Checks whether the incoming SSH traffic for the security groups is accessible.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/restricted-ssh.html
 */
ManagedRuleIdentifiers.EC2_SECURITY_GROUPS_INCOMING_SSH_DISABLED = 'INCOMING_SSH_DISABLED';
/**
 * Checks whether your AWS account is enabled to use multi-factor authentication (MFA) hardware
 * device to sign in with root credentials.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/root-account-hardware-mfa-enabled.html
 */
ManagedRuleIdentifiers.ROOT_ACCOUNT_HARDWARE_MFA_ENABLED = 'ROOT_ACCOUNT_HARDWARE_MFA_ENABLED';
/**
 * Checks whether users of your AWS account require a multi-factor authentication (MFA) device
 * to sign in with root credentials.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/root-account-mfa-enabled.html
 */
ManagedRuleIdentifiers.ROOT_ACCOUNT_MFA_ENABLED = 'ROOT_ACCOUNT_MFA_ENABLED';
/**
 * Checks whether Amazon Simple Storage Service (Amazon S3) bucket has lock enabled, by default.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-default-lock-enabled.html
 */
ManagedRuleIdentifiers.S3_BUCKET_DEFAULT_LOCK_ENABLED = 'S3_BUCKET_DEFAULT_LOCK_ENABLED';
/**
 * Checks whether the Amazon Simple Storage Service (Amazon S3) buckets are encrypted
 * with AWS Key Management Service (AWS KMS).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-default-encryption-kms.html
 */
ManagedRuleIdentifiers.S3_DEFAULT_ENCRYPTION_KMS = 'S3_DEFAULT_ENCRYPTION_KMS';
/**
 * Checks that AWS Security Hub is enabled for an AWS account.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/securityhub-enabled.html
 */
ManagedRuleIdentifiers.SECURITYHUB_ENABLED = 'SECURITYHUB_ENABLED';
/**
 * Checks whether Amazon SNS topic is encrypted with AWS Key Management Service (AWS KMS).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sns-encrypted-kms.html
 */
ManagedRuleIdentifiers.SNS_ENCRYPTED_KMS = 'SNS_ENCRYPTED_KMS';
/**
 * Checks if Amazon Simple Notification Service (SNS) logging is enabled
 * for the delivery status of notification messages sent to a topic for the endpoints.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sns-topic-message-delivery-notification-enabled.html
 */
ManagedRuleIdentifiers.SNS_TOPIC_MESSAGE_DELIVERY_NOTIFICATION_ENABLED = 'SNS_TOPIC_MESSAGE_DELIVERY_NOTIFICATION_ENABLED';
/**
 * Checks if AWS Systems Manager documents owned by the account are public.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ssm-document-not-public.html
 */
ManagedRuleIdentifiers.SSM_DOCUMENT_NOT_PUBLIC = 'SSM_DOCUMENT_NOT_PUBLIC';
/**
 * Checks if a recovery point was created for AWS Storage Gateway volumes.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/storagegateway-last-backup-recovery-point-created.html
 */
ManagedRuleIdentifiers.STORAGEGATEWAY_LAST_BACKUP_RECOVERY_POINT_CREATED = 'STORAGEGATEWAY_LAST_BACKUP_RECOVERY_POINT_CREATED';
/**
 * hecks if Amazon Virtual Private Cloud (Amazon VPC) subnets are assigned a public IP address.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/subnet-auto-assign-public-ip-disabled.html
 */
ManagedRuleIdentifiers.SUBNET_AUTO_ASSIGN_PUBLIC_IP_DISABLED = 'SUBNET_AUTO_ASSIGN_PUBLIC_IP_DISABLED';
/**
 * Checks whether the required public access block settings are configured from account level.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-account-level-public-access-blocks.html
 */
ManagedRuleIdentifiers.S3_ACCOUNT_LEVEL_PUBLIC_ACCESS_BLOCKS = 'S3_ACCOUNT_LEVEL_PUBLIC_ACCESS_BLOCKS';
/**
 * Checks if the required public access block settings are configured from account level.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-account-level-public-access-blocks-periodic.html
 */
ManagedRuleIdentifiers.S3_ACCOUNT_LEVEL_PUBLIC_ACCESS_BLOCKS_PERIODIC = 'S3_ACCOUNT_LEVEL_PUBLIC_ACCESS_BLOCKS_PERIODIC';
/**
 * Checks if Amazon Simple Storage Service (Amazon S3) Buckets allow user permissions through access control lists (ACLs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-acl-prohibited.html
 */
ManagedRuleIdentifiers.S3_BUCKET_ACL_PROHIBITED = 'S3_BUCKET_ACL_PROHIBITED';
/**
 * Checks if the Amazon Simple Storage Service bucket policy does not allow blacklisted bucket-level
 * and object-level actions on resources in the bucket for principals from other AWS accounts.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-blacklisted-actions-prohibited.html
 */
ManagedRuleIdentifiers.S3_BUCKET_BLOCKED_ACTIONS_PROHIBITED = 'S3_BUCKET_BLACKLISTED_ACTIONS_PROHIBITED';
/**
 * Checks if Amazon Simple Storage Service (Amazon S3) buckets are publicly accessible.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-level-public-access-prohibited.html
 */
ManagedRuleIdentifiers.S3_BUCKET_LEVEL_PUBLIC_ACCESS_PROHIBITED = 'S3_BUCKET_LEVEL_PUBLIC_ACCESS_PROHIBITED';
/**
 * Checks whether logging is enabled for your S3 buckets.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-logging-enabled.html
 */
ManagedRuleIdentifiers.S3_BUCKET_LOGGING_ENABLED = 'S3_BUCKET_LOGGING_ENABLED';
/**
 * Checks that the access granted by the Amazon S3 bucket is restricted by any of the AWS principals, federated users,
 * service principals, IP addresses, or VPCs that you provide.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-policy-grantee-check.html
 */
ManagedRuleIdentifiers.S3_BUCKET_POLICY_GRANTEE_CHECK = 'S3_BUCKET_POLICY_GRANTEE_CHECK';
/**
 * Checks if your Amazon Simple Storage Service bucket policies do not allow other inter-account permissions
 * than the control Amazon S3 bucket policy that you provide.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-policy-not-more-permissive.html
 */
ManagedRuleIdentifiers.S3_BUCKET_POLICY_NOT_MORE_PERMISSIVE = 'S3_BUCKET_POLICY_NOT_MORE_PERMISSIVE';
/**
 * Checks if your Amazon S3 buckets do not allow public read access.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-read-prohibited.html
 */
ManagedRuleIdentifiers.S3_BUCKET_PUBLIC_READ_PROHIBITED = 'S3_BUCKET_PUBLIC_READ_PROHIBITED';
/**
 * Checks that your Amazon S3 buckets do not allow public write access.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-write-prohibited.html
 */
ManagedRuleIdentifiers.S3_BUCKET_PUBLIC_WRITE_PROHIBITED = 'S3_BUCKET_PUBLIC_WRITE_PROHIBITED';
/**
 * Checks whether S3 buckets have cross-region replication enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-replication-enabled.html
 */
ManagedRuleIdentifiers.S3_BUCKET_REPLICATION_ENABLED = 'S3_BUCKET_REPLICATION_ENABLED';
/**
 * Checks that your Amazon S3 bucket either has Amazon S3 default encryption enabled or that the
 * S3 bucket policy explicitly denies put-object requests without server side encryption that
 * uses AES-256 or AWS Key Management Service.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-server-side-encryption-enabled.html
 */
ManagedRuleIdentifiers.S3_BUCKET_SERVER_SIDE_ENCRYPTION_ENABLED = 'S3_BUCKET_SERVER_SIDE_ENCRYPTION_ENABLED';
/**
 * Checks whether S3 buckets have policies that require requests to use Secure Socket Layer (SSL).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-ssl-requests-only.html
 */
ManagedRuleIdentifiers.S3_BUCKET_SSL_REQUESTS_ONLY = 'S3_BUCKET_SSL_REQUESTS_ONLY';
/**
 * Checks whether versioning is enabled for your S3 buckets.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-versioning-enabled.html
 */
ManagedRuleIdentifiers.S3_BUCKET_VERSIONING_ENABLED = 'S3_BUCKET_VERSIONING_ENABLED';
/**
 * Checks if Amazon S3 Events Notifications are enabled on an S3 bucket.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-event-notifications-enabled.html
 */
ManagedRuleIdentifiers.S3_EVENT_NOTIFICATIONS_ENABLED = 'S3_EVENT_NOTIFICATIONS_ENABLED';
/**
 * Checks if a recovery point was created for Amazon Simple Storage Service (Amazon S3).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-last-backup-recovery-point-created.html
 */
ManagedRuleIdentifiers.S3_LAST_BACKUP_RECOVERY_POINT_CREATED = 'S3_LAST_BACKUP_RECOVERY_POINT_CREATED';
/**
 * Checks if a lifecycle rule is configured for an Amazon Simple Storage Service (Amazon S3) bucket.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-lifecycle-policy-check.html
 */
ManagedRuleIdentifiers.S3_LIFECYCLE_POLICY_CHECK = 'S3_LIFECYCLE_POLICY_CHECK';
/**
 * Checks if Amazon Simple Storage Service (Amazon S3) buckets are protected by a backup plan.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-resources-protected-by-backup-plan.html
 */
ManagedRuleIdentifiers.S3_RESOURCES_PROTECTED_BY_BACKUP_PLAN = 'S3_RESOURCES_PROTECTED_BY_BACKUP_PLAN';
/**
 * Checks if Amazon Simple Storage Service (Amazon S3) version enabled buckets have lifecycle policy configured.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-version-lifecycle-policy-check.html
 */
ManagedRuleIdentifiers.S3_VERSION_LIFECYCLE_POLICY_CHECK = 'S3_VERSION_LIFECYCLE_POLICY_CHECK';
/**
 * Checks whether AWS Key Management Service (KMS) key is configured for an Amazon SageMaker endpoint configuration.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-endpoint-configuration-kms-key-configured.html
 */
ManagedRuleIdentifiers.SAGEMAKER_ENDPOINT_CONFIGURATION_KMS_KEY_CONFIGURED = 'SAGEMAKER_ENDPOINT_CONFIGURATION_KMS_KEY_CONFIGURED';
/**
 * Check whether an AWS Key Management Service (KMS) key is configured for SageMaker notebook instance.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-instance-kms-key-configured.html
 */
ManagedRuleIdentifiers.SAGEMAKER_NOTEBOOK_INSTANCE_KMS_KEY_CONFIGURED = 'SAGEMAKER_NOTEBOOK_INSTANCE_KMS_KEY_CONFIGURED';
/**
 * Checks whether direct internet access is disabled for an Amazon SageMaker notebook instance.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-no-direct-internet-access.html
 */
ManagedRuleIdentifiers.SAGEMAKER_NOTEBOOK_NO_DIRECT_INTERNET_ACCESS = 'SAGEMAKER_NOTEBOOK_NO_DIRECT_INTERNET_ACCESS';
/**
 * Checks whether AWS Secrets Manager secret has rotation enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-rotation-enabled-check.html
 */
ManagedRuleIdentifiers.SECRETSMANAGER_ROTATION_ENABLED_CHECK = 'SECRETSMANAGER_ROTATION_ENABLED_CHECK';
/**
 * Checks whether AWS Secrets Manager secret rotation has rotated successfully as per the rotation schedule.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-scheduled-rotation-success-check.html
 */
ManagedRuleIdentifiers.SECRETSMANAGER_SCHEDULED_ROTATION_SUCCESS_CHECK = 'SECRETSMANAGER_SCHEDULED_ROTATION_SUCCESS_CHECK';
/**
 * Checks if AWS Secrets Manager secrets have been rotated in the past specified number of days.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-secret-periodic-rotation.html
 */
ManagedRuleIdentifiers.SECRETSMANAGER_SECRET_PERIODIC_ROTATION = 'SECRETSMANAGER_SECRET_PERIODIC_ROTATION';
/**
 * Checks if AWS Secrets Manager secrets have been accessed within a specified number of days.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-secret-unused.html
 */
ManagedRuleIdentifiers.SECRETSMANAGER_SECRET_UNUSED = 'SECRETSMANAGER_SECRET_UNUSED';
/**
 * Checks if all secrets in AWS Secrets Manager are encrypted using the AWS managed key (aws/secretsmanager)
 * or a customer managed key that was created in AWS Key Management Service (AWS KMS).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-using-cmk.html
 */
ManagedRuleIdentifiers.SECRETSMANAGER_USING_CMK = 'SECRETSMANAGER_USING_CMK';
/**
 * Checks whether Service Endpoint for the service provided in rule parameter is created for each Amazon VPC.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/service-vpc-endpoint-enabled.html
 */
ManagedRuleIdentifiers.SERVICE_VPC_ENDPOINT_ENABLED = 'SERVICE_VPC_ENDPOINT_ENABLED';
/**
 * Checks whether EBS volumes are attached to EC2 instances.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/shield-advanced-enabled-autorenew.html
 */
ManagedRuleIdentifiers.SHIELD_ADVANCED_ENABLED_AUTO_RENEW = 'SHIELD_ADVANCED_ENABLED_AUTORENEW';
/**
 * Verify that DDoS response team (DRT) can access AWS account.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/shield-drt-access.html
 */
ManagedRuleIdentifiers.SHIELD_DRT_ACCESS = 'SHIELD_DRT_ACCESS';
/**
 * Checks if a recovery point was created for AWS Backup-Gateway VirtualMachines.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/virtualmachine-last-backup-recovery-point-created.html
 */
ManagedRuleIdentifiers.VIRTUALMACHINE_LAST_BACKUP_RECOVERY_POINT_CREATED = 'VIRTUALMACHINE_LAST_BACKUP_RECOVERY_POINT_CREATED';
/**
 * Checks if AWS Backup-Gateway VirtualMachines are protected by a backup plan.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/virtualmachine-resources-protected-by-backup-plan.html
 */
ManagedRuleIdentifiers.VIRTUALMACHINE_RESOURCES_PROTECTED_BY_BACKUP_PLAN = 'VIRTUALMACHINE_RESOURCES_PROTECTED_BY_BACKUP_PLAN';
/**
 * Checks that the default security group of any Amazon Virtual Private Cloud (VPC) does not
 * allow inbound or outbound traffic. The rule returns NOT_APPLICABLE if the security group
 * is not default.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-default-security-group-closed.html
 */
ManagedRuleIdentifiers.VPC_DEFAULT_SECURITY_GROUP_CLOSED = 'VPC_DEFAULT_SECURITY_GROUP_CLOSED';
/**
 * Checks whether Amazon Virtual Private Cloud flow logs are found and enabled for Amazon VPC.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-flow-logs-enabled.html
 */
ManagedRuleIdentifiers.VPC_FLOW_LOGS_ENABLED = 'VPC_FLOW_LOGS_ENABLED';
/**
 * Checks if there are unused network access control lists (network ACLs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-network-acl-unused-check.html
 */
ManagedRuleIdentifiers.VPC_NETWORK_ACL_UNUSED_CHECK = 'VPC_NETWORK_ACL_UNUSED_CHECK';
/**
 * Checks if DNS resolution from accepter/requester VPC to private IP is enabled.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-peering-dns-resolution-check.html
 */
ManagedRuleIdentifiers.VPC_PEERING_DNS_RESOLUTION_CHECK = 'VPC_PEERING_DNS_RESOLUTION_CHECK';
/**
 * Checks whether the security group with 0.0.0.0/0 of any Amazon Virtual Private Cloud (Amazon VPC)
 * allows only specific inbound TCP or UDP traffic.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-sg-open-only-to-authorized-ports.html
 */
ManagedRuleIdentifiers.VPC_SG_OPEN_ONLY_TO_AUTHORIZED_PORTS = 'VPC_SG_OPEN_ONLY_TO_AUTHORIZED_PORTS';
/**
 * Checks that both AWS Virtual Private Network tunnels provided by AWS Site-to-Site VPN are in
 * UP status.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-vpn-2-tunnels-up.html
 */
ManagedRuleIdentifiers.VPC_VPN_2_TUNNELS_UP = 'VPC_VPN_2_TUNNELS_UP';
/**
 * Checks if logging is enabled on AWS Web Application Firewall (WAF) classic global web ACLs.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/waf-classic-logging-enabled.html
 */
ManagedRuleIdentifiers.WAF_CLASSIC_LOGGING_ENABLED = 'WAF_CLASSIC_LOGGING_ENABLED';
/**
 * Checks whether logging is enabled on AWS Web Application Firewall (WAFV2) regional and global
 * web access control list (ACLs).
 * @see https://docs.aws.amazon.com/config/latest/developerguide/wafv2-logging-enabled.html
 */
ManagedRuleIdentifiers.WAFV2_LOGGING_ENABLED = 'WAFV2_LOGGING_ENABLED';
/**
 * Checks if an AWS WAF Classic rule group contains any rules.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/waf-global-rulegroup-not-empty.html
 */
ManagedRuleIdentifiers.WAF_GLOBAL_RULEGROUP_NOT_EMPTY = 'WAF_GLOBAL_RULEGROUP_NOT_EMPTY';
/**
 * Checks if an AWS WAF global rule contains any conditions.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/waf-global-rule-not-empty.html
 */
ManagedRuleIdentifiers.WAF_GLOBAL_RULE_NOT_EMPTY = 'WAF_GLOBAL_RULE_NOT_EMPTY';
/**
 * Checks whether a WAF Global Web ACL contains any WAF rules or rule groups.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/waf-global-webacl-not-empty.html
 */
ManagedRuleIdentifiers.WAF_GLOBAL_WEBACL_NOT_EMPTY = 'WAF_GLOBAL_WEBACL_NOT_EMPTY';
/**
 * Checks if WAF Regional rule groups contain any rules.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/waf-regional-rulegroup-not-empty.html
 */
ManagedRuleIdentifiers.WAF_REGIONAL_RULEGROUP_NOT_EMPTY = 'WAF_REGIONAL_RULEGROUP_NOT_EMPTY';
/**
 * Checks whether WAF regional rule contains conditions.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/waf-regional-rule-not-empty.html
 */
ManagedRuleIdentifiers.WAF_REGIONAL_RULE_NOT_EMPTY = 'WAF_REGIONAL_RULE_NOT_EMPTY';
/**
 * Checks if a WAF regional Web ACL contains any WAF rules or rule groups.
 * @see https://docs.aws.amazon.com/config/latest/developerguide/waf-regional-webacl-not-empty.html
 */
ManagedRuleIdentifiers.WAF_REGIONAL_WEBACL_NOT_EMPTY = 'WAF_REGIONAL_WEBACL_NOT_EMPTY';
exports.ManagedRuleIdentifiers = ManagedRuleIdentifiers;
/**
 * Resources types that are supported by AWS Config
 * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html
 */
class ResourceType {
    /** A custom resource type to support future cases. */
    static of(type) {
        return new ResourceType(type);
    }
    constructor(type) {
        this.complianceResourceType = type;
    }
}
_f = JSII_RTTI_SYMBOL_1;
ResourceType[_f] = { fqn: "aws-cdk-lib.aws_config.ResourceType", version: "2.74.0" };
/** API Gateway Stage */
ResourceType.APIGATEWAY_STAGE = new ResourceType('AWS::ApiGateway::Stage');
/** API Gatewayv2 Stage */
ResourceType.APIGATEWAYV2_STAGE = new ResourceType('AWS::ApiGatewayV2::Stage');
/** API Gateway REST API */
ResourceType.APIGATEWAY_REST_API = new ResourceType('AWS::ApiGateway::RestApi');
/** API Gatewayv2 API */
ResourceType.APIGATEWAYV2_API = new ResourceType('AWS::ApiGatewayV2::Api');
/** Amazon CloudFront Distribution */
ResourceType.CLOUDFRONT_DISTRIBUTION = new ResourceType('AWS::CloudFront::Distribution');
/** Amazon CloudFront streaming distribution */
ResourceType.CLOUDFRONT_STREAMING_DISTRIBUTION = new ResourceType('AWS::CloudFront::StreamingDistribution');
/** Amazon CloudWatch Alarm */
ResourceType.CLOUDWATCH_ALARM = new ResourceType('AWS::CloudWatch::Alarm');
/** Amazon CloudWatch RUM */
ResourceType.CLOUDWATCH_RUM_APP_MONITOR = new ResourceType('AWS::RUM::AppMonitor');
/** Amazon DynamoDB Table */
ResourceType.DYNAMODB_TABLE = new ResourceType('AWS::DynamoDB::Table');
/** Elastic Block Store (EBS) volume */
ResourceType.EBS_VOLUME = new ResourceType('AWS::EC2::Volume');
/** EC2 host */
ResourceType.EC2_HOST = new ResourceType('AWS::EC2::Host');
/** EC2 Elastic IP */
ResourceType.EC2_EIP = new ResourceType('AWS::EC2::EIP');
/** EC2 instance */
ResourceType.EC2_INSTANCE = new ResourceType('AWS::EC2::Instance');
/** EC2 Network Interface */
ResourceType.EC2_NETWORK_INTERFACE = new ResourceType('AWS::EC2::NetworkInterface');
/** EC2 security group */
ResourceType.EC2_SECURITY_GROUP = new ResourceType('AWS::EC2::SecurityGroup');
/** EC2 NAT gateway */
ResourceType.EC2_NAT_GATEWAY = new ResourceType('AWS::EC2::NatGateway');
/** EC2 Egress only internet gateway */
ResourceType.EC2_EGRESS_ONLY_INTERNET_GATEWAY = new ResourceType('AWS::EC2::EgressOnlyInternetGateway');
/** EC2 flow log */
ResourceType.EC2_FLOW_LOG = new ResourceType('AWS::EC2::FlowLog');
/** EC2 transit gateway */
ResourceType.EC2_TRANSIT_GATEWAY = new ResourceType('AWS::EC2::TransitGateway');
/** EC2 transit gateway attachment */
ResourceType.EC2_TRANSIT_GATEWAY_ATTACHMENT = new ResourceType('AWS::EC2::TransitGatewayAttachment');
/** EC2 transit gateway route table */
ResourceType.EC2_TRANSIT_GATEWAY_ROUTE_TABLE = new ResourceType('AWS::EC2::TransitGatewayRouteTable');
/** EC2 VPC endpoint */
ResourceType.EC2_VPC_ENDPOINT = new ResourceType('AWS::EC2::VPCEndpoint');
/** EC2 VPC endpoint service */
ResourceType.EC2_VPC_ENDPOINT_SERVICE = new ResourceType('AWS::EC2::VPCEndpointService');
/** EC2 VPC peering connection */
ResourceType.EC2_VPC_PEERING_CONNECTION = new ResourceType('AWS::EC2::VPCPeeringConnection');
/** EC2 registered HA instance */
ResourceType.EC2_REGISTERED_HA_INSTANCE = new ResourceType('AWS::EC2::RegisteredHAInstance');
/** EC2 launch template */
ResourceType.EC2_LAUNCH_TEMPLATE = new ResourceType('AWS::EC2::LaunchTemplate');
/** EC2 Network Insights Access Scope Analysis */
ResourceType.EC2_NETWORK_INSIGHTS_ACCESS_SCOPE_ANALYSIS = new ResourceType('AWS::EC2::NetworkInsightsAccessScopeAnalysis');
/** EC2 Image Builder ContainerRecipe */
ResourceType.IMAGEBUILDER_CONTAINER_RECIPE = new ResourceType('AWS::ImageBuilder::ContainerRecipe');
/** EC2 Image Builder DistributionConfiguration */
ResourceType.IMAGEBUILDER_DISTRIBUTION_CONFIGURATION = new ResourceType('AWS::ImageBuilder::DistributionConfiguration');
/** EC2 Image Builder InfrastructureConfiguration */
ResourceType.IMAGEBUILDER_INFRASTRUCTURE_CONFIGURATION = new ResourceType('AWS::ImageBuilder::InfrastructureConfiguration');
/** Amazon ECR repository */
ResourceType.ECR_REPOSITORY = new ResourceType('AWS::ECR::Repository');
/** Amazon ECR registry policy */
ResourceType.ECR_REGISTRY_POLICY = new ResourceType('AWS::ECR::RegistryPolicy');
/** Amazon ECR public repository */
ResourceType.ECR_PUBLIC_REPOSITORY = new ResourceType('AWS::ECR::PublicRepository');
/** Amazon ECS cluster */
ResourceType.ECS_CLUSTER = new ResourceType('AWS::ECS::Cluster');
/** Amazon ECS task definition */
ResourceType.ECS_TASK_DEFINITION = new ResourceType('AWS::ECS::TaskDefinition');
/** Amazon ECS service */
ResourceType.ECS_SERVICE = new ResourceType('AWS::ECS::Service');
/** Amazon EFS file system */
ResourceType.EFS_FILE_SYSTEM = new ResourceType('AWS::EFS::FileSystem');
/** Amazon EFS access point */
ResourceType.EFS_ACCESS_POINT = new ResourceType('AWS::EFS::AccessPoint');
/** Amazon Elastic Kubernetes Service cluster */
ResourceType.EKS_CLUSTER = new ResourceType('AWS::EKS::Cluster');
/** Amazon Elastic Kubernetes Service identity provider config */
ResourceType.EKS_IDENTITY_PROVIDER_CONFIG = new ResourceType('AWS::EKS::IdentityProviderConfig');
/** Amazon Elastic Kubernetes Service addon */
ResourceType.EKS_ADDON = new ResourceType('AWS::EKS::Addon');
/** Amazon EMR security configuration */
ResourceType.EMR_SECURITY_CONFIGURATION = new ResourceType('AWS::EMR::SecurityConfiguration');
/** Amazon EventBridge EventBus */
ResourceType.EVENTBRIDGE_EVENTBUS = new ResourceType('AWS::Events::EventBus');
/** Amazon EventBridge Api Destination */
ResourceType.EVENTBRIDGE_API_DESTINATION = new ResourceType('AWS::Events::ApiDestination');
/** Amazon EventBridge Archive */
ResourceType.EVENTBRIDGE_ARCHIVE = new ResourceType('AWS::Events::Archive');
/** Amazon EventBridge Endpoint */
ResourceType.EVENTBRIDGE_ENDPOINT = new ResourceType('AWS::Events::Endpoint');
/** Amazon EventBridge EventSchemas registry */
ResourceType.EVENTSCHEMAS_REGISTRY = new ResourceType('AWS::EventSchemas::Registry');
/** Amazon EventBridge EventSchemas registry policy */
ResourceType.EVENTSCHEMAS_REGISTRY_POLICY = new ResourceType('AWS::EventSchemas::RegistryPolicy');
/** Amazon EventBridge EventSchemas discoverer */
ResourceType.EVENTSCHEMAS_DISCOVERER = new ResourceType('AWS::EventSchemas::Discoverer');
/** AWS FraudDetector label */
ResourceType.FRAUDDETECTOR_LABEL = new ResourceType('AWS::FraudDetector::Label');
/** AWS FraudDetector entity type */
ResourceType.FRAUDDETECTOR_ENTITY_TYPE = new ResourceType('AWS::FraudDetector::EntityType');
/** AWS FraudDetector variable */
ResourceType.FRAUDDETECTOR_VARIABLE = new ResourceType('AWS::FraudDetector::Variable');
/** AWS FraudDetector outcome */
ResourceType.FRAUDDETECTOR_OUTCOME = new ResourceType('AWS::FraudDetector::Outcome');
/** Amazon GuardDuty detector */
ResourceType.GUARDDUTY_DETECTOR = new ResourceType('AWS::GuardDuty::Detector');
/** Amazon GuardDuty Threat Intel Set */
ResourceType.GUARDDUTY_THREAT_INTEL_SET = new ResourceType('AWS::GuardDuty::ThreatIntelSet');
/** Amazon GuardDuty IP Set */
ResourceType.GUARDDUTY_IP_SET = new ResourceType(' AWS::GuardDuty::IPSet');
/** Amazon GuardDuty Filter */
ResourceType.GUARDDUTY_FILTER = new ResourceType('AWS::GuardDuty::Filter');
/** Amazon ElasticSearch domain */
ResourceType.ELASTICSEARCH_DOMAIN = new ResourceType('AWS::Elasticsearch::Domain');
/** Amazon Interactive Video Service (IVS) channel */
ResourceType.IVS_CHANNEL = new ResourceType('AWS::IVS::Channel');
/** Amazon Interactive Video Service (IVS) recording configuration */
ResourceType.IVS_RECORDING_CONFIGURATION = new ResourceType('AWS::IVS::RecordingConfiguration');
/** Amazon Interactive Video Service (IVS) playback key pair */
ResourceType.IVS_PLAYBACK_KEYPAIR = new ResourceType('AWS::IVS::PlaybackKeyPair');
/** Amazon OpenSearch domain */
ResourceType.OPENSEARCH_DOMAIN = new ResourceType('AWS::OpenSearch::Domain');
/** Amazon QLDB ledger */
ResourceType.QLDB_LEDGER = new ResourceType('AWS::QLDB::Ledger');
/** Amazon Kinesis stream */
ResourceType.KINESIS_STREAM = new ResourceType('AWS::Kinesis::Stream');
/** Amazon Kinesis stream consumer */
ResourceType.KINESIS_STREAM_CONSUMER = new ResourceType('AWS::Kinesis::StreamConsumer');
/** Amazon Kinesis Analytics V2 application */
ResourceType.KINESIS_ANALYTICS_V2_APPLICATION = new ResourceType('AWS::KinesisAnalyticsV2::Application');
/** Amazon Lightsail Certificate */
ResourceType.LIGHTSAIL_CERTIFICATE = new ResourceType('AWS::Lightsail::Certificate');
/** Amazon Lightsail Disk */
ResourceType.LIGHTSAIL_DISK = new ResourceType('AWS::Lightsail::Disk');
/** AWS Lightsail bucket */
ResourceType.LIGHTSAIL_BUCKET = new ResourceType('AWS::Lightsail::Bucket');
/** AWS Lightsail static IP */
ResourceType.LIGHTSAIL_STATIC_IP = new ResourceType('AWS::Lightsail::StaticIp');
/** Amazon MQ broker */
ResourceType.AMAZON_MQ_BROKER = new ResourceType('AWS::AmazonMQ::Broker');
/** Amazon MSK cluster */
ResourceType.MSK_CLUSTER = new ResourceType('AWS::MSK::Cluster');
/** Amazon Redshift cluster */
ResourceType.REDSHIFT_CLUSTER = new ResourceType('AWS::Redshift::Cluster');
/** Amazon Redshift cluster parameter group */
ResourceType.REDSHIFT_CLUSTER_PARAMETER_GROUP = new ResourceType('AWS::Redshift::ClusterParameterGroup');
/** Amazon Redshift cluster security group */
ResourceType.REDSHIFT_CLUSTER_SECURITY_GROUP = new ResourceType('AWS::Redshift::ClusterSecurityGroup');
/** Amazon Redshift cluster snapshot */
ResourceType.REDSHIFT_CLUSTER_SNAPSHOT = new ResourceType('AWS::Redshift::ClusterSnapshot');
/** Amazon Redshift cluster subnet group */
ResourceType.REDSHIFT_CLUSTER_SUBNET_GROUP = new ResourceType('AWS::Redshift::ClusterSubnetGroup');
/** Amazon Redshift event subscription */
ResourceType.REDSHIFT_EVENT_SUBSCRIPTION = new ResourceType('AWS::Redshift::EventSubscription');
/** Amazon RDS database instance */
ResourceType.RDS_DB_INSTANCE = new ResourceType('AWS::RDS::DBInstance');
/** Amazon RDS database security group */
ResourceType.RDS_DB_SECURITY_GROUP = new ResourceType('AWS::RDS::DBSecurityGroup');
/** Amazon RDS database snapshot */
ResourceType.RDS_DB_SNAPSHOT = new ResourceType('AWS::RDS::DBSnapshot');
/** Amazon RDS database subnet group */
ResourceType.RDS_DB_SUBNET_GROUP = new ResourceType('AWS::RDS::DBSubnetGroup');
/** Amazon RDS event subscription */
ResourceType.RDS_EVENT_SUBSCRIPTION = new ResourceType('AWS::RDS::EventSubscription');
/** Amazon RDS database cluster */
ResourceType.RDS_DB_CLUSTER = new ResourceType('AWS::RDS::DBCluster');
/** Amazon RDS database cluster snapshot */
ResourceType.RDS_DB_CLUSTER_SNAPSHOT = new ResourceType('AWS::RDS::DBClusterSnapshot');
/** Amazon RDS global cluster */
ResourceType.RDS_GLOBAL_CLUSTER = new ResourceType('AWS::RDS::GlobalCluster');
/** Amazon Route53 Hosted Zone */
ResourceType.ROUTE53_HOSTED_ZONE = new ResourceType('AWS::Route53::HostedZone');
/** Amazon Route53 Health Check */
ResourceType.ROUTE53_HEALTH_CHECK = new ResourceType('AWS::Route53::HealthCheck');
/** Amazon Route53 resolver resolver endpoint */
ResourceType.ROUTE53_RESOLVER_RESOLVER_ENDPOINT = new ResourceType('AWS::Route53Resolver::ResolverEndpoint');
/** Amazon Route53 resolver resolver rule */
ResourceType.ROUTE53_RESOLVER_RESOLVER_RULE = new ResourceType('AWS::Route53Resolver::ResolverRule');
/** Amazon Route53 resolver resolver rule association */
ResourceType.ROUTE53_RESOLVER_RESOLVER_RULE_ASSOCIATION = new ResourceType('AWS::Route53Resolver::ResolverRuleAssociation');
/** Amazon Route 53 Application Recovery Controller Cell */
ResourceType.ROUTE53_RECOVERY_READINESS_CELL = new ResourceType('AWS::Route53RecoveryReadiness::Cell');
/** Amazon Route 53 Application Recovery Controller Readiness Check */
ResourceType.ROUTE53_RECOVERY_READINESS_READINESS_CHECK = new ResourceType('AWS::Route53RecoveryReadiness::ReadinessCheck');
/** Amazon Route53 recovery readiness recovery group */
ResourceType.ROUTE53_RECOVERY_READINESS_RECOVERY_GROUP = new ResourceType('AWS::Route53RecoveryReadiness::RecoveryGroup');
/** Amazon SQS queue */
ResourceType.SQS_QUEUE = new ResourceType('AWS::SQS::Queue');
/** Amazon SNS topic */
ResourceType.SNS_TOPIC = new ResourceType('AWS::SNS::Topic');
/** Amazon S3 bucket */
ResourceType.S3_BUCKET = new ResourceType('AWS::S3::Bucket');
/** Amazon S3 Multi-Region Access Point */
ResourceType.S3_MULTIREGION_ACCESS_POINT = new ResourceType('AWS::S3::MultiRegionAccessPoint');
/** Amazon SageMaker code repository */
ResourceType.SAGEMAKER_CODE_REPOSITORY = new ResourceType('AWS::SageMaker::CodeRepository');
/** Amazon SageMaker model */
ResourceType.SAGEMAKER_MODEL = new ResourceType('AWS::SageMaker::Model');
/** Amazon SageMaker notebook instance */
ResourceType.SAGEMAKER_NOTEBOOK_INSTANCE = new ResourceType('AWS::SageMaker::NotebookInstance');
/** Amazon SageMaker workteam */
ResourceType.SAGEMAKER_WORKTEAM = new ResourceType('AWS::SageMaker::Workteam');
/** Amazon SES Configuration Set */
ResourceType.SES_CONFIGURATION_SET = new ResourceType('AWS::SES::ConfigurationSet');
/** Amazon SES Contact List */
ResourceType.SES_CONTACT_LIST = new ResourceType('AWS::SES::ContactList');
/** Amazon SES Template */
ResourceType.SES_TEMPLATE = new ResourceType('AWS::SES::Template');
/** Amazon SES ReceiptFilter */
ResourceType.SES_RECEIPT_FILTER = new ResourceType('AWS::SES::ReceiptFilter');
/** Amazon SES ReceiptRuleSet */
ResourceType.SES_RECEIPT_RECEIPT_RULE_SET = new ResourceType('AWS::SES::ReceiptRuleSet');
/** Amazon S3 account public access block */
ResourceType.S3_ACCOUNT_PUBLIC_ACCESS_BLOCK = new ResourceType('AWS::S3::AccountPublicAccessBlock');
/** Amazon EC2 customer gateway */
ResourceType.EC2_CUSTOMER_GATEWAY = new ResourceType('AWS::EC2::CustomerGateway');
/** Amazon EC2 internet gateway */
ResourceType.EC2_INTERNET_GATEWAY = new ResourceType('AWS::EC2::InternetGateway');
/** Amazon EC2 network ACL */
ResourceType.EC2_NETWORK_ACL = new ResourceType('AWS::EC2::NetworkAcl');
/** Amazon EC2 route table */
ResourceType.EC2_ROUTE_TABLE = new ResourceType('AWS::EC2::RouteTable');
/** Amazon EC2 subnet table */
ResourceType.EC2_SUBNET = new ResourceType('AWS::EC2::Subnet');
/** Amazon EC2 VPC */
ResourceType.EC2_VPC = new ResourceType('AWS::EC2::VPC');
/** Amazon EC2 VPN connection */
ResourceType.EC2_VPN_CONNECTION = new ResourceType('AWS::EC2::VPNConnection');
/** Amazon EC2 VPN gateway */
ResourceType.EC2_VPN_GATEWAY = new ResourceType('AWS::EC2::VPNGateway');
/** AWS Auto Scaling group */
ResourceType.AUTO_SCALING_GROUP = new ResourceType('AWS::AutoScaling::AutoScalingGroup');
/** AWS Auto Scaling launch configuration */
ResourceType.AUTO_SCALING_LAUNCH_CONFIGURATION = new ResourceType('AWS::AutoScaling::LaunchConfiguration');
/** AWS Auto Scaling policy */
ResourceType.AUTO_SCALING_POLICY = new ResourceType('AWS::AutoScaling::ScalingPolicy');
/** AWS Auto Scaling scheduled action */
ResourceType.AUTO_SCALING_SCHEDULED_ACTION = new ResourceType('AWS::AutoScaling::ScheduledAction');
/** Amazon WorkSpaces connection alias */
ResourceType.WORKSPACES_CONNECTION_ALIAS = new ResourceType('AWS::WorkSpaces::ConnectionAlias');
/** Amazon WorkSpaces workSpace */
ResourceType.WORKSPACES_WORKSPACE = new ResourceType('AWS::WorkSpaces::Workspace');
/** AWS AppConfig application */
ResourceType.APPCONFIG_APPLICATION = new ResourceType('AWS::AppConfig::Application');
/** AWS AppConfig environment */
ResourceType.APPCONFIG_ENVIRONMENT = new ResourceType('AWS::AppConfig::Environment');
/** AWS AppConfig configuration profile */
ResourceType.APPCONFIG_CONFIGURATION_PROFILE = new ResourceType('AWS::AppConfig::ConfigurationProfile');
/** AWS AppSync GraphQL Api */
ResourceType.APPSYNC_GRAPHQL_API = new ResourceType('AWS::AppSync::GraphQLApi');
/** AWS Backup backup plan */
ResourceType.BACKUP_BACKUP_PLAN = new ResourceType('AWS::Backup::BackupPlan');
/** AWS Backup backup selection */
ResourceType.BACKUP_BACKUP_SELECTION = new ResourceType('AWS::Backup::BackupSelection');
/** AWS Backup backup vault */
ResourceType.BACKUP_BACKUP_VAULT = new ResourceType('AWS::Backup::BackupVault');
/** AWS Backup recovery point */
ResourceType.BACKUP_RECOVERY_POINT = new ResourceType('AWS::Backup::RecoveryPoint');
/** AWS Backup report plan */
ResourceType.BACKUP_REPORT_PLAN = new ResourceType('AWS::Backup::ReportPlan');
/** AWS Batch job queue */
ResourceType.BATCH_JOB_QUEUE = new ResourceType('AWS::Batch::JobQueue');
/** AWS Batch compute environment */
ResourceType.BATCH_COMPUTE_ENVIRONMENT = new ResourceType('AWS::Batch::ComputeEnvironment');
/** AWS Certificate manager certificate */
ResourceType.ACM_CERTIFICATE = new ResourceType('AWS::ACM::Certificate');
/** AWS CloudFormation stack */
ResourceType.CLOUDFORMATION_STACK = new ResourceType('AWS::CloudFormation::Stack');
/** AWS CloudTrail trail */
ResourceType.CLOUDTRAIL_TRAIL = new ResourceType('AWS::CloudTrail::Trail');
/** AWS Cloud9 environment EC2 */
ResourceType.CLOUD9_ENVIRONMENT_EC2 = new ResourceType('AWS::Cloud9::EnvironmentEC2');
/** AWS Cloud Map(ServiceDiscovery) service */
ResourceType.SERVICEDISCOVERY_SERVICE = new ResourceType('AWS::ServiceDiscovery::Service');
/** AWS Cloud Map(ServiceDiscovery) Public Dns Namespace */
ResourceType.SERVICEDISCOVERY_PUBLIC_DNS_NAMESPACE = new ResourceType('AWS::ServiceDiscovery::PublicDnsNamespace');
/** AWS Cloud Map(ServiceDiscovery) Http Namespace */
ResourceType.SERVICEDISCOVERY_HTTP_NAMESPACE = new ResourceType('AWS::ServiceDiscovery::HttpNamespace');
/** AWS CodeBuild project */
ResourceType.CODEBUILD_PROJECT = new ResourceType('AWS::CodeBuild::Project');
/** AWS CodeDeploy application */
ResourceType.CODEDEPLOY_APPLICATION = new ResourceType('AWS::CodeDeploy::Application');
/** AWS CodeDeploy deployment config */
ResourceType.CODEDEPLOY_DEPLOYMENT_CONFIG = new ResourceType('AWS::CodeDeploy::DeploymentConfig');
/** AWS CodeDeploy deployment group */
ResourceType.CODEDEPLOY_DEPLOYMENT_GROUP = new ResourceType('AWS::CodeDeploy::DeploymentGroup');
/** AWS CodePipeline pipeline */
ResourceType.CODEPIPELINE_PIPELINE = new ResourceType('AWS::CodePipeline::Pipeline');
/** AWS Config resource compliance */
ResourceType.CONFIG_RESOURCE_COMPLIANCE = new ResourceType('AWS::Config::ResourceCompliance');
/** AWS Config conformance pack compliance */
ResourceType.CONFIG_CONFORMANCE_PACK_COMPLIANCE = new ResourceType('AWS::Config::ConformancePackCompliance');
/** AWS DMS event subscription */
ResourceType.DMS_EVENT_SUBSCRIPTION = new ResourceType('AWS::DMS::EventSubscription');
/** AWS DMS replication subnet group */
ResourceType.DMS_REPLICATION_SUBNET_GROUP = new ResourceType('AWS::DMS::ReplicationSubnetGroup');
/** AWS DataSync location SMB */
ResourceType.DATASYNC_LOCATION_SMB = new ResourceType('AWS::DataSync::LocationSMB');
/** AWS DataSync location FSx Lustre */
ResourceType.DATASYNC_LOCATION_FSX_LUSTRE = new ResourceType('AWS::DataSync::LocationFSxLustre');
/** AWS DataSync location FSx Windows */
ResourceType.DATASYNC_LOCATION_FSX_WINDOWS = new ResourceType('AWS::DataSync::LocationFSxWindows');
/** AWS DataSync location S3 */
ResourceType.DATASYNC_LOCATION_S3 = new ResourceType('AWS::DataSync::LocationS3');
/** AWS DataSync location EFS */
ResourceType.DATASYNC_LOCATION_EFS = new ResourceType('AWS::DataSync::LocationEFS');
/** AWS DataSync task */
ResourceType.DATASYNC_TASK = new ResourceType('AWS::DataSync::Task');
/** AWS DataSync location NFS */
ResourceType.DATASYNC_LOCATION_NFS = new ResourceType('AWS::DataSync::LocationNFS');
/** AWS DataSync location object storage */
ResourceType.DATASYNC_LOCATION_OBJECT_STORAGE = new ResourceType('AWS::DataSync::LocationObjectStorage');
/** AWS DataSync location HDFS */
ResourceType.DATASYNC_LOCATION_HDFS = new ResourceType('AWS::DataSync::LocationHDFS');
/** AWS Elastic Beanstalk (EB) application */
ResourceType.ELASTIC_BEANSTALK_APPLICATION = new ResourceType('AWS::ElasticBeanstalk::Application');
/** AWS Elastic Beanstalk (EB) application version */
ResourceType.ELASTIC_BEANSTALK_APPLICATION_VERSION = new ResourceType('AWS::ElasticBeanstalk::ApplicationVersion');
/** AWS Elastic Beanstalk (EB) environment */
ResourceType.ELASTIC_BEANSTALK_ENVIRONMENT = new ResourceType('AWS::ElasticBeanstalk::Environment');
/** AWS Fault Injection Simulator Experiment_Template */
ResourceType.FIS_EXPERIMENT_TEMPLATE = new ResourceType('AWS::FIS::ExperimentTemplate');
/** AWS GlobalAccelerator listener */
ResourceType.GLOBALACCELERATOR_LISTENER = new ResourceType('AWS::GlobalAccelerator::Listener');
/** AWS GlobalAccelerator endpoint group */
ResourceType.GLOBALACCELERATOR_ENDPOINT_GROUP = new ResourceType('AWS::GlobalAccelerator::EndpointGroup');
/** AWS GlobalAccelerator accelerator */
ResourceType.GLOBALACCELERATOR_ACCELERATOR = new ResourceType('AWS::GlobalAccelerator::Accelerator');
/** AWS Glue Job */
ResourceType.GLUE_JOB = new ResourceType('AWS::Glue::Job');
/** AWS Glue Classifier */
ResourceType.GLUE_CLASSIFIER = new ResourceType('AWS::Glue::Classifier');
/** AWS Glue machine learning transform */
ResourceType.GLUE_ML_TRANSFORM = new ResourceType('AWS::Glue::MLTransform');
/** AWS IAM user */
ResourceType.IAM_USER = new ResourceType('AWS::IAM::User');
/** AWS IAM group */
ResourceType.IAM_GROUP = new ResourceType('AWS::IAM::Group');
/** AWS IAM role */
ResourceType.IAM_ROLE = new ResourceType('AWS::IAM::Role');
/** AWS IAM policy */
ResourceType.IAM_POLICY = new ResourceType('AWS::IAM::Policy');
/** AWS IAM AccessAnalyzer analyzer */
ResourceType.IAM_ACCESSANALYZER_ANALYZER = new ResourceType('AWS::AccessAnalyzer::Analyzer');
/** AWS IoT authorizer */
ResourceType.IOT_AUTHORIZER = new ResourceType('AWS::IoT::Authorizer');
/** AWS IoT security profile */
ResourceType.IOT_SECURITY_PROFILE = new ResourceType('AWS::IoT::SecurityProfile');
/** AWS IoT role alias */
ResourceType.IOT_ROLE_ALIAS = new ResourceType('AWS::IoT::RoleAlias');
/** AWS IoT dimension */
ResourceType.IOT_DIMENSION = new ResourceType('AWS::IoT::Dimension');
/** AWS IoT policy */
ResourceType.IOT_POLICY = new ResourceType('AWS::IoT::Policy');
/** AWS IoT mitigation action */
ResourceType.IOT_MITIGATION_ACTION = new ResourceType('AWS::IoT::MitigationAction');
/** AWS IoT TwinMaker workspace */
ResourceType.IOT_TWINMAKER_WORKSPACE = new ResourceType('AWS::IoTwinMaker::Workspace');
/** AWS IoT TwinMaker entity */
ResourceType.IOT_TWINMAKER_ENTITY = new ResourceType('AWS::IoTTwinMaker::Entity');
/** AWS IoT Analytics datastore */
ResourceType.IOT_ANALYTICS_DATASTORE = new ResourceType('AWS::IoTAnalytics::Datastore');
/** AWS IoT Analytics dataset */
ResourceType.IOT_ANALYTICS_DATASET = new ResourceType('AWS::IoTAnalytics::Dataset');
/** AWS IoT Analytics pipeline */
ResourceType.IOT_ANALYTICS_PIPELINE = new ResourceType('AWS::IoTAnalytics::Pipeline');
/** AWS IoT Analytics channel */
ResourceType.IOT_ANALYTICS_CHANNEL = new ResourceType('AWS::IoTAnalytics::Channel');
/** AWS IoT Events Input */
ResourceType.IOT_EVENTS_INPUT = new ResourceType('AWS::IoTEvents::Input');
/** AWS IoT Events Detector Model */
ResourceType.IOT_EVENTS_DETECTOR_MODEL = new ResourceType('AWS::IoTEvents::DetectorModel');
/** AWS IoT Events Alarm Model */
ResourceType.IOT_EVENTS_ALARM_MODEL = new ResourceType('AWS::IoTEvents::AlarmModel');
/** AWS IoT SiteWise dashboard */
ResourceType.IOT_SITEWISE_DASHBOARD = new ResourceType('AWS::IoTSiteWise::Dashboard');
/** AWS IoT SiteWise project */
ResourceType.IOT_SITEWISE_PROJECT = new ResourceType('AWS::IoTSiteWise::Project');
/** AWS IoT SiteWise portal */
ResourceType.IOT_SITEWISE_PORTAL = new ResourceType('AWS::IoTSiteWise::Portal');
/** AWS IoT SiteWise asset model */
ResourceType.IOT_SITEWISE_ASSETMODEL = new ResourceType('AWS::IoTSiteWise::AssetModel');
/** AWS KMS Key */
ResourceType.KMS_KEY = new ResourceType('AWS::KMS::Key');
/** AWS Lambda function */
ResourceType.LAMBDA_FUNCTION = new ResourceType('AWS::Lambda::Function');
/** AWS Network Firewall Firewall */
ResourceType.NETWORK_FIREWALL_FIREWALL = new ResourceType('AWS::NetworkFirewall::Firewall');
/** AWS Network Firewall Firewall Policy */
ResourceType.NETWORK_FIREWALL_FIREWALL_POLICY = new ResourceType('AWS::NetworkFirewall::FirewallPolicy');
/** AWS Network Firewall Rule Group */
ResourceType.NETWORK_FIREWALL_RULE_GROUP = new ResourceType('AWS::NetworkFirewall::RuleGroup');
/** AWS ResilienceHub resiliency policy */
ResourceType.RESILIENCEHUB_RESILIENCY_POLICY = new ResourceType('AWS::ResilienceHub::ResiliencyPolicy');
/**AWS Secrets Manager secret */
ResourceType.SECRETS_MANAGER_SECRET = new ResourceType('AWS::SecretsManager::Secret');
/** AWS Service Catalog CloudFormation product */
ResourceType.SERVICE_CATALOG_CLOUDFORMATION_PRODUCT = new ResourceType('AWS::ServiceCatalog::CloudFormationProduct');
/** AWS Service Catalog CloudFormation provisioned product */
ResourceType.SERVICE_CATALOG_CLOUDFORMATION_PROVISIONED_PRODUCT = new ResourceType('AWS::ServiceCatalog::CloudFormationProvisionedProduct');
/** AWS Service Catalog portfolio */
ResourceType.SERVICE_CATALOG_PORTFOLIO = new ResourceType('AWS::ServiceCatalog::Portfolio');
/** AWS Shield protection */
ResourceType.SHIELD_PROTECTION = new ResourceType('AWS::Shield::Protection');
/** AWS Shield regional protection */
ResourceType.SHIELD_REGIONAL_PROTECTION = new ResourceType('AWS::ShieldRegional::Protection');
/** AWS StepFunctions activity */
ResourceType.STEPFUNCTIONS_ACTIVITY = new ResourceType('AWS::StepFunctions::Activity');
/** AWS StepFunctions state machine */
ResourceType.STEPFUNCTIONS_STATE_MACHINE = new ResourceType('AWS::StepFunctions::StateMachine');
/** AWS Systems Manager managed instance inventory */
ResourceType.SYSTEMS_MANAGER_MANAGED_INSTANCE_INVENTORY = new ResourceType('AWS::SSM::ManagedInstanceInventory');
/** AWS Systems Manager patch compliance */
ResourceType.SYSTEMS_MANAGER_PATCH_COMPLIANCE = new ResourceType('AWS::SSM::PatchCompliance');
/** AWS Systems Manager association compliance */
ResourceType.SYSTEMS_MANAGER_ASSOCIATION_COMPLIANCE = new ResourceType('AWS::SSM::AssociationCompliance');
/** AWS Systems Manager file data */
ResourceType.SYSTEMS_MANAGER_FILE_DATA = new ResourceType('AWS::SSM::FileData');
/** AWS Transfer workflow */
ResourceType.TRANSFER_WORKFLOW = new ResourceType('AWS::Transfer::Workflow');
/** AWS WAF rate based rule */
ResourceType.WAF_RATE_BASED_RULE = new ResourceType('AWS::WAF::RateBasedRule');
/** AWS WAF rule */
ResourceType.WAF_RULE = new ResourceType('AWS::WAF::Rule');
/** AWS WAF web ACL */
ResourceType.WAF_WEB_ACL = new ResourceType('AWS::WAF::WebACL');
/** AWS WAF rule group */
ResourceType.WAF_RULE_GROUP = new ResourceType('AWS::WAF::RuleGroup');
/** AWS WAF regional rate based rule */
ResourceType.WAF_REGIONAL_RATE_BASED_RULE = new ResourceType('AWS::WAFRegional::RateBasedRule');
/** AWS WAF regional rule */
ResourceType.WAF_REGIONAL_RULE = new ResourceType('AWS::WAFRegional::Rule');
/** AWS WAF web ACL */
ResourceType.WAF_REGIONAL_WEB_ACL = new ResourceType('AWS::WAFRegional::WebACL');
/** AWS WAF regional rule group */
ResourceType.WAF_REGIONAL_RULE_GROUP = new ResourceType('AWS::WAFRegional::RuleGroup');
/** AWS WAFv2 web ACL */
ResourceType.WAFV2_WEB_ACL = new ResourceType('AWS::WAFv2::WebACL');
/** AWS WAFv2 rule group */
ResourceType.WAFV2_RULE_GROUP = new ResourceType('AWS::WAFv2::RuleGroup');
/** AWS WAFv2 managed rule set */
ResourceType.WAFV2_MANAGED_RULE_SET = new ResourceType('AWS::WAFv2::ManagedRuleSet');
/** AWS WAFv2 ip set */
ResourceType.WAFV2_IP_SET = new ResourceType('AWS::WAFv2::IPSet');
/** AWS WAFv2 regex pattern set */
ResourceType.WAFV2_REGEX_PATTERN_SET = new ResourceType('AWS::WAFv2::RegexPatternSet');
/** AWS X-Ray encryption configuration */
ResourceType.XRAY_ENCRYPTION_CONFIGURATION = new ResourceType('AWS::XRay::EncryptionConfig');
/** AWS ELB classic load balancer */
ResourceType.ELB_LOAD_BALANCER = new ResourceType('AWS::ElasticLoadBalancing::LoadBalancer');
/** AWS ELBv2 network load balancer or AWS ELBv2 application load balancer */
ResourceType.ELBV2_LOAD_BALANCER = new ResourceType('AWS::ElasticLoadBalancingV2::LoadBalancer');
/** AWS ELBv2 application load balancer listener */
ResourceType.ELBV2_LISTENER = new ResourceType('AWS::ElasticLoadBalancingV2::Listener');
/** AWS Elemental MediaPackage packaging group */
ResourceType.MEDIAPACKAGE_PACKAGING_GROUP = new ResourceType('AWS::MediaPackage::PackagingGroup');
exports.ResourceType = ResourceType;
function renderScope(ruleScope) {
    return ruleScope ? {
        complianceResourceId: ruleScope.resourceId,
        complianceResourceTypes: ruleScope.resourceTypes?.map(resource => resource.complianceResourceType),
        tagKey: ruleScope.key,
        tagValue: ruleScope.value,
    } : undefined;
}
//# sourceMappingURL=data:application/json;base64,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