"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReplicaProvider = void 0;
const path = require("path");
const iam = require("../../aws-iam");
const lambda = require("../../aws-lambda");
const core_1 = require("../../core");
const cr = require("../../custom-resources");
class ReplicaProvider extends core_1.NestedStack {
    /**
     * Creates a stack-singleton resource provider nested stack.
     */
    static getOrCreate(scope, props) {
        const stack = core_1.Stack.of(scope);
        const uid = '@aws-cdk/aws-dynamodb.ReplicaProvider';
        return stack.node.tryFindChild(uid) ?? new ReplicaProvider(stack, uid, props);
    }
    constructor(scope, id, props) {
        super(scope, id);
        const code = lambda.Code.fromAsset(path.join(__dirname, 'replica-handler'));
        // Issues UpdateTable API calls
        this.onEventHandler = new lambda.Function(this, 'OnEventHandler', {
            code,
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.onEventHandler',
            timeout: core_1.Duration.minutes(5),
        });
        // Checks if table is back to `ACTIVE` state
        this.isCompleteHandler = new lambda.Function(this, 'IsCompleteHandler', {
            code,
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.isCompleteHandler',
            timeout: core_1.Duration.seconds(30),
        });
        // Allows the creation of the `AWSServiceRoleForDynamoDBReplication` service linked role
        this.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: [core_1.Stack.of(this).formatArn({
                    service: 'iam',
                    region: '',
                    resource: 'role',
                    resourceName: 'aws-service-role/replication.dynamodb.amazonaws.com/AWSServiceRoleForDynamoDBReplication',
                })],
        }));
        // Required for replica table creation
        this.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['dynamodb:DescribeLimits'],
            resources: ['*'],
        }));
        // Required for replica table deletion
        let resources = [];
        props.regions.forEach((region) => {
            resources.push(`arn:aws:dynamodb:${region}:${this.account}:table/${props.tableName}`);
        });
        this.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['dynamodb:DeleteTable', 'dynamodb:DeleteTableReplica'],
            resources: resources,
        }));
        this.provider = new cr.Provider(this, 'Provider', {
            onEventHandler: this.onEventHandler,
            isCompleteHandler: this.isCompleteHandler,
            queryInterval: core_1.Duration.seconds(10),
            totalTimeout: props.timeout,
        });
    }
}
exports.ReplicaProvider = ReplicaProvider;
//# sourceMappingURL=data:application/json;base64,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