"use strict";
var _a, _b, _c, _d, _e, _f, _g;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LookupMachineImage = exports.OperatingSystemType = exports.GenericWindowsImage = exports.GenericLinuxImage = exports.AmazonLinuxStorage = exports.AmazonLinuxVirt = exports.AmazonLinuxEdition = exports.AmazonLinuxKernel = exports.AmazonLinuxGeneration = exports.AmazonLinuxImage = exports.AmazonLinuxCpuType = exports.WindowsImage = exports.GenericSSMParameterImage = exports.MachineImage = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ssm = require("../../aws-ssm");
const cxschema = require("../../cloud-assembly-schema");
const core_1 = require("../../core");
const user_data_1 = require("./user-data");
const windows_versions_1 = require("./windows-versions");
/**
 * Factory functions for standard Amazon Machine Image objects.
 */
class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestWindows(version, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_WindowsVersion(version);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_WindowsImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.latestWindows);
            }
            throw error;
        }
        return new WindowsImage(version, props);
    }
    /**
     * An Amazon Linux image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * N.B.: "latest" in the name of this function indicates that it always uses the most recent
     * image of a particular generation of Amazon Linux, not that it uses the "latest generation".
     * For backwards compatibility, this function uses Amazon Linux 1 if no generation
     * is specified.
     *
     * Specify the desired generation using the `generation` property:
     *
     * ```ts
     * ec2.MachineImage.latestAmazonLinux({
     *   // Use Amazon Linux 2
     *   generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
     * })
     * ```
     */
    static latestAmazonLinux(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_AmazonLinuxImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.latestAmazonLinux);
            }
            throw error;
        }
        return new AmazonLinuxImage(props);
    }
    /**
     * A Linux image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericLinux(amiMap, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_GenericLinuxImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.genericLinux);
            }
            throw error;
        }
        return new GenericLinuxImage(amiMap, props);
    }
    /**
     * A Windows image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericWindows(amiMap, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_GenericWindowsImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.genericWindows);
            }
            throw error;
        }
        return new GenericWindowsImage(amiMap, props);
    }
    /**
     * An image specified in SSM parameter store that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @param parameterName The name of SSM parameter containing the AMi id
     * @param os The operating system type of the AMI
     * @param userData optional user data for the given image
     * @deprecated Use `MachineImage.fromSsmParameter()` instead
     */
    static fromSSMParameter(parameterName, os, userData) {
        return new GenericSSMParameterImage(parameterName, os, userData);
    }
    /**
     * An image specified in SSM parameter store
     *
     * By default, the SSM parameter is refreshed at every deployment,
     * causing your instances to be replaced whenever a new version of the AMI
     * is released.
     *
     * Pass `{ cachedInContext: true }` to keep the AMI ID stable. If you do, you
     * will have to remember to periodically invalidate the context to refresh
     * to the newest AMI ID.
     */
    static fromSsmParameter(parameterName, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_SsmParameterImageOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSsmParameter);
            }
            throw error;
        }
        return new GenericSsmParameterImage(parameterName, options);
    }
    /**
     * Look up a shared Machine Image using DescribeImages
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     *
     * This function can not be used in environment-agnostic stacks.
     */
    static lookup(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_LookupMachineImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.lookup);
            }
            throw error;
        }
        return new LookupMachineImage(props);
    }
}
_a = JSII_RTTI_SYMBOL_1;
MachineImage[_a] = { fqn: "aws-cdk-lib.aws_ec2.MachineImage", version: "2.74.0" };
exports.MachineImage = MachineImage;
/**
 * Select the image based on a given SSM parameter
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class GenericSSMParameterImage {
    constructor(parameterName, os, userData) {
        this.os = os;
        this.userData = userData;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_OperatingSystemType(os);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_UserData(userData);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GenericSSMParameterImage);
            }
            throw error;
        }
        this.parameterName = parameterName;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameterV2(scope, this.parameterName, ssm.ParameterValueType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            osType: this.os,
            userData: this.userData ?? (this.os === OperatingSystemType.WINDOWS ? user_data_1.UserData.forWindows() : user_data_1.UserData.forLinux()),
        };
    }
}
_b = JSII_RTTI_SYMBOL_1;
GenericSSMParameterImage[_b] = { fqn: "aws-cdk-lib.aws_ec2.GenericSSMParameterImage", version: "2.74.0" };
exports.GenericSSMParameterImage = GenericSSMParameterImage;
/**
 * Select the image based on a given SSM parameter
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class GenericSsmParameterImage {
    constructor(parameterName, props = {}) {
        this.parameterName = parameterName;
        this.props = props;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const imageId = lookupImage(scope, this.props.cachedInContext, this.parameterName);
        const osType = this.props.os ?? OperatingSystemType.LINUX;
        return {
            imageId,
            osType,
            userData: this.props.userData ?? (osType === OperatingSystemType.WINDOWS ? user_data_1.UserData.forWindows() : user_data_1.UserData.forLinux()),
        };
    }
}
/**
 * Select the latest version of the indicated Windows version
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
class WindowsImage extends GenericSSMParameterImage {
    constructor(version, props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_WindowsVersion(version);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_WindowsImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, WindowsImage);
            }
            throw error;
        }
        const nonDeprecatedVersionName = WindowsImage.DEPRECATED_VERSION_NAME_MAP[version] ?? version;
        super('/aws/service/ami-windows-latest/' + nonDeprecatedVersionName, OperatingSystemType.WINDOWS, props.userData);
    }
}
_c = JSII_RTTI_SYMBOL_1;
WindowsImage[_c] = { fqn: "aws-cdk-lib.aws_ec2.WindowsImage", version: "2.74.0" };
WindowsImage.DEPRECATED_VERSION_NAME_MAP = {
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_GERMAL_FULL_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_GERMAN_FULL_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_SP1_PORTUGESE_BRAZIL_64BIT_CORE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_SP1_PORTUGUESE_BRAZIL_64BIT_CORE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_PORTUGESE_PORTUGAL_FULL_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_PORTUGUESE_PORTUGAL_FULL_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_RTM_PORTUGESE_BRAZIL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_RTM_PORTUGUESE_BRAZIL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_RTM_PORTUGESE_PORTUGAL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_RTM_PORTUGUESE_PORTUGAL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_PORTUGESE_BRAZIL_FULL_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_PORTUGUESE_BRAZIL_FULL_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_SP2_PORTUGESE_BRAZIL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_SP2_PORTUGUESE_BRAZIL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_RTM_PORTUGESE_BRAZIL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_RTM_PORTUGUESE_BRAZIL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2008_R2_SP1_PORTUGESE_BRAZIL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2008_R2_SP1_PORTUGUESE_BRAZIL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2008_SP2_PORTUGESE_BRAZIL_32BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2008_SP2_PORTUGUESE_BRAZIL_32BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_RTM_PORTUGESE_PORTUGAL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_RTM_PORTUGUESE_PORTUGAL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2019_PORTUGESE_BRAZIL_FULL_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2019_PORTUGUESE_BRAZIL_FULL_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2019_PORTUGESE_PORTUGAL_FULL_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2019_PORTUGUESE_PORTUGAL_FULL_BASE,
};
exports.WindowsImage = WindowsImage;
/**
 * CPU type
 */
var AmazonLinuxCpuType;
(function (AmazonLinuxCpuType) {
    /**
     * arm64 CPU type
     */
    AmazonLinuxCpuType["ARM_64"] = "arm64";
    /**
     * x86_64 CPU type
     */
    AmazonLinuxCpuType["X86_64"] = "x86_64";
})(AmazonLinuxCpuType = exports.AmazonLinuxCpuType || (exports.AmazonLinuxCpuType = {}));
/**
 * Selects the latest version of Amazon Linux
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class AmazonLinuxImage extends GenericSSMParameterImage {
    /**
     * Return the SSM parameter name that will contain the Amazon Linux image with the given attributes
     */
    static ssmParameterName(props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_AmazonLinuxImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.ssmParameterName);
            }
            throw error;
        }
        const generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        const edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        const cpu = (props && props.cpuType) || AmazonLinuxCpuType.X86_64;
        let kernel = (props && props.kernel) || undefined;
        let virtualization;
        let storage;
        if (generation === AmazonLinuxGeneration.AMAZON_LINUX_2022) {
            kernel = AmazonLinuxKernel.KERNEL5_X;
            if (props && props.storage) {
                throw new Error('Storage parameter does not exist in smm parameter name for Amazon Linux 2022.');
            }
            if (props && props.virtualization) {
                throw new Error('Virtualization parameter does not exist in smm parameter name for Amazon Linux 2022.');
            }
        }
        else {
            virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
            storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
        }
        const parts = [
            generation,
            'ami',
            edition !== AmazonLinuxEdition.STANDARD ? edition : undefined,
            kernel,
            virtualization,
            cpu,
            storage,
        ].filter(x => x !== undefined); // Get rid of undefineds
        return '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
    }
    constructor(props = {}) {
        super(AmazonLinuxImage.ssmParameterName(props), OperatingSystemType.LINUX, props.userData);
        this.props = props;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_AmazonLinuxImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AmazonLinuxImage);
            }
            throw error;
        }
        this.cachedInContext = props.cachedInContext ?? false;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const imageId = lookupImage(scope, this.cachedInContext, this.parameterName);
        const osType = OperatingSystemType.LINUX;
        return {
            imageId,
            osType,
            userData: this.props.userData ?? user_data_1.UserData.forLinux(),
        };
    }
}
_d = JSII_RTTI_SYMBOL_1;
AmazonLinuxImage[_d] = { fqn: "aws-cdk-lib.aws_ec2.AmazonLinuxImage", version: "2.74.0" };
exports.AmazonLinuxImage = AmazonLinuxImage;
/**
 * What generation of Amazon Linux to use
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    /**
     * Amazon Linux
     */
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    /**
     * Amazon Linux 2
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
    /**
     * Amazon Linux 2022
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2022"] = "al2022";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux Kernel
 */
var AmazonLinuxKernel;
(function (AmazonLinuxKernel) {
    /**
     * Standard edition
     */
    AmazonLinuxKernel["KERNEL5_X"] = "kernel-5.10";
})(AmazonLinuxKernel = exports.AmazonLinuxKernel || (exports.AmazonLinuxKernel = {}));
/**
 * Amazon Linux edition
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    /**
     * Standard edition
     */
    AmazonLinuxEdition["STANDARD"] = "standard";
    /**
     * Minimal edition
     */
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    /**
     * HVM virtualization (recommended)
     */
    AmazonLinuxVirt["HVM"] = "hvm";
    /**
     * PV virtualization
     */
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    /**
     * EBS-backed storage
     */
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "s3";
    /**
     * General Purpose-based storage (recommended)
     */
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
class GenericLinuxImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_GenericLinuxImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GenericLinuxImage);
            }
            throw error;
        }
    }
    getImage(scope) {
        const userData = this.props.userData ?? user_data_1.UserData.forLinux();
        const osType = OperatingSystemType.LINUX;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            const mapping = {};
            for (const [rgn, ami] of Object.entries(this.amiMap)) {
                mapping[rgn] = { ami };
            }
            const amiMap = new core_1.CfnMapping(scope, 'AmiMap', { mapping });
            return {
                imageId: amiMap.findInMap(core_1.Aws.REGION, 'ami'),
                userData,
                osType,
            };
        }
        const imageId = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!imageId) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId,
            userData,
            osType,
        };
    }
}
_e = JSII_RTTI_SYMBOL_1;
GenericLinuxImage[_e] = { fqn: "aws-cdk-lib.aws_ec2.GenericLinuxImage", version: "2.74.0" };
exports.GenericLinuxImage = GenericLinuxImage;
/**
 * Construct a Windows machine image from an AMI map
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
class GenericWindowsImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_GenericWindowsImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GenericWindowsImage);
            }
            throw error;
        }
    }
    getImage(scope) {
        const userData = this.props.userData ?? user_data_1.UserData.forWindows();
        const osType = OperatingSystemType.WINDOWS;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            const mapping = {};
            for (const [rgn, ami] of Object.entries(this.amiMap)) {
                mapping[rgn] = { ami };
            }
            const amiMap = new core_1.CfnMapping(scope, 'AmiMap', { mapping });
            return {
                imageId: amiMap.findInMap(core_1.Aws.REGION, 'ami'),
                userData,
                osType,
            };
        }
        const imageId = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!imageId) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId,
            userData,
            osType,
        };
    }
}
_f = JSII_RTTI_SYMBOL_1;
GenericWindowsImage[_f] = { fqn: "aws-cdk-lib.aws_ec2.GenericWindowsImage", version: "2.74.0" };
exports.GenericWindowsImage = GenericWindowsImage;
/**
 * The OS type of a particular image
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
    /**
     * Used when the type of the operating system is not known
     * (for example, for imported Auto-Scaling Groups).
     */
    OperatingSystemType[OperatingSystemType["UNKNOWN"] = 2] = "UNKNOWN";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 */
class LookupMachineImage {
    constructor(props) {
        this.props = props;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_LookupMachineImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LookupMachineImage);
            }
            throw error;
        }
    }
    getImage(scope) {
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        const osType = this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX;
        return {
            imageId: value,
            osType,
            userData: this.props.userData ?? user_data_1.UserData.forOperatingSystem(osType),
        };
    }
}
_g = JSII_RTTI_SYMBOL_1;
LookupMachineImage[_g] = { fqn: "aws-cdk-lib.aws_ec2.LookupMachineImage", version: "2.74.0" };
exports.LookupMachineImage = LookupMachineImage;
function lookupImage(scope, cachedInContext, parameterName) {
    return cachedInContext
        ? ssm.StringParameter.valueFromLookup(scope, parameterName)
        : ssm.StringParameter.valueForTypedStringParameterV2(scope, parameterName, ssm.ParameterValueType.AWS_EC2_IMAGE_ID);
}
//# sourceMappingURL=data:application/json;base64,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