"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PlacementGroup = exports.PlacementGroupStrategy = exports.PlacementGroupSpreadLevel = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * Determines how this placement group spreads instances
 */
var PlacementGroupSpreadLevel;
(function (PlacementGroupSpreadLevel) {
    /**
     * Host spread level placement groups are only available with AWS Outposts.
     * For host spread level placement groups, there are no restrictions for running instances per Outposts.
     *
     * @see https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups-outpost.html
     */
    PlacementGroupSpreadLevel["HOST"] = "host";
    /**
     * Each instance is launched on a separate rack.
     * Each has its own network and power source.
     * A rack spread placement group can span multiple Availability Zones in the same Region.
     * For rack spread level placement groups, you can have a maximum of seven running instances per Availability Zone per group.
     */
    PlacementGroupSpreadLevel["RACK"] = "rack";
})(PlacementGroupSpreadLevel = exports.PlacementGroupSpreadLevel || (exports.PlacementGroupSpreadLevel = {}));
/**
 * Which strategy to use when launching instances
 */
var PlacementGroupStrategy;
(function (PlacementGroupStrategy) {
    /**
     * Packs instances close together inside an Availability Zone.
     * This strategy enables workloads to achieve the low-latency network
     * performance necessary for tightly-coupled node-to-node communication that
     * is typical of high-performance computing (HPC) applications.
     */
    PlacementGroupStrategy["CLUSTER"] = "cluster";
    /**
     * Spreads your instances across logical partitions such that groups of instances
     * in one partition do not share the underlying hardware with groups of instances
     * in different partitions.
     *
     * This strategy is typically used by large distributed and replicated workloads,
     * such as Hadoop, Cassandra, and Kafka.
     */
    PlacementGroupStrategy["PARTITION"] = "partition";
    /**
     * Strictly places a small group of instances across distinct underlying hardware
     * to reduce correlated failures.
     */
    PlacementGroupStrategy["SPREAD"] = "spread";
})(PlacementGroupStrategy = exports.PlacementGroupStrategy || (exports.PlacementGroupStrategy = {}));
/**
 * Defines a placement group. Placement groups give you fine-grained control over
 * where your instances are provisioned.
 */
class PlacementGroup extends core_1.Resource {
    /**
     * Import a PlacementGroup by its arn
     */
    static fromPlacementGroupName(scope, id, placementGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.placementGroupName = placementGroupName;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: undefined,
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_PlacementGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, PlacementGroup);
            }
            throw error;
        }
        this.partitions = props?.partitions;
        this.spreadLevel = props?.spreadLevel;
        this.strategy = props?.strategy;
        if (this.partitions && this.strategy) {
            if (this.strategy !== PlacementGroupStrategy.PARTITION) {
                throw new Error(`PlacementGroup '${id}' can only specify 'partitions' with the 'PARTITION' strategy`);
            }
        }
        else if (this.partitions && !this.strategy) {
            this.strategy = PlacementGroupStrategy.PARTITION;
        }
        if (this.spreadLevel) {
            if (!this.strategy) {
                this.strategy = PlacementGroupStrategy.SPREAD;
            }
            if (this.strategy !== PlacementGroupStrategy.SPREAD) {
                throw new Error(`PlacementGroup '${id}' can only specify 'spreadLevel' with the 'SPREAD' strategy`);
            }
        }
        const resource = new ec2_generated_1.CfnPlacementGroup(this, 'Resource', {
            partitionCount: this.partitions,
            spreadLevel: this.spreadLevel,
            strategy: this.strategy,
        });
        this.placementGroupName = this.getResourceArnAttribute(resource.attrGroupName, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
PlacementGroup[_a] = { fqn: "aws-cdk-lib.aws_ec2.PlacementGroup", version: "2.74.0" };
exports.PlacementGroup = PlacementGroup;
//# sourceMappingURL=data:application/json;base64,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