"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateSubnet = exports.PublicSubnet = exports.RouterType = exports.Subnet = exports.Vpc = exports.DefaultInstanceTenancy = exports.SubnetType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("../../cloud-assembly-schema");
const core_1 = require("../../core");
const cxapi = require("../../cx-api");
const constructs_1 = require("constructs");
const client_vpn_endpoint_1 = require("./client-vpn-endpoint");
const ec2_generated_1 = require("./ec2.generated");
const ip_addresses_1 = require("./ip-addresses");
const nat_1 = require("./nat");
const network_acl_1 = require("./network-acl");
const subnet_1 = require("./subnet");
const util_1 = require("./util");
const vpc_endpoint_1 = require("./vpc-endpoint");
const vpc_flow_logs_1 = require("./vpc-flow-logs");
const vpn_1 = require("./vpn");
const VPC_SUBNET_SYMBOL = Symbol.for('@aws-cdk/aws-ec2.VpcSubnet');
const FAKE_AZ_NAME = 'fake-az';
/**
 * The type of Subnet
 */
var SubnetType;
(function (SubnetType) {
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC),
     * and as such, do not require NAT gateways.
     *
     * Isolated subnets can only connect to or be connected to from other
     * instances in the same VPC. A default VPC configuration will not include
     * isolated subnets.
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     */
    SubnetType["PRIVATE_ISOLATED"] = "Isolated";
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC),
     * and as such, do not require NAT gateways.
     *
     * Isolated subnets can only connect to or be connected to from other
     * instances in the same VPC. A default VPC configuration will not include
     * isolated subnets.
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     *
     * @deprecated use `SubnetType.PRIVATE_ISOLATED`
     */
    SubnetType["ISOLATED"] = "Deprecated_Isolated";
    /**
     * Subnet that routes to the internet, but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. Egress to the internet will
     * need to be provided.
     * NAT Gateway(s) are the default solution to providing this subnet type the ability to route Internet traffic.
     * If a NAT Gateway is not required or desired, set `natGateways:0` or use
     * `SubnetType.PRIVATE_ISOLATED` instead.
     *
     * By default, a NAT gateway is created in every public subnet for maximum availability.
     * Be aware that you will be charged for NAT gateways.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     */
    SubnetType["PRIVATE_WITH_EGRESS"] = "Private";
    /**
     * Subnet that routes to the internet (via a NAT gateway), but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. NAT Gateway(s) are
     * required with this subnet type to route the Internet traffic through.
     * If a NAT Gateway is not required or desired, use `SubnetType.PRIVATE_ISOLATED` instead.
     *
     * By default, a NAT gateway is created in every public subnet for maximum availability.
     * Be aware that you will be charged for NAT gateways.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     * @deprecated use `PRIVATE_WITH_EGRESS`
     */
    SubnetType["PRIVATE_WITH_NAT"] = "Deprecated_Private_NAT";
    /**
     * Subnet that routes to the internet, but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. NAT Gateway(s) are
     * required with this subnet type to route the Internet traffic through.
     * If a NAT Gateway is not required or desired, use `SubnetType.PRIVATE_ISOLATED` instead.
     *
     * By default, a NAT gateway is created in every public subnet for maximum availability.
     * Be aware that you will be charged for NAT gateways.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     *
     * @deprecated use `PRIVATE_WITH_EGRESS`
     */
    SubnetType["PRIVATE"] = "Deprecated_Private";
    /**
     * Subnet connected to the Internet
     *
     * Instances in a Public subnet can connect to the Internet and can be
     * connected to from the Internet as long as they are launched with public
     * IPs (controlled on the AutoScalingGroup or other constructs that launch
     * instances).
     *
     * Public subnets route outbound traffic via an Internet Gateway.
     */
    SubnetType["PUBLIC"] = "Public";
})(SubnetType = exports.SubnetType || (exports.SubnetType = {}));
/**
 * A new or imported VPC
 */
class VpcBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Dependencies for NAT connectivity
         *
         * @deprecated - This value is no longer used.
         */
        this.natDependencies = new Array();
        /**
         * If this is set to true, don't error out on trying to select subnets
         */
        this.incompleteSubnetDefinition = false;
    }
    /**
     * Returns IDs of selected subnets
     */
    selectSubnets(selection = {}) {
        const subnets = this.selectSubnetObjects(selection);
        const pubs = new Set(this.publicSubnets);
        return {
            subnetIds: subnets.map(s => s.subnetId),
            get availabilityZones() { return subnets.map(s => s.availabilityZone); },
            internetConnectivityEstablished: tap(new CompositeDependable(), d => subnets.forEach(s => d.add(s.internetConnectivityEstablished))),
            subnets,
            hasPublic: subnets.some(s => pubs.has(s)),
            isPendingLookup: this.incompleteSubnetDefinition,
        };
    }
    /**
     * Adds a VPN Gateway to this VPC
     */
    enableVpnGateway(options) {
        if (this.vpnGatewayId) {
            throw new Error('The VPN Gateway has already been enabled.');
        }
        const vpnGateway = new vpn_1.VpnGateway(this, 'VpnGateway', {
            amazonSideAsn: options.amazonSideAsn,
            type: vpn_1.VpnConnectionType.IPSEC_1,
        });
        this._vpnGatewayId = vpnGateway.gatewayId;
        const attachment = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCVPNGW', {
            vpcId: this.vpcId,
            vpnGatewayId: this._vpnGatewayId,
        });
        // Propagate routes on route tables associated with the right subnets
        const vpnRoutePropagation = options.vpnRoutePropagation ?? [{}];
        const routeTableIds = (0, util_1.allRouteTableIds)((0, util_1.flatten)(vpnRoutePropagation.map(s => this.selectSubnets(s).subnets)));
        if (routeTableIds.length === 0) {
            core_1.Annotations.of(this).addError(`enableVpnGateway: no subnets matching selection: '${JSON.stringify(vpnRoutePropagation)}'. Select other subnets to add routes to.`);
        }
        const routePropagation = new ec2_generated_1.CfnVPNGatewayRoutePropagation(this, 'RoutePropagation', {
            routeTableIds,
            vpnGatewayId: this._vpnGatewayId,
        });
        // The AWS::EC2::VPNGatewayRoutePropagation resource cannot use the VPN gateway
        // until it has successfully attached to the VPC.
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpn-gatewayrouteprop.html
        routePropagation.node.addDependency(attachment);
    }
    /**
     * Adds a new VPN connection to this VPC
     */
    addVpnConnection(id, options) {
        return new vpn_1.VpnConnection(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new client VPN endpoint to this VPC
     */
    addClientVpnEndpoint(id, options) {
        return new client_vpn_endpoint_1.ClientVpnEndpoint(this, id, {
            ...options,
            vpc: this,
        });
    }
    /**
     * Adds a new interface endpoint to this VPC
     */
    addInterfaceEndpoint(id, options) {
        return new vpc_endpoint_1.InterfaceVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new gateway endpoint to this VPC
     */
    addGatewayEndpoint(id, options) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new flow log to this VPC
     */
    addFlowLog(id, options) {
        return new vpc_flow_logs_1.FlowLog(this, id, {
            resourceType: vpc_flow_logs_1.FlowLogResourceType.fromVpc(this),
            ...options,
        });
    }
    /**
     * Returns the id of the VPN Gateway (if enabled)
     */
    get vpnGatewayId() {
        return this._vpnGatewayId;
    }
    /**
     * Return the subnets appropriate for the placement strategy
     */
    selectSubnetObjects(selection = {}) {
        selection = this.reifySelectionDefaults(selection);
        if (selection.subnets !== undefined) {
            return selection.subnets;
        }
        let subnets;
        if (selection.subnetGroupName !== undefined) { // Select by name
            subnets = this.selectSubnetObjectsByName(selection.subnetGroupName);
        }
        else { // Or specify by type
            const type = selection.subnetType || SubnetType.PRIVATE_WITH_EGRESS;
            subnets = this.selectSubnetObjectsByType(type);
        }
        // Apply all the filters
        subnets = this.applySubnetFilters(subnets, selection.subnetFilters ?? []);
        return subnets;
    }
    applySubnetFilters(subnets, filters) {
        let filtered = subnets;
        // Apply each filter in sequence
        for (const filter of filters) {
            filtered = filter.selectSubnets(filtered);
        }
        return filtered;
    }
    selectSubnetObjectsByName(groupName) {
        const allSubnets = [...this.publicSubnets, ...this.privateSubnets, ...this.isolatedSubnets];
        const subnets = allSubnets.filter(s => (0, util_1.subnetGroupNameFromConstructId)(s) === groupName);
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const names = Array.from(new Set(allSubnets.map(util_1.subnetGroupNameFromConstructId)));
            throw new Error(`There are no subnet groups with name '${groupName}' in this VPC. Available names: ${names}`);
        }
        return subnets;
    }
    selectSubnetObjectsByType(subnetType) {
        const allSubnets = {
            [SubnetType.PRIVATE_ISOLATED]: this.isolatedSubnets,
            [SubnetType.ISOLATED]: this.isolatedSubnets,
            [SubnetType.PRIVATE_WITH_NAT]: this.privateSubnets,
            [SubnetType.PRIVATE_WITH_EGRESS]: this.privateSubnets,
            [SubnetType.PRIVATE]: this.privateSubnets,
            [SubnetType.PUBLIC]: this.publicSubnets,
        };
        const subnets = allSubnets[subnetType];
        // Force merge conflict here with https://github.com/aws/aws-cdk/pull/4089
        // see ImportedVpc
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const availableTypes = Object.entries(allSubnets).filter(([_, subs]) => subs.length > 0).map(([typeName, _]) => typeName);
            throw new Error(`There are no '${subnetType}' subnet groups in this VPC. Available types: ${availableTypes}`);
        }
        return subnets;
    }
    /**
     * Validate the fields in a SubnetSelection object, and reify defaults if necessary
     *
     * In case of default selection, select the first type of PRIVATE, ISOLATED,
     * PUBLIC (in that order) that has any subnets.
     */
    reifySelectionDefaults(placement) {
        if (placement.subnetName !== undefined) {
            if (placement.subnetGroupName !== undefined) {
                throw new Error('Please use only \'subnetGroupName\' (\'subnetName\' is deprecated and has the same behavior)');
            }
            else {
                core_1.Annotations.of(this).addWarning('Usage of \'subnetName\' in SubnetSelection is deprecated, use \'subnetGroupName\' instead');
            }
            placement = { ...placement, subnetGroupName: placement.subnetName };
        }
        const exclusiveSelections = ['subnets', 'subnetType', 'subnetGroupName'];
        const providedSelections = exclusiveSelections.filter(key => placement[key] !== undefined);
        if (providedSelections.length > 1) {
            throw new Error(`Only one of '${providedSelections}' can be supplied to subnet selection.`);
        }
        if (placement.subnetType === undefined && placement.subnetGroupName === undefined && placement.subnets === undefined) {
            // Return default subnet type based on subnets that actually exist
            let subnetType = this.privateSubnets.length
                ? SubnetType.PRIVATE_WITH_EGRESS : this.isolatedSubnets.length ? SubnetType.PRIVATE_ISOLATED : SubnetType.PUBLIC;
            placement = { ...placement, subnetType: subnetType };
        }
        // Establish which subnet filters are going to be used
        let subnetFilters = placement.subnetFilters ?? [];
        // Backwards compatibility with existing `availabilityZones` and `onePerAz` functionality
        if (placement.availabilityZones !== undefined) { // Filter by AZs, if specified
            subnetFilters.push(subnet_1.SubnetFilter.availabilityZones(placement.availabilityZones));
        }
        if (!!placement.onePerAz) { // Ensure one per AZ if specified
            subnetFilters.push(subnet_1.SubnetFilter.onePerAz());
        }
        // Overwrite the provided placement filters and remove the availabilityZones and onePerAz properties
        placement = { ...placement, subnetFilters: subnetFilters, availabilityZones: undefined, onePerAz: undefined };
        const { availabilityZones, onePerAz, ...rest } = placement;
        return rest;
    }
}
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The default tenancy of instances launched into the VPC.
 */
var DefaultInstanceTenancy;
(function (DefaultInstanceTenancy) {
    /**
     * Instances can be launched with any tenancy.
     */
    DefaultInstanceTenancy["DEFAULT"] = "default";
    /**
     * Any instance launched into the VPC automatically has dedicated tenancy, unless you launch it with the default tenancy.
     */
    DefaultInstanceTenancy["DEDICATED"] = "dedicated";
})(DefaultInstanceTenancy = exports.DefaultInstanceTenancy || (exports.DefaultInstanceTenancy = {}));
/**
 * Define an AWS Virtual Private Cloud
 *
 * See the package-level documentation of this package for an overview
 * of the various dimensions in which you can configure your VPC.
 *
 * For example:
 *
 * ```ts
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   ipAddresses: IpAddresses.cidr('10.0.0.0/16'),
 * })
 *
 * // Iterate the private subnets
 * const selection = vpc.selectSubnets({
 *   subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS
 * });
 *
 * for (const subnet of selection.subnets) {
 *   // ...
 * }
 * ```
 *
 * @resource AWS::EC2::VPC
 */
class Vpc extends VpcBase {
    /**
     * Import a VPC by supplying all attributes directly
     *
     * NOTE: using `fromVpcAttributes()` with deploy-time parameters (like a `Fn.importValue()` or
     * `CfnParameter` to represent a list of subnet IDs) sometimes accidentally works. It happens
     * to work for constructs that need a list of subnets (like `AutoScalingGroup` and `eks.Cluster`)
     * but it does not work for constructs that need individual subnets (like
     * `Instance`). See https://github.com/aws/aws-cdk/issues/4118 for more
     * information.
     *
     * Prefer to use `Vpc.fromLookup()` instead.
     */
    static fromVpcAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_VpcAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromVpcAttributes);
            }
            throw error;
        }
        return new ImportedVpc(scope, id, attrs, false);
    }
    /**
     * Import an existing VPC by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use VPCs not defined in your CDK
     * application. If you are looking to share a VPC between stacks, you can
     * pass the `Vpc` object between stacks and use it as normal.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     *
     * The VPC information will be cached in `cdk.context.json` and the same VPC
     * will be used on future runs. To refresh the lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static fromLookup(scope, id, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_VpcLookupOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromLookup);
            }
            throw error;
        }
        if (core_1.Token.isUnresolved(options.vpcId)
            || core_1.Token.isUnresolved(options.vpcName)
            || Object.values(options.tags || {}).some(core_1.Token.isUnresolved)
            || Object.keys(options.tags || {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to Vpc.fromLookup() must be concrete (no Tokens)');
        }
        const filter = makeTagFilter(options.tags);
        // We give special treatment to some tags
        if (options.vpcId) {
            filter['vpc-id'] = options.vpcId;
        }
        if (options.vpcName) {
            filter['tag:Name'] = options.vpcName;
        }
        if (options.ownerAccountId) {
            filter['owner-id'] = options.ownerAccountId;
        }
        if (options.isDefault !== undefined) {
            filter.isDefault = options.isDefault ? 'true' : 'false';
        }
        const overrides = {};
        if (options.region) {
            overrides.region = options.region;
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.VPC_PROVIDER,
            props: {
                ...overrides,
                filter,
                returnAsymmetricSubnets: true,
                returnVpnGateways: options.returnVpnGateways,
                subnetGroupNameTag: options.subnetGroupNameTag,
            },
            dummyValue: undefined,
        }).value;
        return new LookedUpVpc(scope, id, attributes ?? DUMMY_VPC_PROPS, attributes === undefined);
        /**
         * Prefixes all keys in the argument with `tag:`.`
         */
        function makeTagFilter(tags) {
            const result = {};
            for (const [name, value] of Object.entries(tags || {})) {
                result[`tag:${name}`] = value;
            }
            return result;
        }
    }
    /**
     * Vpc creates a VPC that spans a whole region.
     * It will automatically divide the provided VPC CIDR range, and create public and private subnets per Availability Zone.
     * Network routing for the public subnets will be configured to allow outbound access directly via an Internet Gateway.
     * Network routing for the private subnets will be configured to allow outbound access via a set of resilient NAT Gateways (one per AZ).
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * List of public subnets in this VPC
         */
        this.publicSubnets = [];
        /**
         * List of private subnets in this VPC
         */
        this.privateSubnets = [];
        /**
         * List of isolated subnets in this VPC
         */
        this.isolatedSubnets = [];
        /**
         * Subnet configurations for this VPC
         */
        this.subnetConfiguration = [];
        this._internetConnectivityEstablished = new constructs_1.DependencyGroup();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_VpcProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Vpc);
            }
            throw error;
        }
        const stack = core_1.Stack.of(this);
        // Can't have enabledDnsHostnames without enableDnsSupport
        if (props.enableDnsHostnames && !props.enableDnsSupport) {
            throw new Error('To use DNS Hostnames, DNS Support must be enabled, however, it was explicitly disabled.');
        }
        if (props.availabilityZones && props.maxAzs) {
            throw new Error('Vpc supports \'availabilityZones\' or \'maxAzs\', but not both.');
        }
        const cidrBlock = ifUndefined(props.cidr, Vpc.DEFAULT_CIDR_RANGE);
        if (core_1.Token.isUnresolved(cidrBlock)) {
            throw new Error('\'cidr\' property must be a concrete CIDR string, got a Token (we need to parse it for automatic subdivision)');
        }
        if (props.ipAddresses && props.cidr) {
            throw new Error('supply at most one of ipAddresses or cidr');
        }
        this.ipAddresses = props.ipAddresses ?? ip_addresses_1.IpAddresses.cidr(cidrBlock);
        this.dnsHostnamesEnabled = props.enableDnsHostnames == null ? true : props.enableDnsHostnames;
        this.dnsSupportEnabled = props.enableDnsSupport == null ? true : props.enableDnsSupport;
        const instanceTenancy = props.defaultInstanceTenancy || 'default';
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
        const vpcIpAddressOptions = this.ipAddresses.allocateVpcCidr();
        // Define a VPC using the provided CIDR range
        this.resource = new ec2_generated_1.CfnVPC(this, 'Resource', {
            cidrBlock: vpcIpAddressOptions.cidrBlock,
            ipv4IpamPoolId: vpcIpAddressOptions.ipv4IpamPoolId,
            ipv4NetmaskLength: vpcIpAddressOptions.ipv4NetmaskLength,
            enableDnsHostnames: this.dnsHostnamesEnabled,
            enableDnsSupport: this.dnsSupportEnabled,
            instanceTenancy,
        });
        this.vpcDefaultNetworkAcl = this.resource.attrDefaultNetworkAcl;
        this.vpcCidrBlockAssociations = this.resource.attrCidrBlockAssociations;
        this.vpcCidrBlock = this.resource.attrCidrBlock;
        this.vpcDefaultSecurityGroup = this.resource.attrDefaultSecurityGroup;
        this.vpcIpv6CidrBlocks = this.resource.attrIpv6CidrBlocks;
        core_1.Tags.of(this).add(NAME_TAG, props.vpcName || this.node.path);
        if (props.availabilityZones) {
            // If given AZs and stack AZs are both resolved, then validate their compatibility.
            const resolvedStackAzs = stack.availabilityZones.filter(az => !core_1.Token.isUnresolved(az));
            const areGivenAzsSubsetOfStack = resolvedStackAzs.length === 0 // stack AZs are tokenized, so we cannot validate it
                || props.availabilityZones.every(az => core_1.Token.isUnresolved(az) // given AZ is tokenized, such as in integ tests, so we cannot validate it
                    || resolvedStackAzs.includes(az));
            if (!areGivenAzsSubsetOfStack) {
                throw new Error(`Given VPC 'availabilityZones' ${props.availabilityZones} must be a subset of the stack's availability zones ${stack.availabilityZones}`);
            }
            this.availabilityZones = props.availabilityZones;
        }
        else {
            const maxAZs = props.maxAzs ?? 3;
            this.availabilityZones = stack.availabilityZones.slice(0, maxAZs);
        }
        for (let i = 0; props.reservedAzs && i < props.reservedAzs; i++) {
            this.availabilityZones.push(FAKE_AZ_NAME);
        }
        this.vpcId = this.resource.ref;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
        }, stack);
        const defaultSubnet = props.natGateways === 0 ? Vpc.DEFAULT_SUBNETS_NO_NAT : Vpc.DEFAULT_SUBNETS;
        this.subnetConfiguration = ifUndefined(props.subnetConfiguration, defaultSubnet);
        const natGatewayPlacement = props.natGatewaySubnets || { subnetType: SubnetType.PUBLIC };
        const natGatewayCount = determineNatGatewayCount(props.natGateways, this.subnetConfiguration, this.availabilityZones.length);
        // subnetConfiguration must be set before calling createSubnets
        this.createSubnets();
        const allowOutbound = this.subnetConfiguration.filter(subnet => (subnet.subnetType !== SubnetType.PRIVATE_ISOLATED && subnet.subnetType !== SubnetType.ISOLATED)).length > 0;
        // Create an Internet Gateway and attach it if necessary
        if (allowOutbound) {
            const igw = new ec2_generated_1.CfnInternetGateway(this, 'IGW', {});
            this.internetGatewayId = igw.ref;
            this._internetConnectivityEstablished.add(igw);
            const att = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCGW', {
                internetGatewayId: igw.ref,
                vpcId: this.resource.ref,
            });
            this.publicSubnets.forEach(publicSubnet => {
                publicSubnet.addDefaultInternetRoute(igw.ref, att);
            });
            // if gateways are needed create them
            if (natGatewayCount > 0) {
                const provider = props.natGatewayProvider || nat_1.NatProvider.gateway();
                this.createNatGateways(provider, natGatewayCount, natGatewayPlacement);
            }
        }
        if (props.vpnGateway && this.publicSubnets.length === 0 && this.privateSubnets.length === 0 && this.isolatedSubnets.length === 0) {
            throw new Error('Can not enable the VPN gateway while the VPC has no subnets at all');
        }
        if ((props.vpnConnections || props.vpnGatewayAsn) && props.vpnGateway === false) {
            throw new Error('Cannot specify `vpnConnections` or `vpnGatewayAsn` when `vpnGateway` is set to false.');
        }
        if (props.vpnGateway || props.vpnConnections || props.vpnGatewayAsn) {
            this.enableVpnGateway({
                amazonSideAsn: props.vpnGatewayAsn,
                type: vpn_1.VpnConnectionType.IPSEC_1,
                vpnRoutePropagation: props.vpnRoutePropagation,
            });
            const vpnConnections = props.vpnConnections || {};
            for (const [connectionId, connection] of Object.entries(vpnConnections)) {
                this.addVpnConnection(connectionId, connection);
            }
        }
        // Allow creation of gateway endpoints on VPC instantiation as those can be
        // immediately functional without further configuration. This is not the case
        // for interface endpoints where the security group must be configured.
        if (props.gatewayEndpoints) {
            const gatewayEndpoints = props.gatewayEndpoints || {};
            for (const [endpointId, endpoint] of Object.entries(gatewayEndpoints)) {
                this.addGatewayEndpoint(endpointId, endpoint);
            }
        }
        // Add flow logs to the VPC
        if (props.flowLogs) {
            const flowLogs = props.flowLogs || {};
            for (const [flowLogId, flowLog] of Object.entries(flowLogs)) {
                this.addFlowLog(flowLogId, flowLog);
            }
        }
    }
    /**
     * Adds a new S3 gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addS3Endpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.S3,
            vpc: this,
            subnets,
        });
    }
    /**
     * Adds a new DynamoDB gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addDynamoDbEndpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.DYNAMODB,
            vpc: this,
            subnets,
        });
    }
    createNatGateways(provider, natCount, placement) {
        const natSubnets = this.selectSubnetObjects(placement);
        for (const sub of natSubnets) {
            if (this.publicSubnets.indexOf(sub) === -1) {
                throw new Error(`natGatewayPlacement ${placement} contains non public subnet ${sub}`);
            }
        }
        provider.configureNat({
            vpc: this,
            natSubnets: natSubnets.slice(0, natCount),
            privateSubnets: this.privateSubnets,
        });
    }
    /**
     * createSubnets creates the subnets specified by the subnet configuration
     * array or creates the `DEFAULT_SUBNETS` configuration
     */
    createSubnets() {
        const requestedSubnets = [];
        this.subnetConfiguration.forEach((configuration) => (this.availabilityZones.forEach((az, index) => {
            requestedSubnets.push({
                availabilityZone: az,
                subnetConstructId: (0, util_1.subnetId)(configuration.name, index),
                configuration,
            });
        })));
        const { allocatedSubnets } = this.ipAddresses.allocateSubnetsCidr({
            vpcCidr: this.vpcCidrBlock,
            requestedSubnets,
        });
        if (allocatedSubnets.length != requestedSubnets.length) {
            throw new Error('Incomplete Subnet Allocation; response array dose not equal input array');
        }
        this.createSubnetResources(requestedSubnets, allocatedSubnets);
    }
    createSubnetResources(requestedSubnets, allocatedSubnets) {
        allocatedSubnets.forEach((allocated, i) => {
            const { configuration: subnetConfig, subnetConstructId, availabilityZone } = requestedSubnets[i];
            if (subnetConfig.reserved === true) {
                // For reserved subnets, do not create any resources
                return;
            }
            if (availabilityZone === FAKE_AZ_NAME) {
                // For reserved azs, do not create any resources
                return;
            }
            // mapPublicIpOnLaunch true in Subnet.Public, false in Subnet.Private or Subnet.Isolated.
            let mapPublicIpOnLaunch = false;
            if (subnetConfig.subnetType !== SubnetType.PUBLIC && subnetConfig.mapPublicIpOnLaunch !== undefined) {
                throw new Error(`${subnetConfig.subnetType} subnet cannot include mapPublicIpOnLaunch parameter`);
            }
            if (subnetConfig.subnetType === SubnetType.PUBLIC) {
                mapPublicIpOnLaunch = (subnetConfig.mapPublicIpOnLaunch !== undefined)
                    ? subnetConfig.mapPublicIpOnLaunch
                    : true;
            }
            const subnetProps = {
                availabilityZone,
                vpcId: this.vpcId,
                cidrBlock: allocated.cidr,
                mapPublicIpOnLaunch: mapPublicIpOnLaunch,
            };
            let subnet;
            switch (subnetConfig.subnetType) {
                case SubnetType.PUBLIC:
                    const publicSubnet = new PublicSubnet(this, subnetConstructId, subnetProps);
                    this.publicSubnets.push(publicSubnet);
                    subnet = publicSubnet;
                    break;
                case SubnetType.PRIVATE_WITH_EGRESS:
                case SubnetType.PRIVATE_WITH_NAT:
                case SubnetType.PRIVATE:
                    const privateSubnet = new PrivateSubnet(this, subnetConstructId, subnetProps);
                    this.privateSubnets.push(privateSubnet);
                    subnet = privateSubnet;
                    break;
                case SubnetType.PRIVATE_ISOLATED:
                case SubnetType.ISOLATED:
                    const isolatedSubnet = new PrivateSubnet(this, subnetConstructId, subnetProps);
                    this.isolatedSubnets.push(isolatedSubnet);
                    subnet = isolatedSubnet;
                    break;
                default:
                    throw new Error(`Unrecognized subnet type: ${subnetConfig.subnetType}`);
            }
            // These values will be used to recover the config upon provider import
            const includeResourceTypes = [ec2_generated_1.CfnSubnet.CFN_RESOURCE_TYPE_NAME];
            core_1.Tags.of(subnet).add(SUBNETNAME_TAG, subnetConfig.name, { includeResourceTypes });
            core_1.Tags.of(subnet).add(SUBNETTYPE_TAG, subnetTypeTagValue(subnetConfig.subnetType), { includeResourceTypes });
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
Vpc[_a] = { fqn: "aws-cdk-lib.aws_ec2.Vpc", version: "2.74.0" };
/**
 * The default CIDR range used when creating VPCs.
 * This can be overridden using VpcProps when creating a VPCNetwork resource.
 * e.g. new VpcResource(this, { cidr: '192.168.0.0./16' })
 */
Vpc.DEFAULT_CIDR_RANGE = '10.0.0.0/16';
/**
 * The default subnet configuration
 *
 * 1 Public and 1 Private subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS = [
    {
        subnetType: SubnetType.PUBLIC,
        name: (0, util_1.defaultSubnetName)(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE_WITH_EGRESS,
        name: (0, util_1.defaultSubnetName)(SubnetType.PRIVATE_WITH_EGRESS),
    },
];
/**
 * The default subnet configuration if natGateways specified to be 0
 *
 * 1 Public and 1 Isolated Subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS_NO_NAT = [
    {
        subnetType: SubnetType.PUBLIC,
        name: (0, util_1.defaultSubnetName)(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE_ISOLATED,
        name: (0, util_1.defaultSubnetName)(SubnetType.PRIVATE_ISOLATED),
    },
];
exports.Vpc = Vpc;
const SUBNETTYPE_TAG = 'aws-cdk:subnet-type';
const SUBNETNAME_TAG = 'aws-cdk:subnet-name';
function subnetTypeTagValue(type) {
    switch (type) {
        case SubnetType.PUBLIC: return 'Public';
        case SubnetType.PRIVATE_WITH_EGRESS:
        case SubnetType.PRIVATE_WITH_NAT:
        case SubnetType.PRIVATE:
            return 'Private';
        case SubnetType.PRIVATE_ISOLATED:
        case SubnetType.ISOLATED:
            return 'Isolated';
    }
}
/**
 * Represents a new VPC subnet resource
 *
 * @resource AWS::EC2::Subnet
 */
class Subnet extends core_1.Resource {
    static isVpcSubnet(x) {
        return VPC_SUBNET_SYMBOL in x;
    }
    static fromSubnetAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_SubnetAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSubnetAttributes);
            }
            throw error;
        }
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * Import existing subnet from id.
     */
    // eslint-disable-next-line @typescript-eslint/no-shadow
    static fromSubnetId(scope, id, subnetId) {
        return this.fromSubnetAttributes(scope, id, { subnetId });
    }
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Parts of this VPC subnet
         */
        this.dependencyElements = [];
        this._internetConnectivityEstablished = new constructs_1.DependencyGroup();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_SubnetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Subnet);
            }
            throw error;
        }
        Object.defineProperty(this, VPC_SUBNET_SYMBOL, { value: true });
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.availabilityZone = props.availabilityZone;
        this.ipv4CidrBlock = props.cidrBlock;
        const subnet = new ec2_generated_1.CfnSubnet(this, 'Subnet', {
            vpcId: props.vpcId,
            cidrBlock: props.cidrBlock,
            availabilityZone: props.availabilityZone,
            mapPublicIpOnLaunch: props.mapPublicIpOnLaunch,
        });
        this.subnetId = subnet.ref;
        this.subnetVpcId = subnet.attrVpcId;
        this.subnetAvailabilityZone = subnet.attrAvailabilityZone;
        this.subnetIpv6CidrBlocks = subnet.attrIpv6CidrBlocks;
        this.subnetOutpostArn = subnet.attrOutpostArn;
        // subnet.attrNetworkAclAssociationId is the default ACL after the subnet
        // was just created. However, the ACL can be replaced at a later time.
        this._networkAcl = network_acl_1.NetworkAcl.fromNetworkAclId(this, 'Acl', subnet.attrNetworkAclAssociationId);
        this.subnetNetworkAclAssociationId = core_1.Lazy.string({ produce: () => this._networkAcl.networkAclId });
        this.node.defaultChild = subnet;
        const table = new ec2_generated_1.CfnRouteTable(this, 'RouteTable', {
            vpcId: props.vpcId,
        });
        this.routeTable = { routeTableId: table.ref };
        // Associate the public route table for this subnet, to this subnet
        const routeAssoc = new ec2_generated_1.CfnSubnetRouteTableAssociation(this, 'RouteTableAssociation', {
            subnetId: this.subnetId,
            routeTableId: table.ref,
        });
        this._internetConnectivityEstablished.add(routeAssoc);
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
    }
    /**
     * Create a default route that points to a passed IGW, with a dependency
     * on the IGW's attachment to the VPC.
     *
     * @param gatewayId the logical ID (ref) of the gateway attached to your VPC
     * @param gatewayAttachment the gateway attachment construct to be added as a dependency
     */
    addDefaultInternetRoute(gatewayId, gatewayAttachment) {
        const route = new ec2_generated_1.CfnRoute(this, 'DefaultRoute', {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: '0.0.0.0/0',
            gatewayId,
        });
        route.node.addDependency(gatewayAttachment);
        // Since the 'route' depends on the gateway attachment, just
        // depending on the route is enough.
        this._internetConnectivityEstablished.add(route);
    }
    /**
     * Network ACL associated with this Subnet
     *
     * Upon creation, this is the default ACL which allows all traffic, except
     * explicit DENY entries that you add.
     *
     * You can replace it with a custom ACL which denies all traffic except
     * the explicit ALLOW entries that you add by creating a `NetworkAcl`
     * object and calling `associateNetworkAcl()`.
     */
    get networkAcl() {
        return this._networkAcl;
    }
    /**
     * Adds an entry to this subnets route table that points to the passed NATGatewayId
     * @param natGatewayId The ID of the NAT gateway
     */
    addDefaultNatRoute(natGatewayId) {
        this.addRoute('DefaultRoute', {
            routerType: RouterType.NAT_GATEWAY,
            routerId: natGatewayId,
            enablesInternetConnectivity: true,
        });
    }
    /**
     * Adds an entry to this subnets route table
     */
    addRoute(id, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_AddRouteOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRoute);
            }
            throw error;
        }
        if (options.destinationCidrBlock && options.destinationIpv6CidrBlock) {
            throw new Error('Cannot specify both \'destinationCidrBlock\' and \'destinationIpv6CidrBlock\'');
        }
        const route = new ec2_generated_1.CfnRoute(this, id, {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: options.destinationCidrBlock || (options.destinationIpv6CidrBlock === undefined ? '0.0.0.0/0' : undefined),
            destinationIpv6CidrBlock: options.destinationIpv6CidrBlock,
            [routerTypeToPropName(options.routerType)]: options.routerId,
        });
        if (options.enablesInternetConnectivity) {
            this._internetConnectivityEstablished.add(route);
        }
    }
    associateNetworkAcl(id, networkAcl) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_INetworkAcl(networkAcl);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.associateNetworkAcl);
            }
            throw error;
        }
        this._networkAcl = networkAcl;
        const scope = networkAcl instanceof constructs_1.Construct ? networkAcl : this;
        const other = networkAcl instanceof constructs_1.Construct ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + core_1.Names.nodeUniqueId(other.node), {
            networkAcl,
            subnet: this,
        });
    }
}
_b = JSII_RTTI_SYMBOL_1;
Subnet[_b] = { fqn: "aws-cdk-lib.aws_ec2.Subnet", version: "2.74.0" };
exports.Subnet = Subnet;
/**
 * Type of router used in route
 */
var RouterType;
(function (RouterType) {
    /**
     * Carrier gateway
     */
    RouterType["CARRIER_GATEWAY"] = "CarrierGateway";
    /**
     * Egress-only Internet Gateway
     */
    RouterType["EGRESS_ONLY_INTERNET_GATEWAY"] = "EgressOnlyInternetGateway";
    /**
     * Internet Gateway
     */
    RouterType["GATEWAY"] = "Gateway";
    /**
     * Instance
     */
    RouterType["INSTANCE"] = "Instance";
    /**
     * Local Gateway
     */
    RouterType["LOCAL_GATEWAY"] = "LocalGateway";
    /**
     * NAT Gateway
     */
    RouterType["NAT_GATEWAY"] = "NatGateway";
    /**
     * Network Interface
     */
    RouterType["NETWORK_INTERFACE"] = "NetworkInterface";
    /**
     * Transit Gateway
     */
    RouterType["TRANSIT_GATEWAY"] = "TransitGateway";
    /**
     * VPC peering connection
     */
    RouterType["VPC_PEERING_CONNECTION"] = "VpcPeeringConnection";
    /**
     * VPC Endpoint for gateway load balancers
     */
    RouterType["VPC_ENDPOINT"] = "VpcEndpoint";
})(RouterType = exports.RouterType || (exports.RouterType = {}));
function routerTypeToPropName(routerType) {
    return ({
        [RouterType.CARRIER_GATEWAY]: 'carrierGatewayId',
        [RouterType.EGRESS_ONLY_INTERNET_GATEWAY]: 'egressOnlyInternetGatewayId',
        [RouterType.GATEWAY]: 'gatewayId',
        [RouterType.INSTANCE]: 'instanceId',
        [RouterType.LOCAL_GATEWAY]: 'localGatewayId',
        [RouterType.NAT_GATEWAY]: 'natGatewayId',
        [RouterType.NETWORK_INTERFACE]: 'networkInterfaceId',
        [RouterType.TRANSIT_GATEWAY]: 'transitGatewayId',
        [RouterType.VPC_PEERING_CONNECTION]: 'vpcPeeringConnectionId',
        [RouterType.VPC_ENDPOINT]: 'vpcEndpointId',
    })[routerType];
}
/**
 * Represents a public VPC subnet resource
 */
class PublicSubnet extends Subnet {
    static fromPublicSubnetAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_PublicSubnetAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromPublicSubnetAttributes);
            }
            throw error;
        }
        return new ImportedSubnet(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_PublicSubnetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, PublicSubnet);
            }
            throw error;
        }
    }
    /**
     * Creates a new managed NAT gateway attached to this public subnet.
     * Also adds the EIP for the managed NAT.
     * @returns A ref to the the NAT Gateway ID
     */
    addNatGateway(eipAllocationId) {
        // Create a NAT Gateway in this public subnet
        const ngw = new ec2_generated_1.CfnNatGateway(this, 'NATGateway', {
            subnetId: this.subnetId,
            allocationId: eipAllocationId ?? new ec2_generated_1.CfnEIP(this, 'EIP', {
                domain: 'vpc',
            }).attrAllocationId,
        });
        ngw.node.addDependency(this.internetConnectivityEstablished);
        return ngw;
    }
}
_c = JSII_RTTI_SYMBOL_1;
PublicSubnet[_c] = { fqn: "aws-cdk-lib.aws_ec2.PublicSubnet", version: "2.74.0" };
exports.PublicSubnet = PublicSubnet;
/**
 * Represents a private VPC subnet resource
 */
class PrivateSubnet extends Subnet {
    static fromPrivateSubnetAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_PrivateSubnetAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromPrivateSubnetAttributes);
            }
            throw error;
        }
        return new ImportedSubnet(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_PrivateSubnetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, PrivateSubnet);
            }
            throw error;
        }
    }
}
_d = JSII_RTTI_SYMBOL_1;
PrivateSubnet[_d] = { fqn: "aws-cdk-lib.aws_ec2.PrivateSubnet", version: "2.74.0" };
exports.PrivateSubnet = PrivateSubnet;
function ifUndefined(value, defaultValue) {
    return value ?? defaultValue;
}
class ImportedVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id, {
            region: props.region,
        });
        this.internetConnectivityEstablished = new constructs_1.DependencyGroup();
        this.vpcId = props.vpcId;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
        }, core_1.Stack.of(this));
        this.cidr = props.vpcCidrBlock;
        this.availabilityZones = props.availabilityZones;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        // None of the values may be unresolved list tokens
        for (const k of Object.keys(props)) {
            if (Array.isArray(props[k]) && core_1.Token.isUnresolved(props[k])) {
                core_1.Annotations.of(this).addWarning(`fromVpcAttributes: '${k}' is a list token: the imported VPC will not work with constructs that require a list of subnets at synthesis time. Use 'Vpc.fromLookup()' or 'Fn.importListValue' instead.`);
            }
        }
        /* eslint-disable max-len */
        const pub = new util_1.ImportSubnetGroup(props.publicSubnetIds, props.publicSubnetNames, props.publicSubnetRouteTableIds, props.publicSubnetIpv4CidrBlocks, SubnetType.PUBLIC, this.availabilityZones, 'publicSubnetIds', 'publicSubnetNames', 'publicSubnetRouteTableIds', 'publicSubnetIpv4CidrBlocks');
        const priv = new util_1.ImportSubnetGroup(props.privateSubnetIds, props.privateSubnetNames, props.privateSubnetRouteTableIds, props.privateSubnetIpv4CidrBlocks, SubnetType.PRIVATE_WITH_EGRESS, this.availabilityZones, 'privateSubnetIds', 'privateSubnetNames', 'privateSubnetRouteTableIds', 'privateSubnetIpv4CidrBlocks');
        const iso = new util_1.ImportSubnetGroup(props.isolatedSubnetIds, props.isolatedSubnetNames, props.isolatedSubnetRouteTableIds, props.isolatedSubnetIpv4CidrBlocks, SubnetType.PRIVATE_ISOLATED, this.availabilityZones, 'isolatedSubnetIds', 'isolatedSubnetNames', 'isolatedSubnetRouteTableIds', 'isolatedSubnetIpv4CidrBlocks');
        /* eslint-enable max-len */
        this.publicSubnets = pub.import(this);
        this.privateSubnets = priv.import(this);
        this.isolatedSubnets = iso.import(this);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not supplied when creating this VPC');
        }
        return this.cidr;
    }
}
class LookedUpVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id, {
            region: props.region,
            account: props.ownerAccountId,
        });
        this.internetConnectivityEstablished = new constructs_1.DependencyGroup();
        this.vpcId = props.vpcId;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
            region: this.env.region,
            account: this.env.account,
        }, core_1.Stack.of(this));
        this.cidr = props.vpcCidrBlock;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        const subnetGroups = props.subnetGroups || [];
        const availabilityZones = Array.from(new Set(flatMap(subnetGroups, subnetGroup => {
            return subnetGroup.subnets.map(subnet => subnet.availabilityZone);
        })));
        availabilityZones.sort((az1, az2) => az1.localeCompare(az2));
        this.availabilityZones = availabilityZones;
        this.publicSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PUBLIC);
        this.privateSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PRIVATE);
        this.isolatedSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.ISOLATED);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            // Value might be cached from an old CLI version, so bumping the CX API protocol to
            // force the value to exist would not have helped.
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not found when looking up this VPC. Use a newer version of the CDK CLI and clear the old context value.');
        }
        return this.cidr;
    }
    extractSubnetsOfType(subnetGroups, subnetGroupType) {
        return flatMap(subnetGroups.filter(subnetGroup => subnetGroup.type === subnetGroupType), subnetGroup => this.subnetGroupToSubnets(subnetGroup));
    }
    subnetGroupToSubnets(subnetGroup) {
        const ret = new Array();
        for (let i = 0; i < subnetGroup.subnets.length; i++) {
            const vpcSubnet = subnetGroup.subnets[i];
            ret.push(Subnet.fromSubnetAttributes(this, `${subnetGroup.name}Subnet${i + 1}`, {
                availabilityZone: vpcSubnet.availabilityZone,
                subnetId: vpcSubnet.subnetId,
                routeTableId: vpcSubnet.routeTableId,
                ipv4CidrBlock: vpcSubnet.cidr,
            }));
        }
        return ret;
    }
}
function flatMap(xs, fn) {
    const ret = new Array();
    for (const x of xs) {
        ret.push(...fn(x));
    }
    return ret;
}
class CompositeDependable {
    constructor() {
        this.dependables = new Array();
        const self = this;
        constructs_1.Dependable.implement(this, {
            get dependencyRoots() {
                const ret = new Array();
                for (const dep of self.dependables) {
                    ret.push(...constructs_1.Dependable.of(dep).dependencyRoots);
                }
                return ret;
            },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(dep) {
        this.dependables.push(dep);
    }
}
/**
 * Invoke a function on a value (for its side effect) and return the value
 */
function tap(x, fn) {
    fn(x);
    return x;
}
class ImportedSubnet extends core_1.Resource {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.internetConnectivityEstablished = new constructs_1.DependencyGroup();
        if (!attrs.routeTableId) {
            // The following looks a little weird, but comes down to:
            //
            // * Is the subnetId itself unresolved ({ Ref: Subnet }); or
            // * Was it the accidentally extracted first element of a list-encoded
            //   token? ({ Fn::ImportValue: Subnets } => ['#{Token[1234]}'] =>
            //   '#{Token[1234]}'
            //
            // There's no other API to test for the second case than to the string back into
            // a list and see if the combination is Unresolved.
            //
            // In both cases we can't output the subnetId literally into the metadata (because it'll
            // be useless). In the 2nd case even, if we output it to metadata, the `resolve()` call
            // that gets done on the metadata will even `throw`, because the '#{Token}' value will
            // occur in an illegal position (not in a list context).
            const ref = core_1.Token.isUnresolved(attrs.subnetId) || core_1.Token.isUnresolved([attrs.subnetId])
                ? `at '${constructs_1.Node.of(scope).path}/${id}'`
                : `'${attrs.subnetId}'`;
            // eslint-disable-next-line max-len
            core_1.Annotations.of(this).addWarning(`No routeTableId was provided to the subnet ${ref}. Attempting to read its .routeTable.routeTableId will return null/undefined. (More info: https://github.com/aws/aws-cdk/pull/3171)`);
        }
        this._ipv4CidrBlock = attrs.ipv4CidrBlock;
        this._availabilityZone = attrs.availabilityZone;
        this.subnetId = attrs.subnetId;
        this.routeTable = {
            // Forcing routeTableId to pretend non-null to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            routeTableId: attrs.routeTableId,
        };
    }
    get availabilityZone() {
        if (!this._availabilityZone) {
            // eslint-disable-next-line max-len
            throw new Error('You cannot reference a Subnet\'s availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()');
        }
        return this._availabilityZone;
    }
    get ipv4CidrBlock() {
        if (!this._ipv4CidrBlock) {
            // tslint:disable-next-line: max-line-length
            throw new Error('You cannot reference an imported Subnet\'s IPv4 CIDR if it was not supplied. Add the ipv4CidrBlock when importing using Subnet.fromSubnetAttributes()');
        }
        return this._ipv4CidrBlock;
    }
    associateNetworkAcl(id, networkAcl) {
        const scope = networkAcl instanceof constructs_1.Construct ? networkAcl : this;
        const other = networkAcl instanceof constructs_1.Construct ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + core_1.Names.nodeUniqueId(other.node), {
            networkAcl,
            subnet: this,
        });
    }
}
/**
 * Determine (and validate) the NAT gateway count w.r.t. the rest of the subnet configuration
 *
 * We have the following requirements:
 *
 * - NatGatewayCount = 0 ==> there are no private subnets
 * - NatGatewayCount > 0 ==> there must be public subnets
 *
 * Do we want to require that there are private subnets if there are NatGateways?
 * They seem pointless but I see no reason to prevent it.
 */
function determineNatGatewayCount(requestedCount, subnetConfig, azCount) {
    const hasPrivateSubnets = subnetConfig.some(c => (c.subnetType === SubnetType.PRIVATE_WITH_EGRESS
        || c.subnetType === SubnetType.PRIVATE || c.subnetType === SubnetType.PRIVATE_WITH_NAT) && !c.reserved);
    const hasPublicSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PUBLIC);
    const hasCustomEgress = subnetConfig.some(c => c.subnetType === SubnetType.PRIVATE_WITH_EGRESS);
    const count = requestedCount !== undefined ? Math.min(requestedCount, azCount) : (hasPrivateSubnets ? azCount : 0);
    if (count === 0 && hasPrivateSubnets && !hasCustomEgress) {
        // eslint-disable-next-line max-len
        throw new Error('If you do not want NAT gateways (natGateways=0), make sure you don\'t configure any PRIVATE(_WITH_NAT) subnets in \'subnetConfiguration\' (make them PUBLIC or ISOLATED instead)');
    }
    if (count > 0 && !hasPublicSubnets) {
        // eslint-disable-next-line max-len
        throw new Error(`If you configure PRIVATE subnets in 'subnetConfiguration', you must also configure PUBLIC subnets to put the NAT gateways into (got ${JSON.stringify(subnetConfig)}.`);
    }
    return count;
}
/**
 * There are returned when the provider has not supplied props yet
 *
 * It's only used for testing and on the first run-through.
 */
const DUMMY_VPC_PROPS = {
    availabilityZones: [],
    vpcCidrBlock: '1.2.3.4/5',
    isolatedSubnetIds: undefined,
    isolatedSubnetNames: undefined,
    isolatedSubnetRouteTableIds: undefined,
    privateSubnetIds: undefined,
    privateSubnetNames: undefined,
    privateSubnetRouteTableIds: undefined,
    publicSubnetIds: undefined,
    publicSubnetNames: undefined,
    publicSubnetRouteTableIds: undefined,
    subnetGroups: [
        {
            name: 'Public',
            type: cxapi.VpcSubnetGroupType.PUBLIC,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 's-12345',
                    routeTableId: 'rtb-12345s',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 's-67890',
                    routeTableId: 'rtb-67890s',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Private',
            type: cxapi.VpcSubnetGroupType.PRIVATE,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Isolated',
            type: cxapi.VpcSubnetGroupType.ISOLATED,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
    ],
    vpcId: 'vpc-12345',
};
//# sourceMappingURL=data:application/json;base64,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