"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepositoryEncryption = exports.TagMutability = exports.Repository = exports.RepositoryBase = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const path = require("path");
const events = require("../../aws-events");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const ecr_generated_1 = require("./ecr.generated");
const lifecycle_1 = require("./lifecycle");
const AUTO_DELETE_IMAGES_RESOURCE_TYPE = 'Custom::ECRAutoDeleteImages';
const AUTO_DELETE_IMAGES_TAG = 'aws-cdk:auto-delete-images';
/**
 * Base class for ECR repository. Reused between imported repositories and owned repositories.
 */
class RepositoryBase extends core_1.Resource {
    /**
     * The URI of this repository (represents the latest image):
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     */
    get repositoryUri() {
        return this.repositoryUriForTag();
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Optional image tag
     */
    repositoryUriForTag(tag) {
        const tagSuffix = tag ? `:${tag}` : '';
        return this.repositoryUriWithSuffix(tagSuffix);
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[@DIGEST]
     *
     * @param digest Optional image digest
     */
    repositoryUriForDigest(digest) {
        const digestSuffix = digest ? `@${digest}` : '';
        return this.repositoryUriWithSuffix(digestSuffix);
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[@DIGEST]
     *
     * @param tagOrDigest Optional image tag or digest (digests must start with `sha256:`)
     */
    repositoryUriForTagOrDigest(tagOrDigest) {
        if (tagOrDigest?.startsWith('sha256:')) {
            return this.repositoryUriForDigest(tagOrDigest);
        }
        else {
            return this.repositoryUriForTag(tagOrDigest);
        }
    }
    /**
     * Returns the repository URI, with an appended suffix, if provided.
     * @param suffix An image tag or an image digest.
     * @private
     */
    repositoryUriWithSuffix(suffix) {
        const parts = this.stack.splitArn(this.repositoryArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        return `${parts.account}.dkr.ecr.${parts.region}.${this.stack.urlSuffix}/${this.repositoryName}${suffix}`;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_events_OnEventOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.onCloudTrailEvent);
            }
            throw error;
        }
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                requestParameters: {
                    repositoryName: [this.repositoryName],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this
     * repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailImagePushed(id, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecr_OnCloudTrailImagePushedOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.onCloudTrailImagePushed);
            }
            throw error;
        }
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutImage'],
                requestParameters: {
                    imageTag: options.imageTag ? [options.imageTag] : undefined,
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image scan is completed
     *
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onImageScanCompleted(id, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecr_OnImageScanCompletedOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.onImageScanCompleted);
            }
            throw error;
        }
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['ECR Image Scan'],
            detail: {
                'repository-name': [this.repositoryName],
                'scan-status': ['COMPLETE'],
                'image-tags': options.imageTags ?? undefined,
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for repository events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_events_OnEventOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.onEvent);
            }
            throw error;
        }
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.ecr'],
            resources: [this.repositoryArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Grant the given principal identity permissions to perform the actions on this repository
     */
    grant(grantee, ...actions) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grant);
            }
            throw error;
        }
        const crossAccountPrincipal = this.unsafeCrossAccountResourcePolicyPrincipal(grantee);
        if (crossAccountPrincipal) {
            // If the principal is from a different account,
            // that means addToPrincipalOrResource() will update the Resource Policy of this repo to trust that principal.
            // However, ECR verifies that the principal used in the Policy exists,
            // and will error out if it doesn't.
            // Because of that, if the principal is a newly created resource,
            // and there is not a dependency relationship between the Stacks of this repo and the principal,
            // trust the entire account of the principal instead
            // (otherwise, deploying this repo will fail).
            // To scope down the permissions as much as possible,
            // only trust principals from that account with a specific tag
            const crossAccountPrincipalStack = core_1.Stack.of(crossAccountPrincipal);
            const roleTag = `${crossAccountPrincipalStack.stackName}_${crossAccountPrincipal.node.addr}`;
            core_1.Tags.of(crossAccountPrincipal).add('aws-cdk:id', roleTag);
            this.addToResourcePolicy(new iam.PolicyStatement({
                actions,
                principals: [new iam.AccountPrincipal(crossAccountPrincipalStack.account)],
                conditions: {
                    StringEquals: { 'aws:PrincipalTag/aws-cdk:id': roleTag },
                },
            }));
            return iam.Grant.addToPrincipal({
                grantee,
                actions,
                resourceArns: [this.repositoryArn],
                scope: this,
            });
        }
        else {
            return iam.Grant.addToPrincipalOrResource({
                grantee,
                actions,
                resourceArns: [this.repositoryArn],
                resourceSelfArns: [],
                resource: this,
            });
        }
    }
    /**
     * Grant the given identity permissions to use the images in this repository
     */
    grantPull(grantee) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantPull);
            }
            throw error;
        }
        const ret = this.grant(grantee, 'ecr:BatchCheckLayerAvailability', 'ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage');
        iam.Grant.addToPrincipal({
            grantee,
            actions: ['ecr:GetAuthorizationToken'],
            resourceArns: ['*'],
            scope: this,
        });
        return ret;
    }
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     */
    grantPullPush(grantee) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantPullPush);
            }
            throw error;
        }
        this.grantPull(grantee);
        return this.grant(grantee, 'ecr:PutImage', 'ecr:InitiateLayerUpload', 'ecr:UploadLayerPart', 'ecr:CompleteLayerUpload');
    }
    /**
     * Returns the resource that backs the given IAM grantee if we cannot put a direct reference
     * to the grantee in the resource policy of this ECR repository,
     * and 'undefined' in case we can.
     */
    unsafeCrossAccountResourcePolicyPrincipal(grantee) {
        // A principal cannot be safely added to the Resource Policy of this ECR repository, if:
        // 1. The principal is from a different account, and
        // 2. The principal is a new resource (meaning, not just referenced), and
        // 3. The Stack this repo belongs to doesn't depend on the Stack the principal belongs to.
        // condition #1
        const principal = grantee.grantPrincipal;
        const principalAccount = principal.principalAccount;
        if (!principalAccount) {
            return undefined;
        }
        const repoAndPrincipalAccountCompare = core_1.Token.compareStrings(this.env.account, principalAccount);
        if (repoAndPrincipalAccountCompare === core_1.TokenComparison.BOTH_UNRESOLVED ||
            repoAndPrincipalAccountCompare === core_1.TokenComparison.SAME) {
            return undefined;
        }
        // condition #2
        if (!iam.principalIsOwnedResource(principal)) {
            return undefined;
        }
        // condition #3
        const principalStack = core_1.Stack.of(principal);
        if (this.stack.dependencies.includes(principalStack)) {
            return undefined;
        }
        return principal;
    }
}
_a = JSII_RTTI_SYMBOL_1;
RepositoryBase[_a] = { fqn: "aws-cdk-lib.aws_ecr.RepositoryBase", version: "2.74.0" };
exports.RepositoryBase = RepositoryBase;
/**
 * Define an ECR repository
 */
class Repository extends RepositoryBase {
    /**
     * Import a repository
     */
    static fromRepositoryAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecr_RepositoryAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromRepositoryAttributes);
            }
            throw error;
        }
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = attrs.repositoryName;
                this.repositoryArn = attrs.repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    static fromRepositoryArn(scope, id, repositoryArn) {
        // if repositoryArn is a token, the repository name is also required. this is because
        // repository names can include "/" (e.g. foo/bar/myrepo) and it is impossible to
        // parse the name from an ARN using CloudFormation's split/select.
        if (core_1.Token.isUnresolved(repositoryArn)) {
            throw new Error('"repositoryArn" is a late-bound value, and therefore "repositoryName" is required. Use `fromRepositoryAttributes` instead');
        }
        const repositoryName = repositoryArn.split('/').slice(1).join('/');
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id, {
            environmentFromArn: repositoryArn,
        });
    }
    static fromRepositoryName(scope, id, repositoryName) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = Repository.arnForLocalRepository(repositoryName, scope);
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    /**
     * Returns an ECR ARN for a repository that resides in the same account/region
     * as the current stack.
     */
    static arnForLocalRepository(repositoryName, scope, account) {
        return core_1.Stack.of(scope).formatArn({
            account,
            service: 'ecr',
            resource: 'repository',
            resourceName: repositoryName,
        });
    }
    static validateRepositoryName(physicalName) {
        const repositoryName = physicalName;
        if (!repositoryName || core_1.Token.isUnresolved(repositoryName)) {
            // the name is a late-bound value, not a defined string,
            // so skip validation
            return;
        }
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecr-repository.html
        if (repositoryName.length < 2 || repositoryName.length > 256) {
            errors.push('Repository name must be at least 2 and no more than 256 characters');
        }
        const isPatternMatch = /^(?:[a-z0-9]+(?:[._-][a-z0-9]+)*\/)*[a-z0-9]+(?:[._-][a-z0-9]+)*$/.test(repositoryName);
        if (!isPatternMatch) {
            errors.push('Repository name must follow the specified pattern: (?:[a-z0-9]+(?:[._-][a-z0-9]+)*/)*[a-z0-9]+(?:[._-][a-z0-9]+)*');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid ECR repository name (value: ${repositoryName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.lifecycleRules = new Array();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecr_RepositoryProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Repository);
            }
            throw error;
        }
        Repository.validateRepositoryName(this.physicalName);
        const resource = new ecr_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: this.physicalName,
            // It says "Text", but they actually mean "Object".
            repositoryPolicyText: core_1.Lazy.any({ produce: () => this.policyDocument }),
            lifecyclePolicy: core_1.Lazy.any({ produce: () => this.renderLifecyclePolicy() }),
            imageScanningConfiguration: props.imageScanOnPush !== undefined ? { scanOnPush: props.imageScanOnPush } : undefined,
            imageTagMutability: props.imageTagMutability || undefined,
            encryptionConfiguration: this.parseEncryption(props),
        });
        this._resource = resource;
        if (props.autoDeleteImages) {
            if (props.removalPolicy !== core_1.RemovalPolicy.DESTROY) {
                throw new Error('Cannot use \'autoDeleteImages\' property on a repository without setting removal policy to \'DESTROY\'.');
            }
            this.enableAutoDeleteImages();
        }
        resource.applyRemovalPolicy(props.removalPolicy);
        this.registryId = props.lifecycleRegistryId;
        if (props.lifecycleRules) {
            props.lifecycleRules.forEach(this.addLifecycleRule.bind(this));
        }
        this.repositoryName = this.getResourceNameAttribute(resource.ref);
        this.repositoryArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'ecr',
            resource: 'repository',
            resourceName: this.physicalName,
        });
        this.node.addValidation({ validate: () => this.policyDocument?.validateForResourcePolicy() ?? [] });
    }
    /**
     * Add a policy statement to the repository's resource policy.
     *
     * While other resources policies in AWS either require or accept a resource section,
     * Cfn for ECR does not allow us to specify a resource policy.
     * It will fail if a resource section is present at all.
     */
    addToResourcePolicy(statement) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToResourcePolicy);
            }
            throw error;
        }
        if (statement.resources.length) {
            core_1.Annotations.of(this).addWarning('ECR resource policy does not allow resource statements.');
        }
        if (this.policyDocument === undefined) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
        return { statementAdded: true, policyDependable: this.policyDocument };
    }
    /**
     * Add a life cycle rule to the repository
     *
     * Life cycle rules automatically expire images from the repository that match
     * certain conditions.
     */
    addLifecycleRule(rule) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecr_LifecycleRule(rule);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLifecycleRule);
            }
            throw error;
        }
        // Validate rule here so users get errors at the expected location
        if (rule.tagStatus === undefined) {
            rule = { ...rule, tagStatus: rule.tagPrefixList === undefined ? lifecycle_1.TagStatus.ANY : lifecycle_1.TagStatus.TAGGED };
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.TAGGED && (rule.tagPrefixList === undefined || rule.tagPrefixList.length === 0)) {
            throw new Error('TagStatus.Tagged requires the specification of a tagPrefixList');
        }
        if (rule.tagStatus !== lifecycle_1.TagStatus.TAGGED && rule.tagPrefixList !== undefined) {
            throw new Error('tagPrefixList can only be specified when tagStatus is set to Tagged');
        }
        if ((rule.maxImageAge !== undefined) === (rule.maxImageCount !== undefined)) {
            throw new Error(`Life cycle rule must contain exactly one of 'maxImageAge' and 'maxImageCount', got: ${JSON.stringify(rule)}`);
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.ANY && this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY).length > 0) {
            throw new Error('Life cycle can only have one TagStatus.Any rule');
        }
        this.lifecycleRules.push({ ...rule });
    }
    /**
     * Render the life cycle policy object
     */
    renderLifecyclePolicy() {
        const stack = core_1.Stack.of(this);
        let lifecyclePolicyText;
        if (this.lifecycleRules.length === 0 && !this.registryId) {
            return undefined;
        }
        if (this.lifecycleRules.length > 0) {
            lifecyclePolicyText = JSON.stringify(stack.resolve({
                rules: this.orderedLifecycleRules().map(renderLifecycleRule),
            }));
        }
        return {
            lifecyclePolicyText,
            registryId: this.registryId,
        };
    }
    /**
     * Return life cycle rules with automatic ordering applied.
     *
     * Also applies validation of the 'any' rule.
     */
    orderedLifecycleRules() {
        if (this.lifecycleRules.length === 0) {
            return [];
        }
        const prioritizedRules = this.lifecycleRules.filter(r => r.rulePriority !== undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const autoPrioritizedRules = this.lifecycleRules.filter(r => r.rulePriority === undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const anyRules = this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
        if (anyRules.length > 0 && anyRules[0].rulePriority !== undefined && autoPrioritizedRules.length > 0) {
            // Supporting this is too complex for very little value. We just prohibit it.
            throw new Error("Cannot combine prioritized TagStatus.Any rule with unprioritized rules. Remove rulePriority from the 'Any' rule.");
        }
        const prios = prioritizedRules.map(r => r.rulePriority);
        let autoPrio = (prios.length > 0 ? Math.max(...prios) : 0) + 1;
        const ret = new Array();
        for (const rule of prioritizedRules.concat(autoPrioritizedRules).concat(anyRules)) {
            ret.push({
                ...rule,
                rulePriority: rule.rulePriority ?? autoPrio++,
            });
        }
        // Do validation on the final array--might still be wrong because the user supplied all prios, but incorrectly.
        validateAnyRuleLast(ret);
        return ret;
    }
    /**
     * Set up key properties and return the Repository encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        // default based on whether encryptionKey is specified
        const encryptionType = props.encryption ?? (props.encryptionKey ? RepositoryEncryption.KMS : RepositoryEncryption.AES_256);
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== RepositoryEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType.value})`);
        }
        if (encryptionType === RepositoryEncryption.AES_256) {
            return undefined;
        }
        if (encryptionType === RepositoryEncryption.KMS) {
            return {
                encryptionType: 'KMS',
                kmsKey: props.encryptionKey?.keyArn,
            };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
    enableAutoDeleteImages() {
        // Use a iam policy to allow the custom resource to list & delete
        // images in the repository and the ability to get all repositories to find the arn needed on delete.
        const provider = core_1.CustomResourceProvider.getOrCreateProvider(this, AUTO_DELETE_IMAGES_RESOURCE_TYPE, {
            codeDirectory: path.join(__dirname, 'auto-delete-images-handler'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_14_X,
            description: `Lambda function for auto-deleting images in ${this.repositoryName} repository.`,
            policyStatements: [
                {
                    Effect: 'Allow',
                    Action: [
                        'ecr:BatchDeleteImage',
                        'ecr:DescribeRepositories',
                        'ecr:ListImages',
                        'ecr:ListTagsForResource',
                    ],
                    Resource: [this._resource.attrArn],
                },
            ],
        });
        const customResource = new core_1.CustomResource(this, 'AutoDeleteImagesCustomResource', {
            resourceType: AUTO_DELETE_IMAGES_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                RepositoryName: core_1.Lazy.any({ produce: () => this.repositoryName }),
            },
        });
        customResource.node.addDependency(this);
        // We also tag the repository to record the fact that we want it autodeleted.
        // The custom resource will check this tag before actually doing the delete.
        // Because tagging and untagging will ALWAYS happen before the CR is deleted,
        // we can set `autoDeleteImages: false` without the removal of the CR emptying
        // the repository as a side effect.
        core_1.Tags.of(this._resource).add(AUTO_DELETE_IMAGES_TAG, 'true');
    }
}
_b = JSII_RTTI_SYMBOL_1;
Repository[_b] = { fqn: "aws-cdk-lib.aws_ecr.Repository", version: "2.74.0" };
exports.Repository = Repository;
function validateAnyRuleLast(rules) {
    const anyRules = rules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
    if (anyRules.length === 1) {
        const maxPrio = Math.max(...rules.map(r => r.rulePriority));
        if (anyRules[0].rulePriority !== maxPrio) {
            throw new Error(`TagStatus.Any rule must have highest priority, has ${anyRules[0].rulePriority} which is smaller than ${maxPrio}`);
        }
    }
}
/**
 * Render the lifecycle rule to JSON
 */
function renderLifecycleRule(rule) {
    return {
        rulePriority: rule.rulePriority,
        description: rule.description,
        selection: {
            tagStatus: rule.tagStatus || lifecycle_1.TagStatus.ANY,
            tagPrefixList: rule.tagPrefixList,
            countType: rule.maxImageAge !== undefined ? "sinceImagePushed" /* CountType.SINCE_IMAGE_PUSHED */ : "imageCountMoreThan" /* CountType.IMAGE_COUNT_MORE_THAN */,
            countNumber: rule.maxImageAge?.toDays() ?? rule.maxImageCount,
            countUnit: rule.maxImageAge !== undefined ? 'days' : undefined,
        },
        action: {
            type: 'expire',
        },
    };
}
/**
 * The tag mutability setting for your repository.
 */
var TagMutability;
(function (TagMutability) {
    /**
     * allow image tags to be overwritten.
     */
    TagMutability["MUTABLE"] = "MUTABLE";
    /**
     * all image tags within the repository will be immutable which will prevent them from being overwritten.
     */
    TagMutability["IMMUTABLE"] = "IMMUTABLE";
})(TagMutability = exports.TagMutability || (exports.TagMutability = {}));
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class RepositoryEncryption {
    /**
     * @param value the string value of the encryption
     */
    constructor(value) {
        this.value = value;
    }
}
_c = JSII_RTTI_SYMBOL_1;
RepositoryEncryption[_c] = { fqn: "aws-cdk-lib.aws_ecr.RepositoryEncryption", version: "2.74.0" };
/**
 * 'AES256'
 */
RepositoryEncryption.AES_256 = new RepositoryEncryption('AES256');
/**
 * 'KMS'
 */
RepositoryEncryption.KMS = new RepositoryEncryption('KMS');
exports.RepositoryEncryption = RepositoryEncryption;
//# sourceMappingURL=data:application/json;base64,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