"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkLoadBalancedServiceBase = exports.NetworkLoadBalancedServiceRecordType = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ecs_1 = require("../../../aws-ecs");
const aws_elasticloadbalancingv2_1 = require("../../../aws-elasticloadbalancingv2");
const aws_route53_1 = require("../../../aws-route53");
const aws_route53_targets_1 = require("../../../aws-route53-targets");
const cdk = require("../../../core");
const constructs_1 = require("constructs");
/**
 * Describes the type of DNS record the service should create
 */
var NetworkLoadBalancedServiceRecordType;
(function (NetworkLoadBalancedServiceRecordType) {
    /**
     * Create Route53 A Alias record
     */
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["ALIAS"] = 0] = "ALIAS";
    /**
     * Create a CNAME record
     */
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["CNAME"] = 1] = "CNAME";
    /**
     * Do not create any DNS records
     */
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["NONE"] = 2] = "NONE";
})(NetworkLoadBalancedServiceRecordType = exports.NetworkLoadBalancedServiceRecordType || (exports.NetworkLoadBalancedServiceRecordType = {}));
/**
 * The base class for NetworkLoadBalancedEc2Service and NetworkLoadBalancedFargateService services.
 */
class NetworkLoadBalancedServiceBase extends constructs_1.Construct {
    /**
     * The Network Load Balancer for the service.
     */
    get loadBalancer() {
        if (!this._networkLoadBalancer) {
            throw new Error('.loadBalancer can only be accessed if the class was constructed with an owned, not imported, load balancer');
        }
        return this._networkLoadBalancer;
    }
    /**
     * Constructs a new instance of the NetworkLoadBalancedServiceBase class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_patterns_NetworkLoadBalancedServiceBaseProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, NetworkLoadBalancedServiceBase);
            }
            throw error;
        }
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        this.desiredCount = props.desiredCount || 1;
        this.internalDesiredCount = props.desiredCount;
        const internetFacing = props.publicLoadBalancer ?? true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        const loadBalancer = props.loadBalancer ?? new aws_elasticloadbalancingv2_1.NetworkLoadBalancer(this, 'LB', lbProps);
        const listenerPort = props.listenerPort ?? 80;
        const targetProps = {
            port: props.taskImageOptions?.containerPort ?? 80,
        };
        this.listener = loadBalancer.addListener('PublicListener', { port: listenerPort });
        this.targetGroup = this.listener.addTargets('ECS', targetProps);
        if (typeof props.domainName !== 'undefined') {
            if (typeof props.domainZone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            switch (props.recordType ?? NetworkLoadBalancedServiceRecordType.ALIAS) {
                case NetworkLoadBalancedServiceRecordType.ALIAS:
                    new aws_route53_1.ARecord(this, 'DNS', {
                        zone: props.domainZone,
                        recordName: props.domainName,
                        target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
                    });
                    break;
                case NetworkLoadBalancedServiceRecordType.CNAME:
                    new aws_route53_1.CnameRecord(this, 'DNS', {
                        zone: props.domainZone,
                        recordName: props.domainName,
                        domainName: loadBalancer.loadBalancerDnsName,
                    });
                    break;
                case NetworkLoadBalancedServiceRecordType.NONE:
                    // Do not create a DNS record
                    break;
            }
        }
        if (loadBalancer instanceof aws_elasticloadbalancingv2_1.NetworkLoadBalancer) {
            this._networkLoadBalancer = loadBalancer;
        }
        if (props.loadBalancer === undefined) {
            new cdk.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
        }
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_IVpc(vpc);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getDefaultCluster);
            }
            throw error;
        }
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = cdk.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Adds service as a target of the target group.
     */
    addServiceAsTarget(service) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_BaseService(service);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addServiceAsTarget);
            }
            throw error;
        }
        this.targetGroup.addTarget(service);
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
_a = JSII_RTTI_SYMBOL_1;
NetworkLoadBalancedServiceBase[_a] = { fqn: "aws-cdk-lib.aws_ecs_patterns.NetworkLoadBalancedServiceBase", version: "2.74.0" };
exports.NetworkLoadBalancedServiceBase = NetworkLoadBalancedServiceBase;
//# sourceMappingURL=data:application/json;base64,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