"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QueueProcessingServiceBase = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ecs_1 = require("../../../aws-ecs");
const aws_sqs_1 = require("../../../aws-sqs");
const core_1 = require("../../../core");
const cxapi = require("../../../cx-api");
const constructs_1 = require("constructs");
/**
 * The base class for QueueProcessingEc2Service and QueueProcessingFargateService services.
 */
class QueueProcessingServiceBase extends constructs_1.Construct {
    /**
     * Constructs a new instance of the QueueProcessingServiceBase class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_patterns_QueueProcessingServiceBaseProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, QueueProcessingServiceBase);
            }
            throw error;
        }
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        if (props.queue && (props.retentionPeriod || props.visibilityTimeout || props.maxReceiveCount)) {
            const errorProps = ['retentionPeriod', 'visibilityTimeout', 'maxReceiveCount'].filter(prop => props.hasOwnProperty(prop));
            throw new Error(`${errorProps.join(', ')} can be set only when queue is not set. Specify them in the QueueProps of the queue`);
        }
        // Create the SQS queue and it's corresponding DLQ if one is not provided
        if (props.queue) {
            this.sqsQueue = props.queue;
        }
        else {
            this.deadLetterQueue = new aws_sqs_1.Queue(this, 'EcsProcessingDeadLetterQueue', {
                retentionPeriod: props.retentionPeriod || core_1.Duration.days(14),
            });
            this.sqsQueue = new aws_sqs_1.Queue(this, 'EcsProcessingQueue', {
                visibilityTimeout: props.visibilityTimeout,
                deadLetterQueue: {
                    queue: this.deadLetterQueue,
                    maxReceiveCount: props.maxReceiveCount || 3,
                },
            });
            new core_1.CfnOutput(this, 'SQSDeadLetterQueue', { value: this.deadLetterQueue.queueName });
            new core_1.CfnOutput(this, 'SQSDeadLetterQueueArn', { value: this.deadLetterQueue.queueArn });
        }
        // Setup autoscaling scaling intervals
        const defaultScalingSteps = [{ upper: 0, change: -1 }, { lower: 100, change: +1 }, { lower: 500, change: +5 }];
        this.scalingSteps = props.scalingSteps ?? defaultScalingSteps;
        // Create log driver if logging is enabled
        const enableLogging = props.enableLogging ?? true;
        this.logDriver = props.logDriver ?? (enableLogging ? this.createAWSLogDriver(this.node.id) : undefined);
        // Add the queue name to environment variables
        this.environment = { ...(props.environment || {}), QUEUE_NAME: this.sqsQueue.queueName };
        this.secrets = props.secrets;
        this.desiredCount = props.desiredTaskCount ?? 1;
        // Determine the desired task count (minimum) and maximum scaling capacity
        if (!core_1.FeatureFlags.of(this).isEnabled(cxapi.ECS_REMOVE_DEFAULT_DESIRED_COUNT)) {
            this.minCapacity = props.minScalingCapacity ?? this.desiredCount;
            this.maxCapacity = props.maxScalingCapacity || (2 * this.desiredCount);
        }
        else {
            if (props.desiredTaskCount != null) {
                this.minCapacity = props.minScalingCapacity ?? this.desiredCount;
                this.maxCapacity = props.maxScalingCapacity || (2 * this.desiredCount);
            }
            else {
                this.minCapacity = props.minScalingCapacity ?? 1;
                this.maxCapacity = props.maxScalingCapacity || 2;
            }
        }
        if (!this.desiredCount && !this.maxCapacity) {
            throw new Error('maxScalingCapacity must be set and greater than 0 if desiredCount is 0');
        }
        new core_1.CfnOutput(this, 'SQSQueue', { value: this.sqsQueue.queueName });
        new core_1.CfnOutput(this, 'SQSQueueArn', { value: this.sqsQueue.queueArn });
    }
    /**
     * Configure autoscaling based off of CPU utilization as well as the number of messages visible in the SQS queue
     *
     * @param service the ECS/Fargate service for which to apply the autoscaling rules to
     */
    configureAutoscalingForService(service) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_BaseService(service);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.configureAutoscalingForService);
            }
            throw error;
        }
        const scalingTarget = service.autoScaleTaskCount({ maxCapacity: this.maxCapacity, minCapacity: this.minCapacity });
        scalingTarget.scaleOnCpuUtilization('CpuScaling', {
            targetUtilizationPercent: 50,
        });
        scalingTarget.scaleOnMetric('QueueMessagesVisibleScaling', {
            metric: this.sqsQueue.metricApproximateNumberOfMessagesVisible(),
            scalingSteps: this.scalingSteps,
        });
    }
    /**
     * Grant SQS permissions to an ECS service.
     * @param service the ECS/Fargate service to which to grant SQS permissions
     */
    grantPermissionsToService(service) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_BaseService(service);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantPermissionsToService);
            }
            throw error;
        }
        this.sqsQueue.grantConsumeMessages(service.taskDefinition.taskRole);
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_IVpc(vpc);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getDefaultCluster);
            }
            throw error;
        }
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Create an AWS Log Driver with the provided streamPrefix
     *
     * @param prefix the Cloudwatch logging prefix
     */
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
_a = JSII_RTTI_SYMBOL_1;
QueueProcessingServiceBase[_a] = { fqn: "aws-cdk-lib.aws_ecs_patterns.QueueProcessingServiceBase", version: "2.74.0" };
exports.QueueProcessingServiceBase = QueueProcessingServiceBase;
//# sourceMappingURL=data:application/json;base64,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