"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppProtocol = exports.Protocol = exports.ServiceConnect = exports.PortMap = exports.ContainerDependencyCondition = exports.UlimitName = exports.ContainerDefinition = exports.Secret = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("../../core");
const constructs_1 = require("constructs");
const task_definition_1 = require("./base/task-definition");
/**
 * A secret environment variable.
 */
class Secret {
    /**
     * Creates an environment variable value from a parameter stored in AWS
     * Systems Manager Parameter Store.
     */
    static fromSsmParameter(parameter) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ssm_IParameter(parameter);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSsmParameter);
            }
            throw error;
        }
        return {
            arn: parameter.parameterArn,
            grantRead: grantee => parameter.grantRead(grantee),
        };
    }
    /**
     * Creates a environment variable value from a secret stored in AWS Secrets
     * Manager.
     *
     * @param secret the secret stored in AWS Secrets Manager
     * @param field the name of the field with the value that you want to set as
     * the environment variable value. Only values in JSON format are supported.
     * If you do not specify a JSON field, then the full content of the secret is
     * used.
     */
    static fromSecretsManager(secret, field) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_ISecret(secret);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSecretsManager);
            }
            throw error;
        }
        return {
            arn: field ? `${secret.secretArn}:${field}::` : secret.secretArn,
            hasField: !!field,
            grantRead: grantee => secret.grantRead(grantee),
        };
    }
    /**
     * Creates a environment variable value from a secret stored in AWS Secrets
     * Manager.
     *
     * @param secret the secret stored in AWS Secrets Manager
     * @param versionInfo the version information to reference the secret
     * @param field the name of the field with the value that you want to set as
     * the environment variable value. Only values in JSON format are supported.
     * If you do not specify a JSON field, then the full content of the secret is
     * used.
     */
    static fromSecretsManagerVersion(secret, versionInfo, field) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_ISecret(secret);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_SecretVersionInfo(versionInfo);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSecretsManagerVersion);
            }
            throw error;
        }
        return {
            arn: `${secret.secretArn}:${field ?? ''}:${versionInfo.versionStage ?? ''}:${versionInfo.versionId ?? ''}`,
            hasField: !!field,
            grantRead: grantee => secret.grantRead(grantee),
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
Secret[_a] = { fqn: "aws-cdk-lib.aws_ecs.Secret", version: "2.74.0" };
exports.Secret = Secret;
/**
 * A container definition is used in a task definition to describe the containers that are launched as part of a task.
 */
class ContainerDefinition extends constructs_1.Construct {
    /**
     * Constructs a new instance of the ContainerDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        /**
         * The mount points for data volumes in your container.
         */
        this.mountPoints = new Array();
        /**
         * The list of port mappings for the container. Port mappings allow containers to access ports
         * on the host container instance to send or receive traffic.
         */
        this.portMappings = new Array();
        /**
         * The data volumes to mount from another container in the same task definition.
         */
        this.volumesFrom = new Array();
        /**
         * An array of ulimits to set in the container.
         */
        this.ulimits = new Array();
        /**
         * An array dependencies defined for container startup and shutdown.
         */
        this.containerDependencies = new Array();
        /**
         * The inference accelerators referenced by this container.
         */
        this.inferenceAcceleratorResources = [];
        /**
         * The configured container links
         */
        this.links = new Array();
        this.secrets = [];
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_ContainerDefinitionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ContainerDefinition);
            }
            throw error;
        }
        if (props.memoryLimitMiB !== undefined && props.memoryReservationMiB !== undefined) {
            if (props.memoryLimitMiB < props.memoryReservationMiB) {
                throw new Error('MemoryLimitMiB should not be less than MemoryReservationMiB.');
            }
        }
        this.essential = props.essential ?? true;
        this.taskDefinition = props.taskDefinition;
        this.memoryLimitSpecified = props.memoryLimitMiB !== undefined || props.memoryReservationMiB !== undefined;
        this.linuxParameters = props.linuxParameters;
        this.containerName = props.containerName ?? this.node.id;
        this.imageConfig = props.image.bind(this, this);
        this.imageName = this.imageConfig.imageName;
        this._namedPorts = new Map();
        if (props.logging) {
            this.logDriverConfig = props.logging.bind(this, this);
        }
        if (props.secrets) {
            for (const [name, secret] of Object.entries(props.secrets)) {
                this.addSecret(name, secret);
            }
        }
        if (props.environment) {
            this.environment = { ...props.environment };
        }
        else {
            this.environment = {};
        }
        if (props.environmentFiles) {
            this.environmentFiles = [];
            for (const environmentFile of props.environmentFiles) {
                this.environmentFiles.push(environmentFile.bind(this));
            }
        }
        if (props.cpu) {
            this.cpu = props.cpu;
        }
        props.taskDefinition._linkContainer(this);
        if (props.portMappings) {
            this.addPortMappings(...props.portMappings);
        }
        if (props.inferenceAcceleratorResources) {
            this.addInferenceAcceleratorResource(...props.inferenceAcceleratorResources);
        }
        this.pseudoTerminal = props.pseudoTerminal;
        if (props.ulimits) {
            this.addUlimits(...props.ulimits);
        }
    }
    /**
     * This method adds a link which allows containers to communicate with each other without the need for port mappings.
     *
     * This parameter is only supported if the task definition is using the bridge network mode.
     * Warning: The --link flag is a legacy feature of Docker. It may eventually be removed.
     */
    addLink(container, alias) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_ContainerDefinition(container);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLink);
            }
            throw error;
        }
        if (this.taskDefinition.networkMode !== task_definition_1.NetworkMode.BRIDGE) {
            throw new Error('You must use network mode Bridge to add container links.');
        }
        if (alias !== undefined) {
            this.links.push(`${container.containerName}:${alias}`);
        }
        else {
            this.links.push(`${container.containerName}`);
        }
    }
    /**
     * This method adds one or more mount points for data volumes to the container.
     */
    addMountPoints(...mountPoints) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_MountPoint(mountPoints);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addMountPoints);
            }
            throw error;
        }
        this.mountPoints.push(...mountPoints);
    }
    /**
     * This method mounts temporary disk space to the container.
     *
     * This adds the correct container mountPoint and task definition volume.
     */
    addScratch(scratch) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_ScratchSpace(scratch);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addScratch);
            }
            throw error;
        }
        const mountPoint = {
            containerPath: scratch.containerPath,
            readOnly: scratch.readOnly,
            sourceVolume: scratch.name,
        };
        const volume = {
            host: {
                sourcePath: scratch.sourcePath,
            },
            name: scratch.name,
        };
        this.taskDefinition.addVolume(volume);
        this.addMountPoints(mountPoint);
    }
    /**
     * This method adds one or more port mappings to the container.
     */
    addPortMappings(...portMappings) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_PortMapping(portMappings);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPortMappings);
            }
            throw error;
        }
        this.portMappings.push(...portMappings.map(pm => {
            const portMap = new PortMap(this.taskDefinition.networkMode, pm);
            portMap.validate();
            const serviceConnect = new ServiceConnect(this.taskDefinition.networkMode, pm);
            if (serviceConnect.isServiceConnect()) {
                serviceConnect.validate();
                this.setNamedPort(pm);
            }
            const sanitizedPM = this.addHostPortIfNeeded(pm);
            return sanitizedPM;
        }));
    }
    /**
     * This method adds an environment variable to the container.
     */
    addEnvironment(name, value) {
        this.environment[name] = value;
    }
    /**
     * This method adds a secret as environment variable to the container.
     */
    addSecret(name, secret) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_Secret(secret);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSecret);
            }
            throw error;
        }
        secret.grantRead(this.taskDefinition.obtainExecutionRole());
        this.secrets.push({
            name,
            valueFrom: secret.arn,
        });
    }
    /**
     * This method adds one or more resources to the container.
     */
    addInferenceAcceleratorResource(...inferenceAcceleratorResources) {
        this.inferenceAcceleratorResources.push(...inferenceAcceleratorResources.map(resource => {
            for (const inferenceAccelerator of this.taskDefinition.inferenceAccelerators) {
                if (resource === inferenceAccelerator.deviceName) {
                    return resource;
                }
            }
            throw new Error(`Resource value ${resource} in container definition doesn't match any inference accelerator device name in the task definition.`);
        }));
    }
    /**
     * This method adds one or more ulimits to the container.
     */
    addUlimits(...ulimits) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_Ulimit(ulimits);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addUlimits);
            }
            throw error;
        }
        this.ulimits.push(...ulimits);
    }
    /**
     * This method adds one or more container dependencies to the container.
     */
    addContainerDependencies(...containerDependencies) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_ContainerDependency(containerDependencies);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addContainerDependencies);
            }
            throw error;
        }
        this.containerDependencies.push(...containerDependencies);
    }
    /**
     * This method adds one or more volumes to the container.
     */
    addVolumesFrom(...volumesFrom) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_VolumeFrom(volumesFrom);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addVolumesFrom);
            }
            throw error;
        }
        this.volumesFrom.push(...volumesFrom);
    }
    /**
     * This method adds the specified statement to the IAM task execution policy in the task definition.
     */
    addToExecutionPolicy(statement) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToExecutionPolicy);
            }
            throw error;
        }
        this.taskDefinition.addToExecutionRolePolicy(statement);
    }
    /**
     * Returns the host port for the requested container port if it exists
     */
    findPortMapping(containerPort, protocol) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_Protocol(protocol);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.findPortMapping);
            }
            throw error;
        }
        for (const portMapping of this.portMappings) {
            const p = portMapping.protocol || Protocol.TCP;
            const c = portMapping.containerPort;
            if (c === containerPort && p === protocol) {
                return portMapping;
            }
        }
        return undefined;
    }
    /**
     * Returns the port mapping with the given name, if it exists.
     */
    findPortMappingByName(name) {
        return this._namedPorts.get(name);
    }
    /**
     * This method adds an namedPort
     */
    setNamedPort(pm) {
        if (!pm.name)
            return;
        if (this._namedPorts.has(pm.name)) {
            throw new Error(`Port mapping name '${pm.name}' already exists on this container`);
        }
        this._namedPorts.set(pm.name, pm);
    }
    /**
     * Set HostPort to 0 When netowork mode is Brdige
     */
    addHostPortIfNeeded(pm) {
        const newPM = {
            ...pm,
        };
        if (this.taskDefinition.networkMode !== task_definition_1.NetworkMode.BRIDGE)
            return newPM;
        if (pm.hostPort !== undefined)
            return newPM;
        newPM.hostPort = 0;
        return newPM;
    }
    /**
     * Whether this container definition references a specific JSON field of a secret
     * stored in Secrets Manager.
     */
    get referencesSecretJsonField() {
        for (const secret of this.secrets) {
            if (secret.valueFrom.endsWith('::')) {
                return true;
            }
        }
        return false;
    }
    /**
     * The inbound rules associated with the security group the task or service will use.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     */
    get ingressPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        if (defaultPortMapping.hostPort !== undefined && defaultPortMapping.hostPort !== 0) {
            return defaultPortMapping.hostPort;
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
            return 0;
        }
        return defaultPortMapping.containerPort;
    }
    /**
     * The port the container will listen on.
     */
    get containerPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        return defaultPortMapping.containerPort;
    }
    /**
     * Render this container definition to a CloudFormation object
     *
     * @param _taskDefinition [disable-awslint:ref-via-interface] (unused but kept to avoid breaking change)
     */
    renderContainerDefinition(_taskDefinition) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_TaskDefinition(_taskDefinition);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.renderContainerDefinition);
            }
            throw error;
        }
        return {
            command: this.props.command,
            cpu: this.props.cpu,
            disableNetworking: this.props.disableNetworking,
            dependsOn: cdk.Lazy.any({ produce: () => this.containerDependencies.map(renderContainerDependency) }, { omitEmptyArray: true }),
            dnsSearchDomains: this.props.dnsSearchDomains,
            dnsServers: this.props.dnsServers,
            dockerLabels: this.props.dockerLabels,
            dockerSecurityOptions: this.props.dockerSecurityOptions,
            entryPoint: this.props.entryPoint,
            essential: this.essential,
            hostname: this.props.hostname,
            image: this.imageConfig.imageName,
            memory: this.props.memoryLimitMiB,
            memoryReservation: this.props.memoryReservationMiB,
            mountPoints: cdk.Lazy.any({ produce: () => this.mountPoints.map(renderMountPoint) }, { omitEmptyArray: true }),
            name: this.containerName,
            portMappings: cdk.Lazy.any({ produce: () => this.portMappings.map(renderPortMapping) }, { omitEmptyArray: true }),
            privileged: this.props.privileged,
            pseudoTerminal: this.props.pseudoTerminal,
            readonlyRootFilesystem: this.props.readonlyRootFilesystem,
            repositoryCredentials: this.imageConfig.repositoryCredentials,
            startTimeout: this.props.startTimeout && this.props.startTimeout.toSeconds(),
            stopTimeout: this.props.stopTimeout && this.props.stopTimeout.toSeconds(),
            ulimits: cdk.Lazy.any({ produce: () => this.ulimits.map(renderUlimit) }, { omitEmptyArray: true }),
            user: this.props.user,
            volumesFrom: cdk.Lazy.any({ produce: () => this.volumesFrom.map(renderVolumeFrom) }, { omitEmptyArray: true }),
            workingDirectory: this.props.workingDirectory,
            logConfiguration: this.logDriverConfig,
            environment: this.environment && Object.keys(this.environment).length ? renderKV(this.environment, 'name', 'value') : undefined,
            environmentFiles: this.environmentFiles && renderEnvironmentFiles(cdk.Stack.of(this).partition, this.environmentFiles),
            secrets: this.secrets.length ? this.secrets : undefined,
            extraHosts: this.props.extraHosts && renderKV(this.props.extraHosts, 'hostname', 'ipAddress'),
            healthCheck: this.props.healthCheck && renderHealthCheck(this.props.healthCheck),
            links: cdk.Lazy.list({ produce: () => this.links }, { omitEmpty: true }),
            linuxParameters: this.linuxParameters && this.linuxParameters.renderLinuxParameters(),
            resourceRequirements: (!this.props.gpuCount && this.inferenceAcceleratorResources.length == 0) ? undefined :
                renderResourceRequirements(this.props.gpuCount, this.inferenceAcceleratorResources),
            systemControls: this.props.systemControls && renderSystemControls(this.props.systemControls),
        };
    }
}
_b = JSII_RTTI_SYMBOL_1;
ContainerDefinition[_b] = { fqn: "aws-cdk-lib.aws_ecs.ContainerDefinition", version: "2.74.0" };
exports.ContainerDefinition = ContainerDefinition;
function renderKV(env, keyName, valueName) {
    const ret = [];
    for (const [key, value] of Object.entries(env)) {
        ret.push({ [keyName]: key, [valueName]: value });
    }
    return ret;
}
function renderEnvironmentFiles(partition, environmentFiles) {
    const ret = [];
    for (const environmentFile of environmentFiles) {
        const s3Location = environmentFile.s3Location;
        if (!s3Location) {
            throw Error('Environment file must specify an S3 location');
        }
        ret.push({
            type: environmentFile.fileType,
            value: `arn:${partition}:s3:::${s3Location.bucketName}/${s3Location.objectKey}`,
        });
    }
    return ret;
}
function renderHealthCheck(hc) {
    if (hc.interval?.toSeconds() !== undefined) {
        if (5 > hc.interval?.toSeconds() || hc.interval?.toSeconds() > 300) {
            throw new Error('Interval must be between 5 seconds and 300 seconds.');
        }
    }
    if (hc.timeout?.toSeconds() !== undefined) {
        if (2 > hc.timeout?.toSeconds() || hc.timeout?.toSeconds() > 120) {
            throw new Error('Timeout must be between 2 seconds and 120 seconds.');
        }
    }
    if (hc.interval?.toSeconds() !== undefined && hc.timeout?.toSeconds() !== undefined) {
        if (hc.interval?.toSeconds() < hc.timeout?.toSeconds()) {
            throw new Error('Health check interval should be longer than timeout.');
        }
    }
    return {
        command: getHealthCheckCommand(hc),
        interval: hc.interval?.toSeconds() ?? 30,
        retries: hc.retries ?? 3,
        startPeriod: hc.startPeriod?.toSeconds(),
        timeout: hc.timeout?.toSeconds() ?? 5,
    };
}
function getHealthCheckCommand(hc) {
    const cmd = hc.command;
    const hcCommand = new Array();
    if (cmd.length === 0) {
        throw new Error('At least one argument must be supplied for health check command.');
    }
    if (cmd.length === 1) {
        hcCommand.push('CMD-SHELL', cmd[0]);
        return hcCommand;
    }
    if (cmd[0] !== 'CMD' && cmd[0] !== 'CMD-SHELL') {
        hcCommand.push('CMD');
    }
    return hcCommand.concat(cmd);
}
function renderResourceRequirements(gpuCount = 0, inferenceAcceleratorResources = []) {
    const ret = [];
    for (const resource of inferenceAcceleratorResources) {
        ret.push({
            type: 'InferenceAccelerator',
            value: resource,
        });
    }
    if (gpuCount > 0) {
        ret.push({
            type: 'GPU',
            value: gpuCount.toString(),
        });
    }
    return ret;
}
/**
 * Type of resource to set a limit on
 */
var UlimitName;
(function (UlimitName) {
    UlimitName["CORE"] = "core";
    UlimitName["CPU"] = "cpu";
    UlimitName["DATA"] = "data";
    UlimitName["FSIZE"] = "fsize";
    UlimitName["LOCKS"] = "locks";
    UlimitName["MEMLOCK"] = "memlock";
    UlimitName["MSGQUEUE"] = "msgqueue";
    UlimitName["NICE"] = "nice";
    UlimitName["NOFILE"] = "nofile";
    UlimitName["NPROC"] = "nproc";
    UlimitName["RSS"] = "rss";
    UlimitName["RTPRIO"] = "rtprio";
    UlimitName["RTTIME"] = "rttime";
    UlimitName["SIGPENDING"] = "sigpending";
    UlimitName["STACK"] = "stack";
})(UlimitName = exports.UlimitName || (exports.UlimitName = {}));
function renderUlimit(ulimit) {
    return {
        name: ulimit.name,
        softLimit: ulimit.softLimit,
        hardLimit: ulimit.hardLimit,
    };
}
var ContainerDependencyCondition;
(function (ContainerDependencyCondition) {
    /**
     * This condition emulates the behavior of links and volumes today.
     * It validates that a dependent container is started before permitting other containers to start.
     */
    ContainerDependencyCondition["START"] = "START";
    /**
     * This condition validates that a dependent container runs to completion (exits) before permitting other containers to start.
     * This can be useful for nonessential containers that run a script and then exit.
     */
    ContainerDependencyCondition["COMPLETE"] = "COMPLETE";
    /**
     * This condition is the same as COMPLETE, but it also requires that the container exits with a zero status.
     */
    ContainerDependencyCondition["SUCCESS"] = "SUCCESS";
    /**
     * This condition validates that the dependent container passes its Docker health check before permitting other containers to start.
     * This requires that the dependent container has health checks configured. This condition is confirmed only at task startup.
     */
    ContainerDependencyCondition["HEALTHY"] = "HEALTHY";
})(ContainerDependencyCondition = exports.ContainerDependencyCondition || (exports.ContainerDependencyCondition = {}));
function renderContainerDependency(containerDependency) {
    return {
        containerName: containerDependency.container.containerName,
        condition: containerDependency.condition || ContainerDependencyCondition.HEALTHY,
    };
}
/**
 * PortMap ValueObjectClass having by ContainerDefinition
 */
class PortMap {
    constructor(networkmode, pm) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_NetworkMode(networkmode);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_PortMapping(pm);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, PortMap);
            }
            throw error;
        }
        this.networkmode = networkmode;
        this.portmapping = pm;
    }
    /**
     * validate invalid portmapping and networkmode parameters.
     * throw Error when invalid parameters.
     */
    validate() {
        if (!this.isvalidPortName()) {
            throw new Error('Port mapping name cannot be an empty string.');
        }
        if (!this.isValidPorts()) {
            const pm = this.portmapping;
            throw new Error(`Host port (${pm.hostPort}) must be left out or equal to container port ${pm.containerPort} for network mode ${this.networkmode}`);
        }
    }
    isvalidPortName() {
        if (this.portmapping.name === '') {
            return false;
        }
        return true;
    }
    isValidPorts() {
        const isAwsVpcMode = this.networkmode == task_definition_1.NetworkMode.AWS_VPC;
        const isHostMode = this.networkmode == task_definition_1.NetworkMode.HOST;
        if (!isAwsVpcMode && !isHostMode)
            return true;
        const hostPort = this.portmapping.hostPort;
        const containerPort = this.portmapping.containerPort;
        if (containerPort !== hostPort && hostPort !== undefined)
            return false;
        return true;
    }
}
_c = JSII_RTTI_SYMBOL_1;
PortMap[_c] = { fqn: "aws-cdk-lib.aws_ecs.PortMap", version: "2.74.0" };
exports.PortMap = PortMap;
/**
 * ServiceConnect ValueObjectClass having by ContainerDefinition
 */
class ServiceConnect {
    constructor(networkmode, pm) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_NetworkMode(networkmode);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_PortMapping(pm);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ServiceConnect);
            }
            throw error;
        }
        this.portmapping = pm;
        this.networkmode = networkmode;
    }
    /**
     * Judge parameters can be serviceconnect logick.
     * If parameters can be serviceConnect return true.
     */
    isServiceConnect() {
        const hasPortname = this.portmapping.name;
        const hasAppProtcol = this.portmapping.appProtocol;
        if (hasPortname || hasAppProtcol)
            return true;
        return false;
    }
    /**
     * Judge serviceconnect parametes are valid.
     * If invalid, throw Error.
     */
    validate() {
        if (!this.isValidNetworkmode()) {
            throw new Error(`Service connect related port mapping fields 'name' and 'appProtocol' are not supported for network mode ${this.networkmode}`);
        }
        if (!this.isValidPortName()) {
            throw new Error('Service connect-related port mapping field \'appProtocol\' cannot be set without \'name\'');
        }
    }
    isValidNetworkmode() {
        const isAwsVpcMode = this.networkmode == task_definition_1.NetworkMode.AWS_VPC;
        const isBridgeMode = this.networkmode == task_definition_1.NetworkMode.BRIDGE;
        if (isAwsVpcMode || isBridgeMode)
            return true;
        return false;
    }
    isValidPortName() {
        if (!this.portmapping.name)
            return false;
        return true;
    }
}
_d = JSII_RTTI_SYMBOL_1;
ServiceConnect[_d] = { fqn: "aws-cdk-lib.aws_ecs.ServiceConnect", version: "2.74.0" };
exports.ServiceConnect = ServiceConnect;
/**
 * Network protocol
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP
     */
    Protocol["TCP"] = "tcp";
    /**
     * UDP
     */
    Protocol["UDP"] = "udp";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Service connect app protocol.
 */
class AppProtocol {
    constructor(value) {
        this.value = value;
    }
}
_e = JSII_RTTI_SYMBOL_1;
AppProtocol[_e] = { fqn: "aws-cdk-lib.aws_ecs.AppProtocol", version: "2.74.0" };
/**
 * HTTP app protocol.
 */
AppProtocol.http = new AppProtocol('http');
/**
 * HTTP2 app protocol.
 */
AppProtocol.http2 = new AppProtocol('http2');
/**
 * GRPC app protocol.
 */
AppProtocol.grpc = new AppProtocol('grpc');
exports.AppProtocol = AppProtocol;
function renderPortMapping(pm) {
    return {
        containerPort: pm.containerPort,
        hostPort: pm.hostPort,
        protocol: pm.protocol || Protocol.TCP,
        appProtocol: pm.appProtocol?.value,
        name: pm.name ? pm.name : undefined,
    };
}
function renderMountPoint(mp) {
    return {
        containerPath: mp.containerPath,
        readOnly: mp.readOnly,
        sourceVolume: mp.sourceVolume,
    };
}
function renderVolumeFrom(vf) {
    return {
        sourceContainer: vf.sourceContainer,
        readOnly: vf.readOnly,
    };
}
function renderSystemControls(systemControls) {
    return systemControls.map(sc => ({
        namespace: sc.namespace,
        value: sc.value,
    }));
}
//# sourceMappingURL=data:application/json;base64,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