"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirelensLogRouter = exports.obtainDefaultFluentBitECRImage = exports.FirelensConfigFileType = exports.FirelensLogRouterType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const ssm = require("../../aws-ssm");
const cdk = require("../../core");
const container_definition_1 = require("./container-definition");
const container_image_1 = require("./container-image");
/**
 * Firelens log router type, fluentbit or fluentd.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html
 */
var FirelensLogRouterType;
(function (FirelensLogRouterType) {
    /**
     * fluentbit
     */
    FirelensLogRouterType["FLUENTBIT"] = "fluentbit";
    /**
     * fluentd
     */
    FirelensLogRouterType["FLUENTD"] = "fluentd";
})(FirelensLogRouterType = exports.FirelensLogRouterType || (exports.FirelensLogRouterType = {}));
/**
 * Firelens configuration file type, s3 or file path.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-taskdef-customconfig
 */
var FirelensConfigFileType;
(function (FirelensConfigFileType) {
    /**
     * s3
     */
    FirelensConfigFileType["S3"] = "s3";
    /**
     * fluentd
     */
    FirelensConfigFileType["FILE"] = "file";
})(FirelensConfigFileType = exports.FirelensConfigFileType || (exports.FirelensConfigFileType = {}));
/**
 * Render to CfnTaskDefinition.FirelensConfigurationProperty from FirelensConfig
 */
function renderFirelensConfig(firelensConfig) {
    if (!firelensConfig.options) {
        return { type: firelensConfig.type };
    }
    else if (firelensConfig.options.configFileValue === undefined) {
        // config file options work as a pair together to define a custom config source
        // a custom config source is optional,
        // and thus the `config-file-x` keys should be set together or not at all
        return {
            type: firelensConfig.type,
            options: {
                'enable-ecs-log-metadata': firelensConfig.options.enableECSLogMetadata ? 'true' : 'false',
            },
        };
    }
    else {
        // firelensConfig.options.configFileType has been filled with s3 or file type in constructor.
        return {
            type: firelensConfig.type,
            options: {
                'enable-ecs-log-metadata': firelensConfig.options.enableECSLogMetadata ? 'true' : 'false',
                'config-file-type': firelensConfig.options.configFileType,
                'config-file-value': firelensConfig.options.configFileValue,
            },
        };
    }
}
/**
 * SSM parameters for latest fluent bit docker image in ECR
 * https://github.com/aws/aws-for-fluent-bit#using-ssm-to-find-available-versions
 */
const fluentBitImageSSMPath = '/aws/service/aws-for-fluent-bit';
/**
 * Obtain Fluent Bit image in Amazon ECR and setup corresponding IAM permissions.
 * ECR image pull permissions will be granted in task execution role.
 * Cloudwatch logs, Kinesis data stream or firehose permissions will be grant by check options in logDriverConfig.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-using-fluentbit
 */
function obtainDefaultFluentBitECRImage(task, logDriverConfig, imageTag) {
    // grant ECR image pull permissions to executor role
    task.addToExecutionRolePolicy(new iam.PolicyStatement({
        actions: [
            'ecr:GetAuthorizationToken',
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:BatchGetImage',
        ],
        resources: ['*'],
    }));
    // grant cloudwatch or firehose permissions to task role
    const logName = logDriverConfig && logDriverConfig.logDriver === 'awsfirelens'
        && logDriverConfig.options && logDriverConfig.options.Name;
    if (logName === 'cloudwatch') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:DescribeLogStreams',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'firehose') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'firehose:PutRecordBatch',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'kinesis') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'kinesis:PutRecords',
            ],
            resources: ['*'],
        }));
    }
    const fluentBitImageTag = imageTag || 'latest';
    const fluentBitImage = `${fluentBitImageSSMPath}/${fluentBitImageTag}`;
    // Not use ContainerImage.fromEcrRepository since it's not support parsing ECR repo URI,
    // use repo ARN might result in complex Fn:: functions in cloudformation template.
    return container_image_1.ContainerImage.fromRegistry(ssm.StringParameter.valueForStringParameter(task, fluentBitImage));
}
exports.obtainDefaultFluentBitECRImage = obtainDefaultFluentBitECRImage;
/**
 * Firelens log router
 */
class FirelensLogRouter extends container_definition_1.ContainerDefinition {
    /**
     * Constructs a new instance of the FirelensLogRouter class.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_FirelensLogRouterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, FirelensLogRouter);
            }
            throw error;
        }
        const options = props.firelensConfig.options;
        if (options) {
            if ((options.configFileValue && options.configFileType === undefined) || (options.configFileValue === undefined && options.configFileType)) {
                throw new Error('configFileValue and configFileType must be set together to define a custom config source');
            }
            const hasConfig = (options.configFileValue !== undefined);
            const enableECSLogMetadata = options.enableECSLogMetadata || options.enableECSLogMetadata === undefined;
            const configFileType = (options.configFileType === undefined || options.configFileType === FirelensConfigFileType.S3) &&
                (cdk.Token.isUnresolved(options.configFileValue) || /arn:aws[a-zA-Z-]*:s3:::.+/.test(options.configFileValue || ''))
                ? FirelensConfigFileType.S3 : FirelensConfigFileType.FILE;
            this.firelensConfig = {
                type: props.firelensConfig.type,
                options: {
                    enableECSLogMetadata,
                    ...(hasConfig ? {
                        configFileType,
                        configFileValue: options.configFileValue,
                    } : {}),
                },
            };
            if (hasConfig) {
                // grant s3 access permissions
                if (configFileType === FirelensConfigFileType.S3) {
                    props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                        actions: [
                            's3:GetObject',
                        ],
                        resources: [(options.configFileValue ?? '')],
                    }));
                    props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                        actions: [
                            's3:GetBucketLocation',
                        ],
                        resources: [(options.configFileValue ?? '').split('/')[0]],
                    }));
                }
            }
        }
        else {
            this.firelensConfig = props.firelensConfig;
        }
    }
    /**
     * Render this container definition to a CloudFormation object
     */
    renderContainerDefinition(_taskDefinition) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecs_TaskDefinition(_taskDefinition);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.renderContainerDefinition);
            }
            throw error;
        }
        return {
            ...(super.renderContainerDefinition()),
            firelensConfiguration: this.firelensConfig && renderFirelensConfig(this.firelensConfig),
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
FirelensLogRouter[_a] = { fqn: "aws-cdk-lib.aws_ecs.FirelensLogRouter", version: "2.74.0" };
exports.FirelensLogRouter = FirelensLogRouter;
//# sourceMappingURL=data:application/json;base64,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