"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystem = exports.ThroughputMode = exports.PerformanceMode = exports.OutOfInfrequentAccessPolicy = exports.LifecyclePolicy = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const cxapi = require("../../cx-api");
const constructs_1 = require("constructs");
const access_point_1 = require("./access-point");
const efs_generated_1 = require("./efs.generated");
/**
 * EFS Lifecycle Policy, if a file is not accessed for given days, it will move to EFS Infrequent Access.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-lifecyclepolicies
 */
var LifecyclePolicy;
(function (LifecyclePolicy) {
    /**
     * After 1 day of not being accessed.
     */
    LifecyclePolicy["AFTER_1_DAY"] = "AFTER_1_DAY";
    /**
     * After 7 days of not being accessed.
     */
    LifecyclePolicy["AFTER_7_DAYS"] = "AFTER_7_DAYS";
    /**
     * After 14 days of not being accessed.
     */
    LifecyclePolicy["AFTER_14_DAYS"] = "AFTER_14_DAYS";
    /**
     * After 30 days of not being accessed.
     */
    LifecyclePolicy["AFTER_30_DAYS"] = "AFTER_30_DAYS";
    /**
     * After 60 days of not being accessed.
     */
    LifecyclePolicy["AFTER_60_DAYS"] = "AFTER_60_DAYS";
    /**
     * After 90 days of not being accessed.
     */
    LifecyclePolicy["AFTER_90_DAYS"] = "AFTER_90_DAYS";
})(LifecyclePolicy = exports.LifecyclePolicy || (exports.LifecyclePolicy = {}));
/**
 * EFS Out Of Infrequent Access Policy, if a file is accessed given times, it will move back to primary
 * storage class.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-efs-filesystem-lifecyclepolicy.html#cfn-efs-filesystem-lifecyclepolicy-transitiontoprimarystorageclass
 */
var OutOfInfrequentAccessPolicy;
(function (OutOfInfrequentAccessPolicy) {
    /**
     * After 1 access
     */
    OutOfInfrequentAccessPolicy["AFTER_1_ACCESS"] = "AFTER_1_ACCESS";
})(OutOfInfrequentAccessPolicy = exports.OutOfInfrequentAccessPolicy || (exports.OutOfInfrequentAccessPolicy = {}));
/**
 * EFS Performance mode.
 *
 * @see https://docs.aws.amazon.com/efs/latest/ug/performance.html#performancemodes
 */
var PerformanceMode;
(function (PerformanceMode) {
    /**
     * General Purpose is ideal for latency-sensitive use cases, like web serving
     * environments, content management systems, home directories, and general file serving.
     * Recommended for the majority of Amazon EFS file systems.
     */
    PerformanceMode["GENERAL_PURPOSE"] = "generalPurpose";
    /**
     * File systems in the Max I/O mode can scale to higher levels of aggregate
     * throughput and operations per second. This scaling is done with a tradeoff
     * of slightly higher latencies for file metadata operations.
     * Highly parallelized applications and workloads, such as big data analysis,
     * media processing, and genomics analysis, can benefit from this mode.
     */
    PerformanceMode["MAX_IO"] = "maxIO";
})(PerformanceMode = exports.PerformanceMode || (exports.PerformanceMode = {}));
/**
 * EFS Throughput mode.
 *
 * @see https://docs.aws.amazon.com/efs/latest/ug/performance.html#throughput-modes
 */
var ThroughputMode;
(function (ThroughputMode) {
    /**
     * This mode scales as the size of the file system in the standard storage class grows.
     */
    ThroughputMode["BURSTING"] = "bursting";
    /**
     * This mode can instantly provision the throughput of the file system (in MiB/s) independent of the amount of data stored.
     */
    ThroughputMode["PROVISIONED"] = "provisioned";
    /**
    * This mode scales the throughput automatically regardless of file system size.
    */
    ThroughputMode["ELASTIC"] = "elastic";
})(ThroughputMode = exports.ThroughputMode || (exports.ThroughputMode = {}));
class FileSystemBase extends core_1.Resource {
    /**
     * Grant the actions defined in actions to the given grantee
     * on this File System resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: grantee,
            actions: actions,
            resourceArns: [this.fileSystemArn],
        });
    }
    /**
     * Adds a statement to the resource policy associated with this file system.
     * A resource policy will be automatically created upon the first call to `addToResourcePolicy`.
     *
     * Note that this does not work with imported file systems.
     *
     * @param statement The policy statement to add
     */
    addToResourcePolicy(statement) {
        if (!this._resource) {
            return { statementAdded: false };
        }
        this._fileSystemPolicy = this._fileSystemPolicy ?? new iam.PolicyDocument({ statements: [] });
        this._fileSystemPolicy.addStatements(statement);
        return {
            statementAdded: true,
            policyDependable: this,
        };
    }
}
/**
 * The Elastic File System implementation of IFileSystem.
 * It creates a new, empty file system in Amazon Elastic File System (Amazon EFS).
 * It also creates mount target (AWS::EFS::MountTarget) implicitly to mount the
 * EFS file system on an Amazon Elastic Compute Cloud (Amazon EC2) instance or another resource.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html
 *
 * @resource AWS::EFS::FileSystem
 */
class FileSystem extends FileSystemBase {
    /**
     * Import an existing File System from the given properties.
     */
    static fromFileSystemAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_efs_FileSystemAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromFileSystemAttributes);
            }
            throw error;
        }
        return new ImportedFileSystem(scope, id, attrs);
    }
    /**
     * Constructor for creating a new EFS FileSystem.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this._mountTargetsAvailable = new constructs_1.DependencyGroup();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_efs_FileSystemProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, FileSystem);
            }
            throw error;
        }
        if (props.throughputMode === ThroughputMode.PROVISIONED && props.provisionedThroughputPerSecond === undefined) {
            throw new Error('Property provisionedThroughputPerSecond is required when throughputMode is PROVISIONED');
        }
        if (props.throughputMode === ThroughputMode.ELASTIC && props.performanceMode === PerformanceMode.MAX_IO) {
            throw new Error('ThroughputMode ELASTIC is not supported for file systems with performanceMode MAX_IO');
        }
        // we explictly use 'undefined' to represent 'false' to maintain backwards compatibility since
        // its considered an actual change in CloudFormations eyes, even though they have the same meaning.
        const encrypted = props.encrypted ?? (core_1.FeatureFlags.of(this).isEnabled(cxapi.EFS_DEFAULT_ENCRYPTION_AT_REST) ? true : undefined);
        // LifecyclePolicies is an array of lists containing a single policy
        let lifecyclePolicies = [];
        if (props.lifecyclePolicy) {
            lifecyclePolicies.push({ transitionToIa: props.lifecyclePolicy });
        }
        if (props.outOfInfrequentAccessPolicy) {
            lifecyclePolicies.push({ transitionToPrimaryStorageClass: props.outOfInfrequentAccessPolicy });
        }
        this._resource = new efs_generated_1.CfnFileSystem(this, 'Resource', {
            encrypted: encrypted,
            kmsKeyId: props.kmsKey?.keyArn,
            lifecyclePolicies: lifecyclePolicies.length > 0 ? lifecyclePolicies : undefined,
            performanceMode: props.performanceMode,
            throughputMode: props.throughputMode,
            provisionedThroughputInMibps: props.provisionedThroughputPerSecond?.toMebibytes(),
            backupPolicy: props.enableAutomaticBackups ? { status: 'ENABLED' } : undefined,
            fileSystemPolicy: core_1.Lazy.any({ produce: () => this._fileSystemPolicy }),
        });
        this._resource.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = this._resource.ref;
        this.fileSystemArn = this._resource.attrArn;
        this._fileSystemPolicy = props.fileSystemPolicy;
        core_1.Tags.of(this).add('Name', props.fileSystemName || this.node.path);
        const securityGroup = (props.securityGroup || new ec2.SecurityGroup(this, 'EfsSecurityGroup', {
            vpc: props.vpc,
        }));
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        const subnets = props.vpc.selectSubnets(props.vpcSubnets ?? { onePerAz: true });
        // We now have to create the mount target for each of the mentioned subnet
        let mountTargetCount = 0;
        this.mountTargetsAvailable = [];
        subnets.subnetIds.forEach((subnetId) => {
            const mountTarget = new efs_generated_1.CfnMountTarget(this, 'EfsMountTarget' + (++mountTargetCount), {
                fileSystemId: this.fileSystemId,
                securityGroups: Array.of(securityGroup.securityGroupId),
                subnetId,
            });
            this._mountTargetsAvailable.add(mountTarget);
        });
        this.mountTargetsAvailable = this._mountTargetsAvailable;
    }
    /**
     * create access point from this filesystem
     */
    addAccessPoint(id, accessPointOptions = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_efs_AccessPointOptions(accessPointOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAccessPoint);
            }
            throw error;
        }
        return new access_point_1.AccessPoint(this, id, {
            fileSystem: this,
            ...accessPointOptions,
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
FileSystem[_a] = { fqn: "aws-cdk-lib.aws_efs.FileSystem", version: "2.74.0" };
/**
 * The default port File System listens on.
 */
FileSystem.DEFAULT_PORT = 2049;
exports.FileSystem = FileSystem;
class ImportedFileSystem extends FileSystemBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        if (!!attrs.fileSystemId === !!attrs.fileSystemArn) {
            throw new Error('One of fileSystemId or fileSystemArn, but not both, must be provided.');
        }
        this.fileSystemArn = attrs.fileSystemArn ?? core_1.Stack.of(scope).formatArn({
            service: 'elasticfilesystem',
            resource: 'file-system',
            resourceName: attrs.fileSystemId,
        });
        const parsedArn = core_1.Stack.of(scope).splitArn(this.fileSystemArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!parsedArn.resourceName) {
            throw new Error(`Invalid FileSystem Arn ${this.fileSystemArn}`);
        }
        this.fileSystemId = attrs.fileSystemId ?? parsedArn.resourceName;
        this.connections = new ec2.Connections({
            securityGroups: [attrs.securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        this.mountTargetsAvailable = new constructs_1.DependencyGroup();
    }
}
//# sourceMappingURL=data:application/json;base64,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