"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterResourceProvider = void 0;
const path = require("path");
const lambda = require("../../aws-lambda");
const core_1 = require("../../core");
const cr = require("../../custom-resources");
const lambda_layer_node_proxy_agent_1 = require("../../lambda-layer-node-proxy-agent");
const HANDLER_DIR = path.join(__dirname, 'cluster-resource-handler');
const HANDLER_RUNTIME = lambda.Runtime.NODEJS_14_X;
/**
 * A custom resource provider that handles cluster operations. It serves
 * multiple custom resources such as the cluster resource and the fargate
 * resource.
 *
 * @internal
 */
class ClusterResourceProvider extends core_1.NestedStack {
    static getOrCreate(scope, props) {
        const stack = core_1.Stack.of(scope);
        const uid = '@aws-cdk/aws-eks.ClusterResourceProvider';
        return stack.node.tryFindChild(uid) ?? new ClusterResourceProvider(stack, uid, props);
    }
    constructor(scope, id, props) {
        super(scope, id);
        // The NPM dependency proxy-agent is required in order to support proxy routing with the AWS JS SDK.
        const nodeProxyAgentLayer = new lambda_layer_node_proxy_agent_1.NodeProxyAgentLayer(this, 'NodeProxyAgentLayer');
        const onEvent = new lambda.Function(this, 'OnEventHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'onEvent handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            environment: {
                AWS_STS_REGIONAL_ENDPOINTS: 'regional',
                ...props.environment,
            },
            handler: 'index.onEvent',
            timeout: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
            // Allow user to override the layer.
            layers: props.onEventLayer ? [props.onEventLayer] : [nodeProxyAgentLayer],
        });
        const isComplete = new lambda.Function(this, 'IsCompleteHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'isComplete handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            environment: {
                AWS_STS_REGIONAL_ENDPOINTS: 'regional',
                ...props.environment,
            },
            handler: 'index.isComplete',
            timeout: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
            layers: [nodeProxyAgentLayer],
        });
        this.provider = new cr.Provider(this, 'Provider', {
            onEventHandler: onEvent,
            isCompleteHandler: isComplete,
            totalTimeout: core_1.Duration.hours(1),
            queryInterval: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
        });
        props.adminRole.grant(onEvent.role, 'sts:AssumeRole');
        props.adminRole.grant(isComplete.role, 'sts:AssumeRole');
    }
    /**
     * The custom resource service token for this provider.
     */
    get serviceToken() { return this.provider.serviceToken; }
}
exports.ClusterResourceProvider = ClusterResourceProvider;
//# sourceMappingURL=data:application/json;base64,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