"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MachineImageType = exports.DefaultCapacityType = exports.CoreDnsComputeType = exports.CpuArch = exports.NodeType = exports.EksOptimizedImage = exports.Cluster = exports.ClusterLoggingTypes = exports.KubernetesVersion = exports.EndpointAccess = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const autoscaling = require("../../aws-autoscaling");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const ssm = require("../../aws-ssm");
const core_1 = require("../../core");
const constructs_1 = require("constructs");
const semver = require("semver");
const YAML = require("yaml");
const alb_controller_1 = require("./alb-controller");
const aws_auth_1 = require("./aws-auth");
const cluster_resource_1 = require("./cluster-resource");
const fargate_profile_1 = require("./fargate-profile");
const helm_chart_1 = require("./helm-chart");
const instance_types_1 = require("./instance-types");
const k8s_manifest_1 = require("./k8s-manifest");
const k8s_object_value_1 = require("./k8s-object-value");
const k8s_patch_1 = require("./k8s-patch");
const kubectl_provider_1 = require("./kubectl-provider");
const managed_nodegroup_1 = require("./managed-nodegroup");
const oidc_provider_1 = require("./oidc-provider");
const bottlerocket_1 = require("./private/bottlerocket");
const service_account_1 = require("./service-account");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * Endpoint access characteristics.
 */
class EndpointAccess {
    constructor(
    /**
     * Configuration properties.
     *
     * @internal
     */
    _config) {
        this._config = _config;
        if (!_config.publicAccess && _config.publicCidrs && _config.publicCidrs.length > 0) {
            throw new Error('CIDR blocks can only be configured when public access is enabled');
        }
    }
    /**
     * Restrict public access to specific CIDR blocks.
     * If public access is disabled, this method will result in an error.
     *
     * @param cidr CIDR blocks.
     */
    onlyFrom(...cidr) {
        if (!this._config.privateAccess) {
            // when private access is disabled, we can't restric public
            // access since it will render the kubectl provider unusable.
            throw new Error('Cannot restric public access to endpoint when private access is disabled. Use PUBLIC_AND_PRIVATE.onlyFrom() instead.');
        }
        return new EndpointAccess({
            ...this._config,
            // override CIDR
            publicCidrs: cidr,
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
EndpointAccess[_a] = { fqn: "aws-cdk-lib.aws_eks.EndpointAccess", version: "2.74.0" };
/**
 * The cluster endpoint is accessible from outside of your VPC.
 * Worker node traffic will leave your VPC to connect to the endpoint.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @param cidr The CIDR blocks.
 */
EndpointAccess.PUBLIC = new EndpointAccess({ privateAccess: false, publicAccess: true });
/**
 * The cluster endpoint is only accessible through your VPC.
 * Worker node traffic to the endpoint will stay within your VPC.
 */
EndpointAccess.PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: false });
/**
 * The cluster endpoint is accessible from outside of your VPC.
 * Worker node traffic to the endpoint will stay within your VPC.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC_AND_PRIVATE.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @param cidr The CIDR blocks.
 */
EndpointAccess.PUBLIC_AND_PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: true });
exports.EndpointAccess = EndpointAccess;
/**
 * Kubernetes cluster version
 * @see https://docs.aws.amazon.com/eks/latest/userguide/kubernetes-versions.html#kubernetes-release-calendar
 */
class KubernetesVersion {
    /**
     * Custom cluster version
     * @param version custom version number
     */
    static of(version) { return new KubernetesVersion(version); }
    /**
     *
     * @param version cluster version number
     */
    constructor(version) {
        this.version = version;
    }
}
_b = JSII_RTTI_SYMBOL_1;
KubernetesVersion[_b] = { fqn: "aws-cdk-lib.aws_eks.KubernetesVersion", version: "2.74.0" };
/**
 * Kubernetes version 1.14
 * @deprecated Use newer version of EKS
 */
KubernetesVersion.V1_14 = KubernetesVersion.of('1.14');
/**
 * Kubernetes version 1.15
 * @deprecated Use newer version of EKS
 */
KubernetesVersion.V1_15 = KubernetesVersion.of('1.15');
/**
 * Kubernetes version 1.16
 * @deprecated Use newer version of EKS
 */
KubernetesVersion.V1_16 = KubernetesVersion.of('1.16');
/**
 * Kubernetes version 1.17
 * @deprecated Use newer version of EKS
 */
KubernetesVersion.V1_17 = KubernetesVersion.of('1.17');
/**
 * Kubernetes version 1.18
 * @deprecated Use newer version of EKS
 */
KubernetesVersion.V1_18 = KubernetesVersion.of('1.18');
/**
 * Kubernetes version 1.19
 * @deprecated Use newer version of EKS
 */
KubernetesVersion.V1_19 = KubernetesVersion.of('1.19');
/**
 * Kubernetes version 1.20
 * @deprecated Use newer version of EKS
 */
KubernetesVersion.V1_20 = KubernetesVersion.of('1.20');
/**
 * Kubernetes version 1.21
 */
KubernetesVersion.V1_21 = KubernetesVersion.of('1.21');
/**
 * Kubernetes version 1.22
 *
 * When creating a `Cluster` with this version, you need to also specify the
 * `kubectlLayer` property with a `KubectlV22Layer` from
 * `@aws-cdk/lambda-layer-kubectl-v22`.
 */
KubernetesVersion.V1_22 = KubernetesVersion.of('1.22');
/**
 * Kubernetes version 1.23
 *
 * When creating a `Cluster` with this version, you need to also specify the
 * `kubectlLayer` property with a `KubectlV23Layer` from
 * `@aws-cdk/lambda-layer-kubectl-v23`.
 */
KubernetesVersion.V1_23 = KubernetesVersion.of('1.23');
/**
 * Kubernetes version 1.24
 *
 * When creating a `Cluster` with this version, you need to also specify the
 * `kubectlLayer` property with a `KubectlV24Layer` from
 * `@aws-cdk/lambda-layer-kubectl-v24`.
 */
KubernetesVersion.V1_24 = KubernetesVersion.of('1.24');
/**
 * Kubernetes version 1.25
 *
 * When creating a `Cluster` with this version, you need to also specify the
 * `kubectlLayer` property with a `KubectlV25Layer` from
 * `@aws-cdk/lambda-layer-kubectl-v25`.
 */
KubernetesVersion.V1_25 = KubernetesVersion.of('1.25');
exports.KubernetesVersion = KubernetesVersion;
/**
 * EKS cluster logging types
 */
var ClusterLoggingTypes;
(function (ClusterLoggingTypes) {
    /**
     * Logs pertaining to API requests to the cluster.
     */
    ClusterLoggingTypes["API"] = "api";
    /**
     * Logs pertaining to cluster access via the Kubernetes API.
     */
    ClusterLoggingTypes["AUDIT"] = "audit";
    /**
     * Logs pertaining to authentication requests into the cluster.
     */
    ClusterLoggingTypes["AUTHENTICATOR"] = "authenticator";
    /**
     * Logs pertaining to state of cluster controllers.
     */
    ClusterLoggingTypes["CONTROLLER_MANAGER"] = "controllerManager";
    /**
     * Logs pertaining to scheduling decisions.
     */
    ClusterLoggingTypes["SCHEDULER"] = "scheduler";
})(ClusterLoggingTypes = exports.ClusterLoggingTypes || (exports.ClusterLoggingTypes = {}));
class ClusterBase extends core_1.Resource {
    /**
     * Defines a Kubernetes resource in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest
     * @param manifest a list of Kubernetes resource specifications
     * @returns a `KubernetesResource` object.
     */
    addManifest(id, ...manifest) {
        return new k8s_manifest_1.KubernetesManifest(this, `manifest-${id}`, { cluster: this, manifest });
    }
    /**
     * Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` construct
     */
    addHelmChart(id, options) {
        return new helm_chart_1.HelmChart(this, `chart-${id}`, { cluster: this, ...options });
    }
    /**
     * Defines a CDK8s chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param chart the cdk8s chart.
     * @returns a `KubernetesManifest` construct representing the chart.
     */
    addCdk8sChart(id, chart, options = {}) {
        const cdk8sChart = chart;
        // see https://github.com/awslabs/cdk8s/blob/master/packages/cdk8s/src/chart.ts#L84
        if (typeof cdk8sChart.toJson !== 'function') {
            throw new Error(`Invalid cdk8s chart. Must contain a 'toJson' method, but found ${typeof cdk8sChart.toJson}`);
        }
        const manifest = new k8s_manifest_1.KubernetesManifest(this, id, {
            cluster: this,
            manifest: cdk8sChart.toJson(),
            ...options,
        });
        return manifest;
    }
    addServiceAccount(id, options = {}) {
        return new service_account_1.ServiceAccount(this, id, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Installs the AWS spot instance interrupt handler on the cluster if it's not
     * already added.
     */
    addSpotInterruptHandler() {
        if (!this._spotInterruptHandler) {
            this._spotInterruptHandler = this.addHelmChart('spot-interrupt-handler', {
                chart: 'aws-node-termination-handler',
                version: '0.18.0',
                repository: 'https://aws.github.io/eks-charts',
                namespace: 'kube-system',
                values: {
                    nodeSelector: {
                        lifecycle: user_data_1.LifecycleLabel.SPOT,
                    },
                },
            });
        }
        return this._spotInterruptHandler;
    }
    /**
     * Connect capacity in the form of an existing AutoScalingGroup to the EKS cluster.
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addAutoScalingGroupCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    connectAutoScalingGroupCapacity(autoScalingGroup, options) {
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        // allow traffic to/from managed node groups (eks attaches this security group to the managed nodes)
        autoScalingGroup.addSecurityGroup(this.clusterSecurityGroup);
        const bootstrapEnabled = options.bootstrapEnabled ?? true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error('Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false');
        }
        if (bootstrapEnabled) {
            const userData = options.machineImageType === MachineImageType.BOTTLEROCKET ?
                (0, user_data_1.renderBottlerocketUserData)(this) :
                (0, user_data_1.renderAmazonLinuxUserData)(this, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        // https://docs.aws.amazon.com/eks/latest/userguide/worker.html
        core_1.Tags.of(autoScalingGroup).add(`kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true,
            // exclude security groups to avoid multiple "owned" security groups.
            // (the cluster security group already has this tag)
            excludeResourceTypes: ['AWS::EC2::SecurityGroup'],
        });
        // do not attempt to map the role if `kubectl` is not enabled for this
        // cluster or if `mapRole` is set to false. By default this should happen.
        let mapRole = options.mapRole ?? true;
        if (mapRole && !(this instanceof Cluster)) {
            // do the mapping...
            core_1.Annotations.of(autoScalingGroup).addWarning('Auto-mapping aws-auth role for imported cluster is not supported, please map role manually');
            mapRole = false;
        }
        if (mapRole) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.awsAuth.addRoleMapping(autoScalingGroup.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        else {
            // since we are not mapping the instance role to RBAC, synthesize an
            // output so it can be pasted into `aws-auth-cm.yaml`
            new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
                value: autoScalingGroup.role.roleArn,
            });
        }
        const addSpotInterruptHandler = options.spotInterruptHandler ?? true;
        // if this is an ASG with spot instances, install the spot interrupt handler (only if kubectl is enabled).
        if (autoScalingGroup.spotPrice && addSpotInterruptHandler) {
            this.addSpotInterruptHandler();
        }
        if (this instanceof Cluster && this.albController) {
            // the controller runs on the worker nodes so they cannot
            // be deleted before the controller.
            constructs_1.Node.of(this.albController).addDependency(autoScalingGroup);
        }
    }
}
;
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 */
class Cluster extends ClusterBase {
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_eks_ClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromClusterAttributes);
            }
            throw error;
        }
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param id the id of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * If the cluster has one (or more) FargateProfiles associated, this array
         * will hold a reference to each.
         */
        this._fargateProfiles = [];
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_eks_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Cluster);
            }
            throw error;
        }
        const stack = core_1.Stack.of(this);
        this.prune = props.prune ?? true;
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        const kubectlVersion = new semver.SemVer(`${props.version.version}.0`);
        if (semver.gte(kubectlVersion, '1.22.0') && !props.kubectlLayer) {
            core_1.Annotations.of(this).addWarning(`You created a cluster with Kubernetes Version ${props.version.version} without specifying the kubectlLayer property. This may cause failures as the kubectl version provided with aws-cdk-lib is 1.20, which is only guaranteed to be compatible with Kubernetes versions 1.19-1.21. Please provide a kubectlLayer from @aws-cdk/lambda-layer-kubectl-v${kubectlVersion.minor}.`);
        }
        ;
        this.version = props.version;
        this.kubectlLambdaRole = props.kubectlLambdaRole ? props.kubectlLambdaRole : undefined;
        this.tagSubnets();
        // this is the role used by EKS when interacting with AWS resources
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.vpcSubnets = props.vpcSubnets ?? [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS }];
        const selectedSubnetIdsPerGroup = this.vpcSubnets.map(s => this.vpc.selectSubnets(s).subnetIds);
        if (selectedSubnetIdsPerGroup.some(core_1.Token.isUnresolved) && selectedSubnetIdsPerGroup.length > 1) {
            throw new Error('eks.Cluster: cannot select multiple subnet groups from a VPC imported from list tokens with unknown length. Select only one subnet group, pass a length to Fn.split, or switch to Vpc.fromLookup.');
        }
        // Get subnetIds for all selected subnets
        const subnetIds = Array.from(new Set(flatten(selectedSubnetIdsPerGroup)));
        this.logging = props.clusterLogging ? {
            clusterLogging: [
                {
                    enabled: true,
                    types: Object.values(props.clusterLogging),
                },
            ],
        } : undefined;
        this.endpointAccess = props.endpointAccess ?? EndpointAccess.PUBLIC_AND_PRIVATE;
        this.kubectlEnvironment = props.kubectlEnvironment;
        this.kubectlLayer = props.kubectlLayer;
        this.awscliLayer = props.awscliLayer;
        this.kubectlMemory = props.kubectlMemory;
        this.onEventLayer = props.onEventLayer;
        this.clusterHandlerSecurityGroup = props.clusterHandlerSecurityGroup;
        const privateSubnets = this.selectPrivateSubnets().slice(0, 16);
        const publicAccessDisabled = !this.endpointAccess._config.publicAccess;
        const publicAccessRestricted = !publicAccessDisabled
            && this.endpointAccess._config.publicCidrs
            && this.endpointAccess._config.publicCidrs.length !== 0;
        // validate endpoint access configuration
        if (privateSubnets.length === 0 && publicAccessDisabled) {
            // no private subnets and no public access at all, no good.
            throw new Error('Vpc must contain private subnets when public endpoint access is disabled');
        }
        if (privateSubnets.length === 0 && publicAccessRestricted) {
            // no private subnets and public access is restricted, no good.
            throw new Error('Vpc must contain private subnets when public endpoint access is restricted');
        }
        const placeClusterHandlerInVpc = props.placeClusterHandlerInVpc ?? false;
        if (placeClusterHandlerInVpc && privateSubnets.length === 0) {
            throw new Error('Cannot place cluster handler in the VPC since no private subnets could be selected');
        }
        if (props.clusterHandlerSecurityGroup && !placeClusterHandlerInVpc) {
            throw new Error('Cannot specify clusterHandlerSecurityGroup without placeClusterHandlerInVpc set to true');
        }
        const resource = this._clusterResource = new cluster_resource_1.ClusterResource(this, 'Resource', {
            name: this.physicalName,
            environment: props.clusterHandlerEnvironment,
            roleArn: this.role.roleArn,
            version: props.version.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds,
            },
            ...(props.secretsEncryptionKey ? {
                encryptionConfig: [{
                        provider: {
                            keyArn: props.secretsEncryptionKey.keyArn,
                        },
                        resources: ['secrets'],
                    }],
            } : {}),
            kubernetesNetworkConfig: props.serviceIpv4Cidr ? {
                serviceIpv4Cidr: props.serviceIpv4Cidr,
            } : undefined,
            endpointPrivateAccess: this.endpointAccess._config.privateAccess,
            endpointPublicAccess: this.endpointAccess._config.publicAccess,
            publicAccessCidrs: this.endpointAccess._config.publicCidrs,
            secretsEncryptionKey: props.secretsEncryptionKey,
            vpc: this.vpc,
            subnets: placeClusterHandlerInVpc ? privateSubnets : undefined,
            clusterHandlerSecurityGroup: this.clusterHandlerSecurityGroup,
            onEventLayer: this.onEventLayer,
            tags: props.tags,
            logging: this.logging,
        });
        if (this.endpointAccess._config.privateAccess && privateSubnets.length !== 0) {
            // when private access is enabled and the vpc has private subnets, lets connect
            // the provider to the vpc so that it will work even when restricting public access.
            // validate VPC properties according to: https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html
            if (this.vpc instanceof ec2.Vpc && !(this.vpc.dnsHostnamesEnabled && this.vpc.dnsSupportEnabled)) {
                throw new Error('Private endpoint access requires the VPC to have DNS support and DNS hostnames enabled. Use `enableDnsHostnames: true` and `enableDnsSupport: true` when creating the VPC.');
            }
            this.kubectlPrivateSubnets = privateSubnets;
            // the vpc must exist in order to properly delete the cluster (since we run `kubectl delete`).
            // this ensures that.
            this._clusterResource.node.addDependency(this.vpc);
        }
        this.adminRole = resource.adminRole;
        // we use an SSM parameter as a barrier because it's free and fast.
        this._kubectlReadyBarrier = new core_1.CfnResource(this, 'KubectlReadyBarrier', {
            type: 'AWS::SSM::Parameter',
            properties: {
                Type: 'String',
                Value: 'aws:cdk:eks:kubectl-ready',
            },
        });
        // add the cluster resource itself as a dependency of the barrier
        this._kubectlReadyBarrier.node.addDependency(this._clusterResource);
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, (0, cluster_resource_1.clusterArnComponents)(this.physicalName));
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        this.clusterSecurityGroupId = resource.attrClusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = resource.attrEncryptionConfigKeyArn;
        this.clusterSecurityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'ClusterSecurityGroup', this.clusterSecurityGroupId);
        this.connections = new ec2.Connections({
            securityGroups: [this.clusterSecurityGroup, securityGroup],
            defaultPort: ec2.Port.tcp(443), // Control Plane has an HTTPS API
        });
        // we can use the cluster security group since its already attached to the cluster
        // and configured to allow connections from itself.
        this.kubectlSecurityGroup = this.clusterSecurityGroup;
        // use the cluster creation role to issue kubectl commands against the cluster because when the
        // cluster is first created, that's the only role that has "system:masters" permissions
        this.kubectlRole = this.adminRole;
        this._kubectlResourceProvider = this.defineKubectlProvider();
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // if an explicit role is not configured, define a masters role that can
        // be assumed by anyone in the account (with sts:AssumeRole permissions of
        // course)
        const mastersRole = props.mastersRole ?? new iam.Role(this, 'MastersRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // map the IAM role to the `system:masters` group.
        this.awsAuth.addMastersRole(mastersRole);
        if (props.outputMastersRoleArn) {
            new core_1.CfnOutput(this, 'MastersRoleArn', { value: mastersRole.roleArn });
        }
        commonCommandOptions.push(`--role-arn ${mastersRole.roleArn}`);
        if (props.albController) {
            this.albController = alb_controller_1.AlbController.create(this, { ...props.albController, cluster: this });
        }
        // allocate default capacity if non-zero (or default).
        const minCapacity = props.defaultCapacity ?? DEFAULT_CAPACITY_COUNT;
        if (minCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = props.defaultCapacityType === DefaultCapacityType.EC2 ?
                this.addAutoScalingGroupCapacity('DefaultCapacity', { instanceType, minCapacity }) : undefined;
            this.defaultNodegroup = props.defaultCapacityType !== DefaultCapacityType.EC2 ?
                this.addNodegroupCapacity('DefaultCapacity', { instanceTypes: [instanceType], minSize: minCapacity }) : undefined;
        }
        const outputConfigCommand = props.outputConfigCommand ?? true;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
        this.defineCoreDnsComputeType(props.coreDnsComputeType ?? CoreDnsComputeType.EC2);
    }
    /**
     * Fetch the load balancer address of a service of type 'LoadBalancer'.
     *
     * @param serviceName The name of the service.
     * @param options Additional operation options.
     */
    getServiceLoadBalancerAddress(serviceName, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_eks_ServiceLoadBalancerAddressOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getServiceLoadBalancerAddress);
            }
            throw error;
        }
        const loadBalancerAddress = new k8s_object_value_1.KubernetesObjectValue(this, `${serviceName}LoadBalancerAddress`, {
            cluster: this,
            objectType: 'service',
            objectName: serviceName,
            objectNamespace: options.namespace,
            jsonPath: '.status.loadBalancer.ingress[0].hostname',
            timeout: options.timeout,
        });
        return loadBalancerAddress.value;
    }
    /**
     * Fetch the load balancer address of an ingress backed by a load balancer.
     *
     * @param ingressName The name of the ingress.
     * @param options Additional operation options.
     */
    getIngressLoadBalancerAddress(ingressName, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_eks_IngressLoadBalancerAddressOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getIngressLoadBalancerAddress);
            }
            throw error;
        }
        const loadBalancerAddress = new k8s_object_value_1.KubernetesObjectValue(this, `${ingressName}LoadBalancerAddress`, {
            cluster: this,
            objectType: 'ingress',
            objectName: ingressName,
            objectNamespace: options.namespace,
            jsonPath: '.status.loadBalancer.ingress[0].hostname',
            timeout: options.timeout,
        });
        return loadBalancerAddress.value;
    }
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Note that if you specify `updateType: RollingUpdate` or `updateType: ReplacingUpdate`, your nodes might be replaced at deploy
     * time without notice in case the recommended AMI for your machine image type has been updated by AWS.
     * The default behavior for `updateType` is `None`, which means only new instances will be launched using the new AMI.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * In addition, the [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     */
    addAutoScalingGroupCapacity(id, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_eks_AutoScalingGroupCapacityOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAutoScalingGroupCapacity);
            }
            throw error;
        }
        if (options.machineImageType === MachineImageType.BOTTLEROCKET && options.bootstrapOptions !== undefined) {
            throw new Error('bootstrapOptions is not supported for Bottlerocket');
        }
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: options.machineImageType === MachineImageType.BOTTLEROCKET ?
                new bottlerocket_1.BottleRocketImage({
                    kubernetesVersion: this.version.version,
                }) :
                new EksOptimizedImage({
                    nodeType: nodeTypeForInstanceType(options.instanceType),
                    cpuArch: cpuArchForInstanceType(options.instanceType),
                    kubernetesVersion: this.version.version,
                }),
        });
        this.connectAutoScalingGroupCapacity(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled,
            machineImageType: options.machineImageType,
            spotInterruptHandler: options.spotInterruptHandler,
        });
        if (nodeTypeForInstanceType(options.instanceType) === NodeType.INFERENTIA) {
            this.addNeuronDevicePlugin();
        }
        return asg;
    }
    /**
     * Add managed nodegroup to this Amazon EKS cluster
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @param id The ID of the nodegroup
     * @param options options for creating a new nodegroup
     */
    addNodegroupCapacity(id, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_eks_NodegroupOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addNodegroupCapacity);
            }
            throw error;
        }
        return new managed_nodegroup_1.Nodegroup(this, `Nodegroup${id}`, {
            cluster: this,
            ...options,
        });
    }
    /**
     * Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     */
    get awsAuth() {
        if (!this._awsAuth) {
            this._awsAuth = new aws_auth_1.AwsAuth(this, 'AwsAuth', { cluster: this });
        }
        return this._awsAuth;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer url.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuerUrl() {
        return this._clusterResource.attrOpenIdConnectIssuerUrl;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuer() {
        return this._clusterResource.attrOpenIdConnectIssuer;
    }
    /**
     * An `OpenIdConnectProvider` resource associated with this cluster, and which can be used
     * to link this cluster to AWS IAM.
     *
     * A provider will only be defined if this property is accessed (lazy initialization).
     */
    get openIdConnectProvider() {
        if (!this._openIdConnectProvider) {
            this._openIdConnectProvider = new oidc_provider_1.OpenIdConnectProvider(this, 'OpenIdConnectProvider', {
                url: this.clusterOpenIdConnectIssuerUrl,
            });
        }
        return this._openIdConnectProvider;
    }
    /**
     * Adds a Fargate profile to this cluster.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param id the id of this profile
     * @param options profile options
     */
    addFargateProfile(id, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_eks_FargateProfileOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addFargateProfile);
            }
            throw error;
        }
        return new fargate_profile_1.FargateProfile(this, `fargate-profile-${id}`, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Internal API used by `FargateProfile` to keep inventory of Fargate profiles associated with
     * this cluster, for the sake of ensuring the profiles are created sequentially.
     *
     * @returns the list of FargateProfiles attached to this cluster, including the one just attached.
     * @internal
     */
    _attachFargateProfile(fargateProfile) {
        this._fargateProfiles.push(fargateProfile);
        // add all profiles as a dependency of the "kubectl-ready" barrier because all kubectl-
        // resources can only be deployed after all fargate profiles are created.
        this._kubectlReadyBarrier.node.addDependency(fargateProfile);
        return this._fargateProfiles;
    }
    /**
     * Adds a resource scope that requires `kubectl` to this cluster and returns
     * the `KubectlProvider` which is the custom resource provider that should be
     * used as the resource provider.
     *
     * Called from `HelmResource` and `KubernetesResource`
     *
     * @param resourceScope the construct scope in which kubectl resources are defined.
     *
     * @internal
     */
    _attachKubectlResourceScope(resourceScope) {
        constructs_1.Node.of(resourceScope).addDependency(this._kubectlReadyBarrier);
        return this._kubectlResourceProvider;
    }
    defineKubectlProvider() {
        const uid = '@aws-cdk/aws-eks.KubectlProvider';
        // since we can't have the provider connect to multiple networks, and we
        // wanted to avoid resource tear down, we decided for now that we will only
        // support a single EKS cluster per CFN stack.
        if (this.stack.node.tryFindChild(uid)) {
            throw new Error('Only a single EKS cluster can be defined within a CloudFormation stack');
        }
        return new kubectl_provider_1.KubectlProvider(this.stack, uid, { cluster: this });
    }
    selectPrivateSubnets() {
        const privateSubnets = [];
        const vpcPrivateSubnetIds = this.vpc.privateSubnets.map(s => s.subnetId);
        const vpcPublicSubnetIds = this.vpc.publicSubnets.map(s => s.subnetId);
        for (const placement of this.vpcSubnets) {
            for (const subnet of this.vpc.selectSubnets(placement).subnets) {
                if (vpcPrivateSubnetIds.includes(subnet.subnetId)) {
                    // definitely private, take it.
                    privateSubnets.push(subnet);
                    continue;
                }
                if (vpcPublicSubnetIds.includes(subnet.subnetId)) {
                    // definitely public, skip it.
                    continue;
                }
                // neither public and nor private - what is it then? this means its a subnet instance that was explicitly passed
                // in the subnet selection. since ISubnet doesn't contain information on type, we have to assume its private and let it
                // fail at deploy time :\ (its better than filtering it out and preventing a possibly successful deployment)
                privateSubnets.push(subnet);
            }
        }
        return privateSubnets;
    }
    /**
     * Installs the Neuron device plugin on the cluster if it's not
     * already added.
     */
    addNeuronDevicePlugin() {
        if (!this._neuronDevicePlugin) {
            const fileContents = fs.readFileSync(path.join(__dirname, 'addons/neuron-device-plugin.yaml'), 'utf8');
            const sanitized = YAML.parse(fileContents);
            this._neuronDevicePlugin = this.addManifest('NeuronDevicePlugin', sanitized);
        }
        return this._neuronDevicePlugin;
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) || core_1.Token.isUnresolved([subnet.subnetId]) ? '' : ` ${subnet.subnetId}`;
                    core_1.Annotations.of(this).addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                core_1.Tags.of(subnet).add(tag, '1');
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, 'kubernetes.io/role/internal-elb');
        tagAllSubnets('public', this.vpc.publicSubnets, 'kubernetes.io/role/elb');
    }
    /**
     * Patches the CoreDNS deployment configuration and sets the "eks.amazonaws.com/compute-type"
     * annotation to either "ec2" or "fargate". Note that if "ec2" is selected, the resource is
     * omitted/removed, since the cluster is created with the "ec2" compute type by default.
     */
    defineCoreDnsComputeType(type) {
        // ec2 is the "built in" compute type of the cluster so if this is the
        // requested type we can simply omit the resource. since the resource's
        // `restorePatch` is configured to restore the value to "ec2" this means
        // that deletion of the resource will change to "ec2" as well.
        if (type === CoreDnsComputeType.EC2) {
            return;
        }
        // this is the json patch we merge into the resource based off of:
        // https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns
        const renderPatch = (computeType) => ({
            spec: {
                template: {
                    metadata: {
                        annotations: {
                            'eks.amazonaws.com/compute-type': computeType,
                        },
                    },
                },
            },
        });
        new k8s_patch_1.KubernetesPatch(this, 'CoreDnsComputeTypePatch', {
            cluster: this,
            resourceName: 'deployment/coredns',
            resourceNamespace: 'kube-system',
            applyPatch: renderPatch(CoreDnsComputeType.FARGATE),
            restorePatch: renderPatch(CoreDnsComputeType.EC2),
        });
    }
}
_c = JSII_RTTI_SYMBOL_1;
Cluster[_c] = { fqn: "aws-cdk-lib.aws_eks.Cluster", version: "2.74.0" };
exports.Cluster = Cluster;
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends ClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.clusterArn = this.stack.formatArn((0, cluster_resource_1.clusterArnComponents)(props.clusterName));
        this.kubectlRole = props.kubectlRoleArn ? iam.Role.fromRoleArn(this, 'KubectlRole', props.kubectlRoleArn) : undefined;
        this.kubectlLambdaRole = props.kubectlLambdaRole;
        this.kubectlSecurityGroup = props.kubectlSecurityGroupId ? ec2.SecurityGroup.fromSecurityGroupId(this, 'KubectlSecurityGroup', props.kubectlSecurityGroupId) : undefined;
        this.kubectlEnvironment = props.kubectlEnvironment;
        this.kubectlPrivateSubnets = props.kubectlPrivateSubnetIds ? props.kubectlPrivateSubnetIds.map((subnetid, index) => ec2.Subnet.fromSubnetId(this, `KubectlSubnet${index}`, subnetid)) : undefined;
        this.kubectlLayer = props.kubectlLayer;
        this.awscliLayer = props.awscliLayer;
        this.kubectlMemory = props.kubectlMemory;
        this.clusterHandlerSecurityGroup = props.clusterHandlerSecurityGroupId ? ec2.SecurityGroup.fromSecurityGroupId(this, 'ClusterHandlerSecurityGroup', props.clusterHandlerSecurityGroupId) : undefined;
        this.kubectlProvider = props.kubectlProvider;
        this.onEventLayer = props.onEventLayer;
        this.prune = props.prune ?? true;
        let i = 1;
        for (const sgid of props.securityGroupIds ?? []) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgid));
            i++;
        }
        if (props.clusterSecurityGroupId) {
            this._clusterSecurityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'ClusterSecurityGroup', this.clusterSecurityGroupId);
            this.connections.addSecurityGroup(this._clusterSecurityGroup);
        }
    }
    get vpc() {
        if (!this.props.vpc) {
            throw new Error('"vpc" is not defined for this imported cluster');
        }
        return this.props.vpc;
    }
    get clusterSecurityGroup() {
        if (!this._clusterSecurityGroup) {
            throw new Error('"clusterSecurityGroup" is not defined for this imported cluster');
        }
        return this._clusterSecurityGroup;
    }
    get clusterSecurityGroupId() {
        if (!this.props.clusterSecurityGroupId) {
            throw new Error('"clusterSecurityGroupId" is not defined for this imported cluster');
        }
        return this.props.clusterSecurityGroupId;
    }
    get clusterEndpoint() {
        if (!this.props.clusterEndpoint) {
            throw new Error('"clusterEndpoint" is not defined for this imported cluster');
        }
        return this.props.clusterEndpoint;
    }
    get clusterCertificateAuthorityData() {
        if (!this.props.clusterCertificateAuthorityData) {
            throw new Error('"clusterCertificateAuthorityData" is not defined for this imported cluster');
        }
        return this.props.clusterCertificateAuthorityData;
    }
    get clusterEncryptionConfigKeyArn() {
        if (!this.props.clusterEncryptionConfigKeyArn) {
            throw new Error('"clusterEncryptionConfigKeyArn" is not defined for this imported cluster');
        }
        return this.props.clusterEncryptionConfigKeyArn;
    }
    get openIdConnectProvider() {
        if (!this.props.openIdConnectProvider) {
            throw new Error('"openIdConnectProvider" is not defined for this imported cluster');
        }
        return this.props.openIdConnectProvider;
    }
    get awsAuth() {
        throw new Error('"awsAuth" is not supported on imported clusters');
    }
}
/**
 * Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM
 */
class EksOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_eks_EksOptimizedImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EksOptimizedImage);
            }
            throw error;
        }
        this.nodeType = props.nodeType ?? NodeType.STANDARD;
        this.cpuArch = props.cpuArch ?? CpuArch.X86_64;
        this.kubernetesVersion = props.kubernetesVersion ?? LATEST_KUBERNETES_VERSION;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/eks/optimized-ami/${this.kubernetesVersion}/`
            + (this.nodeType === NodeType.STANDARD ? this.cpuArch === CpuArch.X86_64 ?
                'amazon-linux-2/' : 'amazon-linux-2-arm64/' : '')
            + (this.nodeType === NodeType.GPU ? 'amazon-linux-2-gpu/' : '')
            + (this.nodeType === NodeType.INFERENTIA ? 'amazon-linux-2-gpu/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.forLinux(),
        };
    }
}
_d = JSII_RTTI_SYMBOL_1;
EksOptimizedImage[_d] = { fqn: "aws-cdk-lib.aws_eks.EksOptimizedImage", version: "2.74.0" };
exports.EksOptimizedImage = EksOptimizedImage;
// MAINTAINERS: use ./scripts/kube_bump.sh to update LATEST_KUBERNETES_VERSION
const LATEST_KUBERNETES_VERSION = '1.24';
/**
 * Whether the worker nodes should support GPU or just standard instances
 */
var NodeType;
(function (NodeType) {
    /**
     * Standard instances
     */
    NodeType["STANDARD"] = "Standard";
    /**
     * GPU instances
     */
    NodeType["GPU"] = "GPU";
    /**
     * Inferentia instances
     */
    NodeType["INFERENTIA"] = "INFERENTIA";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * CPU architecture
 */
var CpuArch;
(function (CpuArch) {
    /**
     * arm64 CPU type
     */
    CpuArch["ARM_64"] = "arm64";
    /**
     * x86_64 CPU type
     */
    CpuArch["X86_64"] = "x86_64";
})(CpuArch = exports.CpuArch || (exports.CpuArch = {}));
/**
 * The type of compute resources to use for CoreDNS.
 */
var CoreDnsComputeType;
(function (CoreDnsComputeType) {
    /**
     * Deploy CoreDNS on EC2 instances.
     */
    CoreDnsComputeType["EC2"] = "ec2";
    /**
     * Deploy CoreDNS on Fargate-managed instances.
     */
    CoreDnsComputeType["FARGATE"] = "fargate";
})(CoreDnsComputeType = exports.CoreDnsComputeType || (exports.CoreDnsComputeType = {}));
/**
 * The default capacity type for the cluster
 */
var DefaultCapacityType;
(function (DefaultCapacityType) {
    /**
     * managed node group
     */
    DefaultCapacityType[DefaultCapacityType["NODEGROUP"] = 0] = "NODEGROUP";
    /**
     * EC2 autoscaling group
     */
    DefaultCapacityType[DefaultCapacityType["EC2"] = 1] = "EC2";
})(DefaultCapacityType = exports.DefaultCapacityType || (exports.DefaultCapacityType = {}));
/**
 * The machine image type
 */
var MachineImageType;
(function (MachineImageType) {
    /**
     * Amazon EKS-optimized Linux AMI
     */
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    /**
     * Bottlerocket AMI
     */
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
function nodeTypeForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.gpu.includes(instanceType.toString().substring(0, 2)) ? NodeType.GPU :
        instance_types_1.INSTANCE_TYPES.inferentia.includes(instanceType.toString().substring(0, 4)) ? NodeType.INFERENTIA :
            NodeType.STANDARD;
}
function cpuArchForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.graviton2.includes(instanceType.toString().substring(0, 3)) ? CpuArch.ARM_64 :
        instance_types_1.INSTANCE_TYPES.graviton3.includes(instanceType.toString().substring(0, 3)) ? CpuArch.ARM_64 :
            instance_types_1.INSTANCE_TYPES.graviton.includes(instanceType.toString().substring(0, 2)) ? CpuArch.ARM_64 :
                CpuArch.X86_64;
}
function flatten(xss) {
    return Array.prototype.concat.call([], ...xss);
}
//# sourceMappingURL=data:application/json;base64,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