"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateProfile = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const constructs_1 = require("constructs");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * Fargate profiles allows an administrator to declare which pods run on
 * Fargate. This declaration is done through the profile’s selectors. Each
 * profile can have up to five selectors that contain a namespace and optional
 * labels. You must define a namespace for every selector. The label field
 * consists of multiple optional key-value pairs. Pods that match a selector (by
 * matching a namespace for the selector and all of the labels specified in the
 * selector) are scheduled on Fargate. If a namespace selector is defined
 * without any labels, Amazon EKS will attempt to schedule all pods that run in
 * that namespace onto Fargate using the profile. If a to-be-scheduled pod
 * matches any of the selectors in the Fargate profile, then that pod is
 * scheduled on Fargate.
 *
 * If a pod matches multiple Fargate profiles, Amazon EKS picks one of the
 * matches at random. In this case, you can specify which profile a pod should
 * use by adding the following Kubernetes label to the pod specification:
 * eks.amazonaws.com/fargate-profile: profile_name. However, the pod must still
 * match a selector in that profile in order to be scheduled onto Fargate.
 */
class FargateProfile extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_eks_FargateProfileProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, FargateProfile);
            }
            throw error;
        }
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this, {
            adminRole: props.cluster.adminRole,
            onEventLayer: props.cluster.onEventLayer,
        });
        this.podExecutionRole = props.podExecutionRole ?? new iam.Role(this, 'PodExecutionRole', {
            assumedBy: new iam.ServicePrincipal('eks-fargate-pods.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSFargatePodExecutionRolePolicy')],
        });
        this.podExecutionRole.grantPassRole(props.cluster.adminRole);
        if (props.subnetSelection && !props.vpc) {
            core_1.Annotations.of(this).addWarning('Vpc must be defined to use a custom subnet selection. All private subnets belonging to the EKS cluster will be used by default');
        }
        let subnets;
        if (props.vpc) {
            const selection = props.subnetSelection ?? { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS };
            subnets = props.vpc.selectSubnets(selection).subnetIds;
        }
        if (props.selectors.length < 1) {
            throw new Error('Fargate profile requires at least one selector');
        }
        if (props.selectors.length > 5) {
            throw new Error('Fargate profile supports up to five selectors');
        }
        this.tags = new core_1.TagManager(core_1.TagType.MAP, 'AWS::EKS::FargateProfile');
        const resource = new core_1.CustomResource(this, 'Resource', {
            serviceToken: provider.serviceToken,
            resourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
            properties: {
                AssumeRoleArn: props.cluster.adminRole.roleArn,
                Config: {
                    clusterName: props.cluster.clusterName,
                    fargateProfileName: props.fargateProfileName,
                    podExecutionRoleArn: this.podExecutionRole.roleArn,
                    selectors: props.selectors,
                    subnets,
                    tags: core_1.Lazy.any({ produce: () => this.tags.renderTags() }),
                },
            },
        });
        this.fargateProfileArn = resource.getAttString('fargateProfileArn');
        this.fargateProfileName = resource.ref;
        // Fargate profiles must be created sequentially. If other profile(s) already
        // exist on the same cluster, create a dependency to force sequential creation.
        const clusterFargateProfiles = props.cluster._attachFargateProfile(this);
        if (clusterFargateProfiles.length > 1) {
            const previousProfile = clusterFargateProfiles[clusterFargateProfiles.length - 2];
            resource.node.addDependency(previousProfile);
        }
        // map the fargate pod execution role to the relevant groups in rbac
        // see https://github.com/aws/aws-cdk/issues/7981
        props.cluster.awsAuth.addRoleMapping(this.podExecutionRole, {
            username: 'system:node:{{SessionName}}',
            groups: [
                'system:bootstrappers',
                'system:nodes',
                'system:node-proxier',
            ],
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
FargateProfile[_a] = { fqn: "aws-cdk-lib.aws_eks.FargateProfile", version: "2.74.0" };
exports.FargateProfile = FargateProfile;
//# sourceMappingURL=data:application/json;base64,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