"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Nodegroup = exports.TaintEffect = exports.CapacityType = exports.NodegroupAmiType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("../../aws-ec2");
const aws_iam_1 = require("../../aws-iam");
const core_1 = require("../../core");
const constructs_1 = require("constructs");
const cluster_1 = require("./cluster");
const eks_generated_1 = require("./eks.generated");
/**
 * The AMI type for your node group. GPU instance types should use the `AL2_x86_64_GPU` AMI type, which uses the
 * Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the `AL2_x86_64` AMI type, which
 * uses the Amazon EKS-optimized Linux AMI.
 */
var NodegroupAmiType;
(function (NodegroupAmiType) {
    /**
     * Amazon Linux 2 (x86-64)
     */
    NodegroupAmiType["AL2_X86_64"] = "AL2_x86_64";
    /**
     * Amazon Linux 2 with GPU support
     */
    NodegroupAmiType["AL2_X86_64_GPU"] = "AL2_x86_64_GPU";
    /**
     * Amazon Linux 2 (ARM-64)
     */
    NodegroupAmiType["AL2_ARM_64"] = "AL2_ARM_64";
    /**
     *  Bottlerocket Linux(ARM-64)
     */
    NodegroupAmiType["BOTTLEROCKET_ARM_64"] = "BOTTLEROCKET_ARM_64";
    /**
     * Bottlerocket(x86-64)
     */
    NodegroupAmiType["BOTTLEROCKET_X86_64"] = "BOTTLEROCKET_x86_64";
})(NodegroupAmiType = exports.NodegroupAmiType || (exports.NodegroupAmiType = {}));
/**
 * Capacity type of the managed node group
 */
var CapacityType;
(function (CapacityType) {
    /**
     * spot instances
     */
    CapacityType["SPOT"] = "SPOT";
    /**
     * on-demand instances
     */
    CapacityType["ON_DEMAND"] = "ON_DEMAND";
})(CapacityType = exports.CapacityType || (exports.CapacityType = {}));
/**
 * Effect types of kubernetes node taint.
 */
var TaintEffect;
(function (TaintEffect) {
    /**
     * NoSchedule
     */
    TaintEffect["NO_SCHEDULE"] = "NO_SCHEDULE";
    /**
     * PreferNoSchedule
     */
    TaintEffect["PREFER_NO_SCHEDULE"] = "PREFER_NO_SCHEDULE";
    /**
     * NoExecute
     */
    TaintEffect["NO_EXECUTE"] = "NO_EXECUTE";
})(TaintEffect = exports.TaintEffect || (exports.TaintEffect = {}));
/**
 * The Nodegroup resource class
 */
class Nodegroup extends core_1.Resource {
    /**
     * Import the Nodegroup from attributes
     */
    static fromNodegroupName(scope, id, nodegroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.nodegroupName = nodegroupName;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.nodegroupName,
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_eks_NodegroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Nodegroup);
            }
            throw error;
        }
        this.cluster = props.cluster;
        this.desiredSize = props.desiredSize ?? props.minSize ?? 2;
        this.maxSize = props.maxSize ?? this.desiredSize;
        this.minSize = props.minSize ?? 1;
        (0, core_1.withResolved)(this.desiredSize, this.maxSize, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (desired > max) {
                throw new Error(`Desired capacity ${desired} can't be greater than max size ${max}`);
            }
        });
        (0, core_1.withResolved)(this.desiredSize, this.minSize, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Minimum capacity ${min} can't be greater than desired size ${desired}`);
            }
        });
        if (props.launchTemplateSpec && props.diskSize) {
            // see - https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html
            // and https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-disksize
            throw new Error('diskSize must be specified within the launch template');
        }
        if (props.instanceType && props.instanceTypes) {
            throw new Error('"instanceType is deprecated, please use "instanceTypes" only.');
        }
        if (props.instanceType) {
            core_1.Annotations.of(this).addWarning('"instanceType" is deprecated and will be removed in the next major version. please use "instanceTypes" instead');
        }
        const instanceTypes = props.instanceTypes ?? (props.instanceType ? [props.instanceType] : undefined);
        let possibleAmiTypes = [];
        if (instanceTypes && instanceTypes.length > 0) {
            /**
             * if the user explicitly configured instance types, we can't caculate the expected ami type as we support
             * Amazon Linux 2 and Bottlerocket now. However we can check:
             *
             * 1. instance types of different CPU architectures are not mixed(e.g. X86 with ARM).
             * 2. user-specified amiType should be included in `possibleAmiTypes`.
             */
            possibleAmiTypes = getPossibleAmiTypes(instanceTypes);
            // if the user explicitly configured an ami type, make sure it's included in the possibleAmiTypes
            if (props.amiType && !possibleAmiTypes.includes(props.amiType)) {
                throw new Error(`The specified AMI does not match the instance types architecture, either specify one of ${possibleAmiTypes} or don't specify any`);
            }
        }
        if (!props.nodeRole) {
            const ngRole = new aws_iam_1.Role(this, 'NodeGroupRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            });
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
            this.role = ngRole;
        }
        else {
            this.role = props.nodeRole;
        }
        const resource = new eks_generated_1.CfnNodegroup(this, 'Resource', {
            clusterName: this.cluster.clusterName,
            nodegroupName: props.nodegroupName,
            nodeRole: this.role.roleArn,
            subnets: this.cluster.vpc.selectSubnets(props.subnets).subnetIds,
            /**
             * Case 1: If launchTemplate is explicitly specified with custom AMI, we cannot specify amiType, or the node group deployment will fail.
             * As we don't know if the custom AMI is specified in the lauchTemplate, we just use props.amiType.
             *
             * Case 2: If launchTemplate is not specified, we try to determine amiType from the instanceTypes and it could be either AL2 or Bottlerocket.
             * To avoid breaking changes, we use possibleAmiTypes[0] if amiType is undefined and make sure AL2 is always the first element in possibleAmiTypes
             * as AL2 is previously the `expectedAmi` and this avoids breaking changes.
             *
             * That being said, users now either have to explicitly specify correct amiType or just leave it undefined.
             */
            amiType: props.launchTemplateSpec ? props.amiType : (props.amiType ?? possibleAmiTypes[0]),
            capacityType: props.capacityType ? props.capacityType.valueOf() : undefined,
            diskSize: props.diskSize,
            forceUpdateEnabled: props.forceUpdate ?? true,
            // note that we don't check if a launch template is configured here (even though it might configure instance types as well)
            // because this doesn't have a default value, meaning the user had to explicitly configure this.
            instanceTypes: instanceTypes?.map(t => t.toString()),
            labels: props.labels,
            taints: props.taints,
            launchTemplate: props.launchTemplateSpec,
            releaseVersion: props.releaseVersion,
            remoteAccess: props.remoteAccess ? {
                ec2SshKey: props.remoteAccess.sshKeyName,
                sourceSecurityGroups: props.remoteAccess.sourceSecurityGroups ?
                    props.remoteAccess.sourceSecurityGroups.map(m => m.securityGroupId) : undefined,
            } : undefined,
            scalingConfig: {
                desiredSize: this.desiredSize,
                maxSize: this.maxSize,
                minSize: this.minSize,
            },
            tags: props.tags,
        });
        // managed nodegroups update the `aws-auth` on creation, but we still need to track
        // its state for consistency.
        if (this.cluster instanceof cluster_1.Cluster) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.cluster.awsAuth.addRoleMapping(this.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
            // the controller runs on the worker nodes so they cannot
            // be deleted before the controller.
            if (this.cluster.albController) {
                constructs_1.Node.of(this.cluster.albController).addDependency(this);
            }
        }
        this.nodegroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'eks',
            resource: 'nodegroup',
            resourceName: this.physicalName,
        });
        this.nodegroupName = this.getResourceNameAttribute(resource.ref);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Nodegroup[_a] = { fqn: "aws-cdk-lib.aws_eks.Nodegroup", version: "2.74.0" };
exports.Nodegroup = Nodegroup;
/**
 * AMI types of different architectures. Make sure AL2 is always the first element, which will be the default
 * AmiType if amiType and launchTemplateSpec are both undefined.
 */
const arm64AmiTypes = [NodegroupAmiType.AL2_ARM_64, NodegroupAmiType.BOTTLEROCKET_ARM_64];
const x8664AmiTypes = [NodegroupAmiType.AL2_X86_64, NodegroupAmiType.BOTTLEROCKET_X86_64];
const gpuAmiTypes = [NodegroupAmiType.AL2_X86_64_GPU];
/**
 * This function check if the instanceType is GPU instance.
 * @param instanceType The EC2 instance type
 */
function isGpuInstanceType(instanceType) {
    // capture the family, generation, capabilities, and size portions of the instance type id
    const instanceTypeComponents = instanceType.toString().match(/^([a-z]+)(\d{1,2})([a-z]*)\.([a-z0-9]+)$/);
    if (instanceTypeComponents == null) {
        throw new Error('Malformed instance type identifier');
    }
    const family = instanceTypeComponents[1];
    return ['p', 'g', 'inf'].includes(family);
}
/**
 * This function examines the CPU architecture of every instance type and determines
 * what AMI types are compatible for all of them. it either throws or produces an array of possible AMI types because
 * instance types of different CPU architectures are not supported.
 * @param instanceTypes The instance types
 * @returns NodegroupAmiType[]
 */
function getPossibleAmiTypes(instanceTypes) {
    function typeToArch(instanceType) {
        return isGpuInstanceType(instanceType) ? 'GPU' : instanceType.architecture;
    }
    const archAmiMap = new Map([
        [aws_ec2_1.InstanceArchitecture.ARM_64, arm64AmiTypes],
        [aws_ec2_1.InstanceArchitecture.X86_64, x8664AmiTypes],
        ['GPU', gpuAmiTypes],
    ]);
    const architectures = new Set(instanceTypes.map(typeToArch));
    if (architectures.size === 0) { // protective code, the current implementation will never result in this.
        throw new Error(`Cannot determine any ami type comptaible with instance types: ${instanceTypes.map(i => i.toString).join(',')}`);
    }
    if (architectures.size > 1) {
        throw new Error('instanceTypes of different architectures is not allowed');
    }
    return archAmiMap.get(Array.from(architectures)[0]);
}
//# sourceMappingURL=data:application/json;base64,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