"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceAccount = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("../../aws-iam");
const core_1 = require("../../core");
const constructs_1 = require("constructs");
const k8s_manifest_1 = require("./k8s-manifest");
/**
 * Service Account
 */
class ServiceAccount extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_eks_ServiceAccountProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ServiceAccount);
            }
            throw error;
        }
        const { cluster } = props;
        this.serviceAccountName = props.name ?? core_1.Names.uniqueId(this).toLowerCase();
        this.serviceAccountNamespace = props.namespace ?? 'default';
        // From K8s docs: https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/
        if (!this.isValidDnsSubdomainName(this.serviceAccountName)) {
            throw RangeError('The name of a ServiceAccount object must be a valid DNS subdomain name.');
        }
        // From K8s docs: https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/#namespaces-and-dns
        if (!this.isValidDnsLabelName(this.serviceAccountNamespace)) {
            throw RangeError('All namespace names must be valid RFC 1123 DNS labels.');
        }
        /* Add conditions to the role to improve security. This prevents other pods in the same namespace to assume the role.
        * See documentation: https://docs.aws.amazon.com/eks/latest/userguide/create-service-account-iam-policy-and-role.html
        */
        const conditions = new core_1.CfnJson(this, 'ConditionJson', {
            value: {
                [`${cluster.openIdConnectProvider.openIdConnectProviderIssuer}:aud`]: 'sts.amazonaws.com',
                [`${cluster.openIdConnectProvider.openIdConnectProviderIssuer}:sub`]: `system:serviceaccount:${this.serviceAccountNamespace}:${this.serviceAccountName}`,
            },
        });
        const principal = new aws_iam_1.OpenIdConnectPrincipal(cluster.openIdConnectProvider).withConditions({
            StringEquals: conditions,
        });
        this.role = new aws_iam_1.Role(this, 'Role', { assumedBy: principal });
        this.assumeRoleAction = this.role.assumeRoleAction;
        this.grantPrincipal = this.role.grantPrincipal;
        this.policyFragment = this.role.policyFragment;
        // Note that we cannot use `cluster.addManifest` here because that would create the manifest
        // constrct in the scope of the cluster stack, which might be a different stack than this one.
        // This means that the cluster stack would depend on this stack because of the role,
        // and since this stack inherintely depends on the cluster stack, we will have a circular dependency.
        new k8s_manifest_1.KubernetesManifest(this, `manifest-${id}ServiceAccountResource`, {
            cluster,
            manifest: [{
                    apiVersion: 'v1',
                    kind: 'ServiceAccount',
                    metadata: {
                        name: this.serviceAccountName,
                        namespace: this.serviceAccountNamespace,
                        labels: {
                            'app.kubernetes.io/name': this.serviceAccountName,
                            ...props.labels,
                        },
                        annotations: {
                            'eks.amazonaws.com/role-arn': this.role.roleArn,
                            ...props.annotations,
                        },
                    },
                }],
        });
    }
    /**
     * @deprecated use `addToPrincipalPolicy()`
     */
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(statement) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPrincipalPolicy);
            }
            throw error;
        }
        return this.role.addToPrincipalPolicy(statement);
    }
    /**
     * If the value is a DNS subdomain name as defined in RFC 1123, from K8s docs.
     *
     * https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-subdomain-names
     */
    isValidDnsSubdomainName(value) {
        return value.length <= 253 && /^[a-z0-9]+[a-z0-9-.]*[a-z0-9]+$/.test(value);
    }
    /**
     * If the value follows DNS label standard as defined in RFC 1123, from K8s docs.
     *
     * https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-label-names
     */
    isValidDnsLabelName(value) {
        return value.length <= 63 && /^[a-z0-9]+[a-z0-9-]*[a-z0-9]+$/.test(value);
    }
}
_a = JSII_RTTI_SYMBOL_1;
ServiceAccount[_a] = { fqn: "aws-cdk-lib.aws_eks.ServiceAccount", version: "2.74.0" };
exports.ServiceAccount = ServiceAccount;
//# sourceMappingURL=data:application/json;base64,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