"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerPort = exports.InstanceTarget = exports.LoadBalancer = exports.LoadBalancingProtocol = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("../../aws-ec2");
const core_1 = require("../../core");
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener
 *
 * Routes to a fleet of of instances in a VPC.
 */
class LoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        this.instanceIds = [];
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_elasticloadbalancing_LoadBalancerProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LoadBalancer);
            }
            throw error;
        }
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.any({ produce: () => this.listeners }),
            instances: core_1.Lazy.list({ produce: () => this.instanceIds }, { omitEmpty: true }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: props.crossZone ?? true,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        if (props.accessLoggingPolicy !== undefined) {
            this.elb.accessLoggingPolicy = props.accessLoggingPolicy;
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer
     *
     * @returns A ListenerPort object that controls connections to the listener port
     */
    addListener(listener) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_elasticloadbalancing_LoadBalancerListener(listener);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addListener);
            }
            throw error;
        }
        if (listener.sslCertificateArn && listener.sslCertificateId) {
            throw new Error('"sslCertificateId" is deprecated, please use "sslCertificateArn" only.');
        }
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const sslCertificateArn = listener.sslCertificateArn || listener.sslCertificateId;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: sslCertificateArn,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    addTarget(target) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_elasticloadbalancing_ILoadBalancerTarget(target);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addTarget);
            }
            throw error;
        }
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @attribute
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @attribute
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
    /**
     * Add instance to the load balancer.
     * @internal
     */
    _addInstanceId(instanceId) {
        this.instanceIds.push(instanceId);
    }
}
_a = JSII_RTTI_SYMBOL_1;
LoadBalancer[_a] = { fqn: "aws-cdk-lib.aws_elasticloadbalancing.LoadBalancer", version: "2.74.0" };
exports.LoadBalancer = LoadBalancer;
/**
 * An EC2 instance that is the target for load balancing
 */
class InstanceTarget {
    /**
     * Create a new Instance target.
     *
     * @param instance Instance to register to.
     */
    constructor(instance) {
        this.instance = instance;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_Instance(instance);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, InstanceTarget);
            }
            throw error;
        }
        this.connections = instance.connections;
    }
    attachToClassicLB(loadBalancer) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_elasticloadbalancing_LoadBalancer(loadBalancer);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.attachToClassicLB);
            }
            throw error;
        }
        loadBalancer._addInstanceId(this.instance.instanceId);
    }
}
_b = JSII_RTTI_SYMBOL_1;
InstanceTarget[_b] = { fqn: "aws-cdk-lib.aws_elasticloadbalancing.InstanceTarget", version: "2.74.0" };
exports.InstanceTarget = InstanceTarget;
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *    const listener = elb.addListener(...);
 *
 *    listener.connections.allowDefaultPortFromAnyIPv4();
 *    // or
 *    instance.connections.allowToDefaultPort(listener);
 */
class ListenerPort {
    constructor(securityGroup, defaultPort) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_ISecurityGroup(securityGroup);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ec2_Port(defaultPort);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ListenerPort);
            }
            throw error;
        }
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
_c = JSII_RTTI_SYMBOL_1;
ListenerPort[_c] = { fqn: "aws-cdk-lib.aws_elasticloadbalancing.ListenerPort", version: "2.74.0" };
exports.ListenerPort = ListenerPort;
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibG9hZC1iYWxhbmNlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImxvYWQtYmFsYW5jZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEsMkNBR3VCO0FBQ3ZCLHFDQUFzRDtBQUV0RCxxRkFBbUU7QUEwTm5FLElBQVkscUJBS1g7QUFMRCxXQUFZLHFCQUFxQjtJQUMvQixvQ0FBVyxDQUFBO0lBQ1gsb0NBQVcsQ0FBQTtJQUNYLHNDQUFhLENBQUE7SUFDYix3Q0FBZSxDQUFBO0FBQ2pCLENBQUMsRUFMVyxxQkFBcUIsR0FBckIsNkJBQXFCLEtBQXJCLDZCQUFxQixRQUtoQztBQUVEOzs7O0dBSUc7QUFDSCxNQUFhLFlBQWEsU0FBUSxlQUFRO0lBbUJ4QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXdCO1FBQ2hFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFkbkI7O1dBRUc7UUFDYSxrQkFBYSxHQUFtQixFQUFFLENBQUM7UUFJbEMsY0FBUyxHQUF3QyxFQUFFLENBQUM7UUFFcEQsa0JBQWEsR0FBYSxFQUFFLENBQUM7UUFDN0IsWUFBTyxHQUEwQixFQUFFLENBQUM7UUFDcEMsZ0JBQVcsR0FBYSxFQUFFLENBQUM7Ozs7OzsrQ0FqQmpDLFlBQVk7Ozs7UUFzQnJCLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSx1QkFBYSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUUsRUFBRSxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQzNHLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxxQkFBVyxDQUFDLEVBQUUsY0FBYyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUM3RSwwRkFBMEY7UUFDMUYsTUFBTSxlQUFlLEdBQW9CLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXBFLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxnREFBZSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDL0MsY0FBYyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxlQUFlLENBQUM7WUFDcEQsT0FBTyxFQUFFLGVBQWUsQ0FBQyxTQUFTO1lBQ2xDLFNBQVMsRUFBRSxXQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsQ0FBQztZQUN0RCxTQUFTLEVBQUUsV0FBSSxDQUFDLElBQUksQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDOUUsTUFBTSxFQUFFLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxVQUFVO1lBQzdELFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVyxJQUFJLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUM7WUFDdEUsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTLElBQUksSUFBSTtTQUNuQyxDQUFDLENBQUM7UUFDSCxJQUFJLEtBQUssQ0FBQyxjQUFjLEVBQUU7WUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsSUFBSSxLQUFLLENBQUMsbUJBQW1CLEtBQUssU0FBUyxFQUFFO1lBQzNDLElBQUksQ0FBQyxHQUFHLENBQUMsbUJBQW1CLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixDQUFDO1NBQzFEO1FBRUQsV0FBVyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ25FLFdBQVcsQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztLQUNoRTtJQUVEOzs7O09BSUc7SUFDSSxXQUFXLENBQUMsUUFBOEI7Ozs7Ozs7Ozs7UUFDL0MsSUFBSSxRQUFRLENBQUMsaUJBQWlCLElBQUksUUFBUSxDQUFDLGdCQUFnQixFQUFFO1lBQzNELE1BQU0sSUFBSSxLQUFLLENBQUMsd0VBQXdFLENBQUMsQ0FBQztTQUMzRjtRQUNELE1BQU0sUUFBUSxHQUFHLGVBQWUsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLEVBQUUsR0FBRyxFQUFFLENBQUMsaUJBQWlCLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUM7UUFDNUcsTUFBTSxZQUFZLEdBQUcsUUFBUSxDQUFDLFlBQVksSUFBSSxRQUFRLENBQUMsWUFBWSxDQUFDO1FBQ3BFLE1BQU0saUJBQWlCLEdBQUcsUUFBUSxDQUFDLGlCQUFpQixJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQztRQUNsRixNQUFNLGdCQUFnQixHQUFHLFdBQVcsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLEVBQzVELFdBQVcsQ0FBQyxvQkFBb0IsQ0FBQyxZQUFZLENBQUMsRUFDNUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLHFCQUFxQixDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFFeEYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUM7WUFDbEIsZ0JBQWdCLEVBQUUsUUFBUSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUU7WUFDbEQsUUFBUTtZQUNSLFlBQVksRUFBRSxZQUFZLENBQUMsUUFBUSxFQUFFO1lBQ3JDLGdCQUFnQjtZQUNoQixnQkFBZ0IsRUFBRSxpQkFBaUI7WUFDbkMsV0FBVyxFQUFFLFFBQVEsQ0FBQyxXQUFXO1NBQ2xDLENBQUMsQ0FBQztRQUVILE1BQU0sSUFBSSxHQUFHLElBQUksWUFBWSxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsY0FBSSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztRQUVuRixrRkFBa0Y7UUFDbEYsV0FBVyxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsRUFBRSxDQUFDLGNBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQzFFLElBQUksQ0FBQyxXQUFXLENBQUMsb0JBQW9CLENBQUMsSUFBSSxFQUFFLHlCQUF5QixRQUFRLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQztRQUNoRyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxlQUFlLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFbkMsbUdBQW1HO1FBQ25HLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTlCLE9BQU8sSUFBSSxDQUFDO0tBQ2I7SUFFTSxTQUFTLENBQUMsTUFBMkI7Ozs7Ozs7Ozs7UUFDMUMsTUFBTSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBRS9CLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLENBQUM7S0FDeEI7SUFFRDs7T0FFRztJQUNILElBQVcsZ0JBQWdCO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUM7S0FDckI7SUFFRDs7T0FFRztJQUNILElBQVcscUNBQXFDO1FBQzlDLE9BQU8sSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQztLQUMvQztJQUVEOztPQUVHO0lBQ0gsSUFBVyxtQ0FBbUM7UUFDNUMsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDO0tBQzdDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLG1CQUFtQjtRQUM1QixPQUFPLElBQUksQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDO0tBQzdCO0lBRUQ7O09BRUc7SUFDSCxJQUFXLHdDQUF3QztRQUNqRCxPQUFPLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUM7S0FDbEQ7SUFFRDs7T0FFRztJQUNILElBQVcseUNBQXlDO1FBQ2xELE9BQU8sSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQ0FBaUMsQ0FBQztLQUNuRDtJQUVEOztPQUVHO0lBQ0ssZUFBZSxDQUFDLFlBQW9CO1FBQzFDLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLFlBQVksRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRXZFLHdDQUF3QztRQUN4QyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztLQUN2QztJQUVEOztPQUVHO0lBQ0ssU0FBUyxDQUFDLE1BQTJCO1FBQzNDLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBRXZFLDZDQUE2QztRQUM3QyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztLQUMzQjtJQUVEOztPQUVHO0lBQ0sscUJBQXFCLENBQUMsWUFBb0IsRUFBRSxNQUEyQjtRQUM3RSxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FDdEIsTUFBTSxFQUNOLGNBQUksQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEVBQ3RCLFFBQVEsWUFBWSxjQUFjLENBQUMsQ0FBQztLQUN2QztJQUVEOzs7T0FHRztJQUNJLGNBQWMsQ0FBQyxVQUFrQjtRQUN0QyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztLQUNuQzs7OztBQTVLVSxvQ0FBWTtBQStLekI7O0dBRUc7QUFDSCxNQUFhLGNBQWM7SUFFekI7Ozs7T0FJRztJQUNILFlBQTRCLFFBQWtCO1FBQWxCLGFBQVEsR0FBUixRQUFRLENBQVU7Ozs7OzsrQ0FQbkMsY0FBYzs7OztRQVF2QixJQUFJLENBQUMsV0FBVyxHQUFHLFFBQVEsQ0FBQyxXQUFXLENBQUM7S0FDekM7SUFFTSxpQkFBaUIsQ0FBQyxZQUEwQjs7Ozs7Ozs7OztRQUNqRCxZQUFZLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUM7S0FDdkQ7Ozs7QUFiVSx3Q0FBYztBQWdCM0I7Ozs7Ozs7Ozs7OztHQVlHO0FBQ0gsTUFBYSxZQUFZO0lBR3ZCLFlBQVksYUFBNkIsRUFBRSxXQUFpQjs7Ozs7OzsrQ0FIakQsWUFBWTs7OztRQUlyQixJQUFJLENBQUMsV0FBVyxHQUFHLElBQUkscUJBQVcsQ0FBQyxFQUFFLGNBQWMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUM7S0FDdEY7Ozs7QUFMVSxvQ0FBWTtBQVF6QixTQUFTLGlCQUFpQixDQUFDLElBQVk7SUFDckMsTUFBTSxLQUFLLEdBQUcsb0JBQW9CLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDekMsSUFBSSxDQUFDLEtBQUssRUFBRTtRQUNWLE1BQU0sSUFBSSxLQUFLLENBQUMsMENBQTBDLElBQUksRUFBRSxDQUFDLENBQUM7S0FDbkU7SUFDRCxPQUFPLEtBQUssQ0FBQztBQUNmLENBQUM7QUFFRCxTQUFTLG9CQUFvQixDQUFDLElBQVk7SUFDeEMsSUFBSSxJQUFJLEtBQUssRUFBRSxFQUFFO1FBQUUsT0FBTyxxQkFBcUIsQ0FBQyxJQUFJLENBQUM7S0FBRTtJQUN2RCxJQUFJLElBQUksS0FBSyxHQUFHLEVBQUU7UUFBRSxPQUFPLHFCQUFxQixDQUFDLEtBQUssQ0FBQztLQUFFO0lBQ3pELE9BQU8sU0FBUyxDQUFDO0FBQ25CLENBQUM7QUFFRCxTQUFTLGNBQWMsQ0FBQyxLQUE0QjtJQUNsRCxPQUFPLEtBQUssS0FBSyxxQkFBcUIsQ0FBQyxLQUFLLElBQUksS0FBSyxLQUFLLHFCQUFxQixDQUFDLElBQUksQ0FBQztBQUN2RixDQUFDO0FBRUQsU0FBUyxXQUFXLENBQUksQ0FBZ0IsRUFBRSxHQUFNO0lBQzlDLE9BQU8sQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUM7QUFDN0IsQ0FBQztBQUVELFNBQVMsZUFBZSxDQUFJLENBQWdCLEVBQUUsR0FBWTtJQUN4RCxPQUFPLENBQUMsSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxFQUFFLENBQUM7QUFDL0IsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxpQkFBaUIsQ0FBQyxXQUF3QjtJQUNqRCxNQUFNLFFBQVEsR0FBRyxXQUFXLENBQUMsV0FBVyxDQUFDLFFBQVEsRUFDL0MsV0FBVyxDQUFDLG9CQUFvQixDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsRUFDaEQscUJBQXFCLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUVoQyxNQUFNLElBQUksR0FBRyxRQUFRLEtBQUsscUJBQXFCLENBQUMsSUFBSSxJQUFJLFFBQVEsS0FBSyxxQkFBcUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFFM0ksTUFBTSxNQUFNLEdBQUcsR0FBRyxRQUFRLENBQUMsV0FBVyxFQUFFLElBQUksV0FBVyxDQUFDLElBQUksR0FBRyxJQUFJLEVBQUUsQ0FBQztJQUV0RSxPQUFPO1FBQ0wsZ0JBQWdCLEVBQUUsV0FBVyxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUU7UUFDekUsUUFBUSxFQUFFLENBQUMsV0FBVyxDQUFDLFFBQVEsSUFBSSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxFQUFFLENBQUMsUUFBUSxFQUFFO1FBQy9FLE1BQU07UUFDTixPQUFPLEVBQUUsQ0FBQyxXQUFXLENBQUMsT0FBTyxJQUFJLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLEVBQUUsQ0FBQyxRQUFRLEVBQUU7UUFDNUUsa0JBQWtCLEVBQUUsV0FBVyxDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUU7S0FDOUUsQ0FBQztBQUNKLENBQUM7QUFFRCxTQUFTLG1CQUFtQixDQUFDLEtBQXdCO0lBQ25ELElBQUksS0FBSyxDQUFDLGVBQWUsS0FBSyxTQUFTLEVBQUU7UUFDdkMsT0FBTyxLQUFLLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUM7S0FDdkQ7U0FBTSxJQUFJLEtBQUssQ0FBQyxjQUFjLEVBQUU7UUFDL0IsT0FBTyxLQUFLLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQztZQUM3QixVQUFVLEVBQUUsb0JBQVUsQ0FBQyxNQUFNO1NBQzlCLENBQUMsQ0FBQztLQUNKO1NBQU07UUFDTCxPQUFPLEtBQUssQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDO1lBQzdCLFVBQVUsRUFBRSxvQkFBVSxDQUFDLG1CQUFtQjtTQUMzQyxDQUFDLENBQUM7S0FDSjtBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge1xuICBDb25uZWN0aW9ucywgSUNvbm5lY3RhYmxlLCBJbnN0YW5jZSwgSVNlY3VyaXR5R3JvdXAsIElWcGMsIFBlZXIsIFBvcnQsXG4gIFNlY3VyaXR5R3JvdXAsIFNlbGVjdGVkU3VibmV0cywgU3VibmV0U2VsZWN0aW9uLCBTdWJuZXRUeXBlLFxufSBmcm9tICcuLi8uLi9hd3MtZWMyJztcbmltcG9ydCB7IER1cmF0aW9uLCBMYXp5LCBSZXNvdXJjZSB9IGZyb20gJy4uLy4uL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5Mb2FkQmFsYW5jZXIgfSBmcm9tICcuL2VsYXN0aWNsb2FkYmFsYW5jaW5nLmdlbmVyYXRlZCc7XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgTG9hZEJhbGFuY2VyXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTG9hZEJhbGFuY2VyUHJvcHMge1xuICAvKipcbiAgICogVlBDIG5ldHdvcmsgb2YgdGhlIGZsZWV0IGluc3RhbmNlc1xuICAgKi9cbiAgcmVhZG9ubHkgdnBjOiBJVnBjO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgaXMgYW4gaW50ZXJuZXQtZmFjaW5nIExvYWQgQmFsYW5jZXJcbiAgICpcbiAgICogVGhpcyBjb250cm9scyB3aGV0aGVyIHRoZSBMQiBoYXMgYSBwdWJsaWMgSVAgYWRkcmVzcyBhc3NpZ25lZC4gSXQgZG9lc1xuICAgKiBub3Qgb3BlbiB1cCB0aGUgTG9hZCBCYWxhbmNlcidzIHNlY3VyaXR5IGdyb3VwcyB0byBwdWJsaWMgaW50ZXJuZXQgYWNjZXNzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgaW50ZXJuZXRGYWNpbmc/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGF0IGxpc3RlbmVycyB0byBzZXQgdXAgZm9yIHRoZSBsb2FkIGJhbGFuY2VyLlxuICAgKlxuICAgKiBDYW4gYWxzbyBiZSBhZGRlZCBieSAuYWRkTGlzdGVuZXIoKVxuICAgKlxuICAgKiBAZGVmYXVsdCAtXG4gICAqL1xuICByZWFkb25seSBsaXN0ZW5lcnM/OiBMb2FkQmFsYW5jZXJMaXN0ZW5lcltdO1xuXG4gIC8qKlxuICAgKiBXaGF0IHRhcmdldHMgdG8gbG9hZCBiYWxhbmNlIHRvLlxuICAgKlxuICAgKiBDYW4gYWxzbyBiZSBhZGRlZCBieSAuYWRkVGFyZ2V0KClcbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lLlxuICAgKi9cbiAgcmVhZG9ubHkgdGFyZ2V0cz86IElMb2FkQmFsYW5jZXJUYXJnZXRbXTtcblxuICAvKipcbiAgICogSGVhbHRoIGNoZWNrIHNldHRpbmdzIGZvciB0aGUgbG9hZCBiYWxhbmNpbmcgdGFyZ2V0cy5cbiAgICpcbiAgICogTm90IHJlcXVpcmVkIGJ1dCByZWNvbW1lbmRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lLlxuICAgKi9cbiAgcmVhZG9ubHkgaGVhbHRoQ2hlY2s/OiBIZWFsdGhDaGVjaztcblxuICAvKipcbiAgICogV2hldGhlciBjcm9zcyB6b25lIGxvYWQgYmFsYW5jaW5nIGlzIGVuYWJsZWRcbiAgICpcbiAgICogVGhpcyBjb250cm9scyB3aGV0aGVyIHRoZSBsb2FkIGJhbGFuY2VyIGV2ZW5seSBkaXN0cmlidXRlcyByZXF1ZXN0c1xuICAgKiBhY3Jvc3MgZWFjaCBhdmFpbGFiaWxpdHkgem9uZVxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBjcm9zc1pvbmU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGljaCBzdWJuZXRzIHRvIGRlcGxveSB0aGUgbG9hZCBiYWxhbmNlclxuICAgKlxuICAgKiBDYW4gYmUgdXNlZCB0byBkZWZpbmUgYSBzcGVjaWZpYyBzZXQgb2Ygc3VibmV0cyB0byBkZXBsb3kgdGhlIGxvYWQgYmFsYW5jZXIgdG8uXG4gICAqIFVzZWZ1bCBtdWx0aXBsZSBwdWJsaWMgb3IgcHJpdmF0ZSBzdWJuZXRzIGFyZSBjb3ZlcmluZyB0aGUgc2FtZSBhdmFpbGFiaWxpdHkgem9uZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBQdWJsaWMgc3VibmV0cyBpZiBpbnRlcm5ldEZhY2luZywgUHJpdmF0ZSBzdWJuZXRzIG90aGVyd2lzZVxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0U2VsZWN0aW9uPzogU3VibmV0U2VsZWN0aW9uO1xuXG4gIC8qKlxuICAgKiBFbmFibGUgTG9hZGJhbGFuY2VyIGFjY2VzcyBsb2dzXG4gICAqIENhbiBiZSB1c2VkIHRvIGF2b2lkIG1hbnVhbCB3b3JrIGFzIGF3cyBjb25zb2xlXG4gICAqIFJlcXVpcmVkIFMzIGJ1Y2tldCBuYW1lICwgZW5hYmxlZCBmbGFnXG4gICAqIENhbiBhZGQgaW50ZXJ2YWwgZm9yIHB1c2hpbmcgbG9nXG4gICAqIENhbiBzZXQgYnVja2V0IHByZWZpeCBpbiBvcmRlciB0byBwcm92aWRlIGZvbGRlciBuYW1lIGluc2lkZSBidWNrZXRcbiAgICogQGRlZmF1bHQgLSBkaXNhYmxlZFxuICAgKi9cbiAgcmVhZG9ubHkgYWNjZXNzTG9nZ2luZ1BvbGljeT86IENmbkxvYWRCYWxhbmNlci5BY2Nlc3NMb2dnaW5nUG9saWN5UHJvcGVydHk7XG5cbn1cblxuLyoqXG4gKiBEZXNjcmliZSB0aGUgaGVhbHRoIGNoZWNrIHRvIGEgbG9hZCBiYWxhbmNlclxuICovXG5leHBvcnQgaW50ZXJmYWNlIEhlYWx0aENoZWNrIHtcbiAgLyoqXG4gICAqIFdoYXQgcG9ydCBudW1iZXIgdG8gaGVhbHRoIGNoZWNrIG9uXG4gICAqL1xuICByZWFkb25seSBwb3J0OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFdoYXQgcHJvdG9jb2wgdG8gdXNlIGZvciBoZWFsdGggY2hlY2tpbmdcbiAgICpcbiAgICogVGhlIHByb3RvY29sIGlzIGF1dG9tYXRpY2FsbHkgZGV0ZXJtaW5lZCBmcm9tIHRoZSBwb3J0IGlmIGl0J3Mgbm90IHN1cHBsaWVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBBdXRvbWF0aWNcbiAgICovXG4gIHJlYWRvbmx5IHByb3RvY29sPzogTG9hZEJhbGFuY2luZ1Byb3RvY29sO1xuXG4gIC8qKlxuICAgKiBXaGF0IHBhdGggdG8gdXNlIGZvciBIVFRQIG9yIEhUVFBTIGhlYWx0aCBjaGVjayAobXVzdCByZXR1cm4gMjAwKVxuICAgKlxuICAgKiBGb3IgU1NMIGFuZCBUQ1AgaGVhbHRoIGNoZWNrcywgYWNjZXB0aW5nIGNvbm5lY3Rpb25zIGlzIGVub3VnaCB0byBiZSBjb25zaWRlcmVkXG4gICAqIGhlYWx0aHkuXG4gICAqXG4gICAqIEBkZWZhdWx0IFwiL1wiXG4gICAqL1xuICByZWFkb25seSBwYXRoPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBZnRlciBob3cgbWFueSBzdWNjZXNzZnVsIGNoZWNrcyBpcyBhbiBpbnN0YW5jZSBjb25zaWRlcmVkIGhlYWx0aHlcbiAgICpcbiAgICogQGRlZmF1bHQgMlxuICAgKi9cbiAgcmVhZG9ubHkgaGVhbHRoeVRocmVzaG9sZD86IG51bWJlcjtcblxuICAvKipcbiAgICogQWZ0ZXIgaG93IG1hbnkgdW5zdWNjZXNzZnVsIGNoZWNrcyBpcyBhbiBpbnN0YW5jZSBjb25zaWRlcmVkIHVuaGVhbHRoeVxuICAgKlxuICAgKiBAZGVmYXVsdCA1XG4gICAqL1xuICByZWFkb25seSB1bmhlYWx0aHlUaHJlc2hvbGQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIE51bWJlciBvZiBzZWNvbmRzIGJldHdlZW4gaGVhbHRoIGNoZWNrc1xuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5zZWNvbmRzKDMwKVxuICAgKi9cbiAgcmVhZG9ubHkgaW50ZXJ2YWw/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogSGVhbHRoIGNoZWNrIHRpbWVvdXRcbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24uc2Vjb25kcyg1KVxuICAgKi9cbiAgcmVhZG9ubHkgdGltZW91dD86IER1cmF0aW9uO1xufVxuXG4vKipcbiAqIEludGVyZmFjZSB0aGF0IGlzIGdvaW5nIHRvIGJlIGltcGxlbWVudGVkIGJ5IGNvbnN0cnVjdHMgdGhhdCB5b3UgY2FuIGxvYWQgYmFsYW5jZSB0b1xuICovXG5leHBvcnQgaW50ZXJmYWNlIElMb2FkQmFsYW5jZXJUYXJnZXQgZXh0ZW5kcyBJQ29ubmVjdGFibGUge1xuICAvKipcbiAgICogQXR0YWNoIGxvYWQtYmFsYW5jZWQgdGFyZ2V0IHRvIGEgY2xhc3NpYyBFTEJcbiAgICogQHBhcmFtIGxvYWRCYWxhbmNlciBbZGlzYWJsZS1hd3NsaW50OnJlZi12aWEtaW50ZXJmYWNlXSBUaGUgbG9hZCBiYWxhbmNlciB0byBhdHRhY2ggdGhlIHRhcmdldCB0b1xuICAgKi9cbiAgYXR0YWNoVG9DbGFzc2ljTEIobG9hZEJhbGFuY2VyOiBMb2FkQmFsYW5jZXIpOiB2b2lkO1xufVxuXG4vKipcbiAqIEFkZCBhIGJhY2tlbmQgdG8gdGhlIGxvYWQgYmFsYW5jZXJcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMb2FkQmFsYW5jZXJMaXN0ZW5lciB7XG4gIC8qKlxuICAgKiBFeHRlcm5hbCBsaXN0ZW5pbmcgcG9ydFxuICAgKi9cbiAgcmVhZG9ubHkgZXh0ZXJuYWxQb3J0OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFdoYXQgcHVibGljIHByb3RvY29sIHRvIHVzZSBmb3IgbG9hZCBiYWxhbmNpbmdcbiAgICpcbiAgICogRWl0aGVyICd0Y3AnLCAnc3NsJywgJ2h0dHAnIG9yICdodHRwcycuXG4gICAqXG4gICAqIE1heSBiZSBvbWl0dGVkIGlmIHRoZSBleHRlcm5hbCBwb3J0IGlzIGVpdGhlciA4MCBvciA0NDMuXG4gICAqL1xuICByZWFkb25seSBleHRlcm5hbFByb3RvY29sPzogTG9hZEJhbGFuY2luZ1Byb3RvY29sO1xuXG4gIC8qKlxuICAgKiBJbnN0YW5jZSBsaXN0ZW5pbmcgcG9ydFxuICAgKlxuICAgKiBTYW1lIGFzIHRoZSBleHRlcm5hbFBvcnQgaWYgbm90IHNwZWNpZmllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgZXh0ZXJuYWxQb3J0XG4gICAqL1xuICByZWFkb25seSBpbnRlcm5hbFBvcnQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFdoYXQgcHVibGljIHByb3RvY29sIHRvIHVzZSBmb3IgbG9hZCBiYWxhbmNpbmdcbiAgICpcbiAgICogRWl0aGVyICd0Y3AnLCAnc3NsJywgJ2h0dHAnIG9yICdodHRwcycuXG4gICAqXG4gICAqIE1heSBiZSBvbWl0dGVkIGlmIHRoZSBpbnRlcm5hbCBwb3J0IGlzIGVpdGhlciA4MCBvciA0NDMuXG4gICAqXG4gICAqIFRoZSBpbnN0YW5jZSBwcm90b2NvbCBpcyAndGNwJyBpZiB0aGUgZnJvbnQtZW5kIHByb3RvY29sXG4gICAqIGlzICd0Y3AnIG9yICdzc2wnLCB0aGUgaW5zdGFuY2UgcHJvdG9jb2wgaXMgJ2h0dHAnIGlmIHRoZVxuICAgKiBmcm9udC1lbmQgcHJvdG9jb2wgaXMgJ2h0dHBzJy5cbiAgICovXG4gIHJlYWRvbmx5IGludGVybmFsUHJvdG9jb2w/OiBMb2FkQmFsYW5jaW5nUHJvdG9jb2w7XG5cbiAgLyoqXG4gICAqIFNTTCBwb2xpY3kgbmFtZXNcbiAgICovXG4gIHJlYWRvbmx5IHBvbGljeU5hbWVzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIHRoZSBBUk4gb2YgdGhlIFNTTCBjZXJ0aWZpY2F0ZVxuICAgKiBAZGVwcmVjYXRlZCAtIHVzZSBzc2xDZXJ0aWZpY2F0ZUFybiBpbnN0ZWFkXG4gICAqL1xuICByZWFkb25seSBzc2xDZXJ0aWZpY2F0ZUlkPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiB0aGUgQVJOIG9mIHRoZSBTU0wgY2VydGlmaWNhdGVcbiAgICpcbiAgICogQGRlZmF1bHQgLSBub25lXG4gICAqL1xuICByZWFkb25seSBzc2xDZXJ0aWZpY2F0ZUFybj86IHN0cmluZztcblxuICAvKipcbiAgICogQWxsb3cgY29ubmVjdGlvbnMgdG8gdGhlIGxvYWQgYmFsYW5jZXIgZnJvbSB0aGUgZ2l2ZW4gc2V0IG9mIGNvbm5lY3Rpb24gcGVlcnNcbiAgICpcbiAgICogQnkgZGVmYXVsdCwgY29ubmVjdGlvbnMgd2lsbCBiZSBhbGxvd2VkIGZyb20gYW55d2hlcmUuIFNldCB0aGlzIHRvIGFuIGVtcHR5IGxpc3RcbiAgICogdG8gZGVueSBjb25uZWN0aW9ucywgb3Igc3VwcGx5IGEgY3VzdG9tIGxpc3Qgb2YgcGVlcnMgdG8gYWxsb3cgY29ubmVjdGlvbnMgZnJvbVxuICAgKiAoSVAgcmFuZ2VzIG9yIHNlY3VyaXR5IGdyb3VwcykuXG4gICAqXG4gICAqIEBkZWZhdWx0IEFueXdoZXJlXG4gICAqL1xuICByZWFkb25seSBhbGxvd0Nvbm5lY3Rpb25zRnJvbT86IElDb25uZWN0YWJsZVtdO1xufVxuXG5leHBvcnQgZW51bSBMb2FkQmFsYW5jaW5nUHJvdG9jb2wge1xuICBUQ1AgPSAndGNwJyxcbiAgU1NMID0gJ3NzbCcsXG4gIEhUVFAgPSAnaHR0cCcsXG4gIEhUVFBTID0gJ2h0dHBzJ1xufVxuXG4vKipcbiAqIEEgbG9hZCBiYWxhbmNlciB3aXRoIGEgc2luZ2xlIGxpc3RlbmVyXG4gKlxuICogUm91dGVzIHRvIGEgZmxlZXQgb2Ygb2YgaW5zdGFuY2VzIGluIGEgVlBDLlxuICovXG5leHBvcnQgY2xhc3MgTG9hZEJhbGFuY2VyIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJQ29ubmVjdGFibGUge1xuICAvKipcbiAgICogQ29udHJvbCBhbGwgY29ubmVjdGlvbnMgZnJvbSBhbmQgdG8gdGhpcyBsb2FkIGJhbGFuY2VyXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IENvbm5lY3Rpb25zO1xuXG4gIC8qKlxuICAgKiBBbiBvYmplY3QgY29udHJvbGxpbmcgc3BlY2lmaWNhbGx5IHRoZSBjb25uZWN0aW9ucyBmb3IgZWFjaCBsaXN0ZW5lciBhZGRlZCB0byB0aGlzIGxvYWQgYmFsYW5jZXJcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBsaXN0ZW5lclBvcnRzOiBMaXN0ZW5lclBvcnRbXSA9IFtdO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgZWxiOiBDZm5Mb2FkQmFsYW5jZXI7XG4gIHByaXZhdGUgcmVhZG9ubHkgc2VjdXJpdHlHcm91cDogU2VjdXJpdHlHcm91cDtcbiAgcHJpdmF0ZSByZWFkb25seSBsaXN0ZW5lcnM6IENmbkxvYWRCYWxhbmNlci5MaXN0ZW5lcnNQcm9wZXJ0eVtdID0gW107XG5cbiAgcHJpdmF0ZSByZWFkb25seSBpbnN0YW5jZVBvcnRzOiBudW1iZXJbXSA9IFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IHRhcmdldHM6IElMb2FkQmFsYW5jZXJUYXJnZXRbXSA9IFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IGluc3RhbmNlSWRzOiBzdHJpbmdbXSA9IFtdO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBMb2FkQmFsYW5jZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnNlY3VyaXR5R3JvdXAgPSBuZXcgU2VjdXJpdHlHcm91cCh0aGlzLCAnU2VjdXJpdHlHcm91cCcsIHsgdnBjOiBwcm9wcy52cGMsIGFsbG93QWxsT3V0Ym91bmQ6IGZhbHNlIH0pO1xuICAgIHRoaXMuY29ubmVjdGlvbnMgPSBuZXcgQ29ubmVjdGlvbnMoeyBzZWN1cml0eUdyb3VwczogW3RoaXMuc2VjdXJpdHlHcm91cF0gfSk7XG4gICAgLy8gRGVwZW5kaW5nIG9uIHdoZXRoZXIgdGhlIEVMQiBoYXMgcHVibGljIG9yIGludGVybmFsIElQcywgcGljayB0aGUgcmlnaHQgYmFja2VuZCBzdWJuZXRzXG4gICAgY29uc3Qgc2VsZWN0ZWRTdWJuZXRzOiBTZWxlY3RlZFN1Ym5ldHMgPSBsb2FkQmFsYW5jZXJTdWJuZXRzKHByb3BzKTtcblxuICAgIHRoaXMuZWxiID0gbmV3IENmbkxvYWRCYWxhbmNlcih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBzZWN1cml0eUdyb3VwczogW3RoaXMuc2VjdXJpdHlHcm91cC5zZWN1cml0eUdyb3VwSWRdLFxuICAgICAgc3VibmV0czogc2VsZWN0ZWRTdWJuZXRzLnN1Ym5ldElkcyxcbiAgICAgIGxpc3RlbmVyczogTGF6eS5hbnkoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLmxpc3RlbmVycyB9KSxcbiAgICAgIGluc3RhbmNlczogTGF6eS5saXN0KHsgcHJvZHVjZTogKCkgPT4gdGhpcy5pbnN0YW5jZUlkcyB9LCB7IG9taXRFbXB0eTogdHJ1ZSB9KSxcbiAgICAgIHNjaGVtZTogcHJvcHMuaW50ZXJuZXRGYWNpbmcgPyAnaW50ZXJuZXQtZmFjaW5nJyA6ICdpbnRlcm5hbCcsXG4gICAgICBoZWFsdGhDaGVjazogcHJvcHMuaGVhbHRoQ2hlY2sgJiYgaGVhbHRoQ2hlY2tUb0pTT04ocHJvcHMuaGVhbHRoQ2hlY2spLFxuICAgICAgY3Jvc3Nab25lOiBwcm9wcy5jcm9zc1pvbmUgPz8gdHJ1ZSxcbiAgICB9KTtcbiAgICBpZiAocHJvcHMuaW50ZXJuZXRGYWNpbmcpIHtcbiAgICAgIHRoaXMuZWxiLm5vZGUuYWRkRGVwZW5kZW5jeShzZWxlY3RlZFN1Ym5ldHMuaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZCk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmFjY2Vzc0xvZ2dpbmdQb2xpY3kgIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhpcy5lbGIuYWNjZXNzTG9nZ2luZ1BvbGljeSA9IHByb3BzLmFjY2Vzc0xvZ2dpbmdQb2xpY3k7XG4gICAgfVxuXG4gICAgaWZVbmRlZmluZWQocHJvcHMubGlzdGVuZXJzLCBbXSkuZm9yRWFjaChiID0+IHRoaXMuYWRkTGlzdGVuZXIoYikpO1xuICAgIGlmVW5kZWZpbmVkKHByb3BzLnRhcmdldHMsIFtdKS5mb3JFYWNoKHQgPT4gdGhpcy5hZGRUYXJnZXQodCkpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIGJhY2tlbmQgdG8gdGhlIGxvYWQgYmFsYW5jZXJcbiAgICpcbiAgICogQHJldHVybnMgQSBMaXN0ZW5lclBvcnQgb2JqZWN0IHRoYXQgY29udHJvbHMgY29ubmVjdGlvbnMgdG8gdGhlIGxpc3RlbmVyIHBvcnRcbiAgICovXG4gIHB1YmxpYyBhZGRMaXN0ZW5lcihsaXN0ZW5lcjogTG9hZEJhbGFuY2VyTGlzdGVuZXIpOiBMaXN0ZW5lclBvcnQge1xuICAgIGlmIChsaXN0ZW5lci5zc2xDZXJ0aWZpY2F0ZUFybiAmJiBsaXN0ZW5lci5zc2xDZXJ0aWZpY2F0ZUlkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1wic3NsQ2VydGlmaWNhdGVJZFwiIGlzIGRlcHJlY2F0ZWQsIHBsZWFzZSB1c2UgXCJzc2xDZXJ0aWZpY2F0ZUFyblwiIG9ubHkuJyk7XG4gICAgfVxuICAgIGNvbnN0IHByb3RvY29sID0gaWZVbmRlZmluZWRMYXp5KGxpc3RlbmVyLmV4dGVybmFsUHJvdG9jb2wsICgpID0+IHdlbGxLbm93blByb3RvY29sKGxpc3RlbmVyLmV4dGVybmFsUG9ydCkpO1xuICAgIGNvbnN0IGluc3RhbmNlUG9ydCA9IGxpc3RlbmVyLmludGVybmFsUG9ydCB8fCBsaXN0ZW5lci5leHRlcm5hbFBvcnQ7XG4gICAgY29uc3Qgc3NsQ2VydGlmaWNhdGVBcm4gPSBsaXN0ZW5lci5zc2xDZXJ0aWZpY2F0ZUFybiB8fCBsaXN0ZW5lci5zc2xDZXJ0aWZpY2F0ZUlkO1xuICAgIGNvbnN0IGluc3RhbmNlUHJvdG9jb2wgPSBpZlVuZGVmaW5lZChsaXN0ZW5lci5pbnRlcm5hbFByb3RvY29sLFxuICAgICAgaWZVbmRlZmluZWQodHJ5V2VsbEtub3duUHJvdG9jb2woaW5zdGFuY2VQb3J0KSxcbiAgICAgICAgaXNIdHRwUHJvdG9jb2wocHJvdG9jb2wpID8gTG9hZEJhbGFuY2luZ1Byb3RvY29sLkhUVFAgOiBMb2FkQmFsYW5jaW5nUHJvdG9jb2wuVENQKSk7XG5cbiAgICB0aGlzLmxpc3RlbmVycy5wdXNoKHtcbiAgICAgIGxvYWRCYWxhbmNlclBvcnQ6IGxpc3RlbmVyLmV4dGVybmFsUG9ydC50b1N0cmluZygpLFxuICAgICAgcHJvdG9jb2wsXG4gICAgICBpbnN0YW5jZVBvcnQ6IGluc3RhbmNlUG9ydC50b1N0cmluZygpLFxuICAgICAgaW5zdGFuY2VQcm90b2NvbCxcbiAgICAgIHNzbENlcnRpZmljYXRlSWQ6IHNzbENlcnRpZmljYXRlQXJuLFxuICAgICAgcG9saWN5TmFtZXM6IGxpc3RlbmVyLnBvbGljeU5hbWVzLFxuICAgIH0pO1xuXG4gICAgY29uc3QgcG9ydCA9IG5ldyBMaXN0ZW5lclBvcnQodGhpcy5zZWN1cml0eUdyb3VwLCBQb3J0LnRjcChsaXN0ZW5lci5leHRlcm5hbFBvcnQpKTtcblxuICAgIC8vIEFsbG93IGNvbm5lY3Rpb25zIG9uIHRoZSBwdWJsaWMgcG9ydCBmb3IgYWxsIHN1cHBsaWVkIHBlZXJzIChkZWZhdWx0OiBldmVyeW9uZSlcbiAgICBpZlVuZGVmaW5lZChsaXN0ZW5lci5hbGxvd0Nvbm5lY3Rpb25zRnJvbSwgW1BlZXIuYW55SXB2NCgpXSkuZm9yRWFjaChwZWVyID0+IHtcbiAgICAgIHBvcnQuY29ubmVjdGlvbnMuYWxsb3dEZWZhdWx0UG9ydEZyb20ocGVlciwgYERlZmF1bHQgcnVsZSBhbGxvdyBvbiAke2xpc3RlbmVyLmV4dGVybmFsUG9ydH1gKTtcbiAgICB9KTtcblxuICAgIHRoaXMubmV3SW5zdGFuY2VQb3J0KGluc3RhbmNlUG9ydCk7XG5cbiAgICAvLyBLZWVwIHRyYWNrIHVzaW5nIGFycmF5IHNvIHVzZXIgY2FuIGdldCB0byB0aGVtIGV2ZW4gaWYgdGhleSB3ZXJlIGFsbCBzdXBwbGllZCBpbiB0aGUgY29uc3RydWN0b3JcbiAgICB0aGlzLmxpc3RlbmVyUG9ydHMucHVzaChwb3J0KTtcblxuICAgIHJldHVybiBwb3J0O1xuICB9XG5cbiAgcHVibGljIGFkZFRhcmdldCh0YXJnZXQ6IElMb2FkQmFsYW5jZXJUYXJnZXQpIHtcbiAgICB0YXJnZXQuYXR0YWNoVG9DbGFzc2ljTEIodGhpcyk7XG5cbiAgICB0aGlzLm5ld1RhcmdldCh0YXJnZXQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyBnZXQgbG9hZEJhbGFuY2VyTmFtZSgpIHtcbiAgICByZXR1cm4gdGhpcy5lbGIucmVmO1xuICB9XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyBnZXQgbG9hZEJhbGFuY2VyQ2Fub25pY2FsSG9zdGVkWm9uZU5hbWVJZCgpIHtcbiAgICByZXR1cm4gdGhpcy5lbGIuYXR0ckNhbm9uaWNhbEhvc3RlZFpvbmVOYW1lSWQ7XG4gIH1cblxuICAvKipcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIGdldCBsb2FkQmFsYW5jZXJDYW5vbmljYWxIb3N0ZWRab25lTmFtZSgpIHtcbiAgICByZXR1cm4gdGhpcy5lbGIuYXR0ckNhbm9uaWNhbEhvc3RlZFpvbmVOYW1lO1xuICB9XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyBnZXQgbG9hZEJhbGFuY2VyRG5zTmFtZSgpIHtcbiAgICByZXR1cm4gdGhpcy5lbGIuYXR0ckRuc05hbWU7XG4gIH1cblxuICAvKipcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIGdldCBsb2FkQmFsYW5jZXJTb3VyY2VTZWN1cml0eUdyb3VwR3JvdXBOYW1lKCkge1xuICAgIHJldHVybiB0aGlzLmVsYi5hdHRyU291cmNlU2VjdXJpdHlHcm91cEdyb3VwTmFtZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgZ2V0IGxvYWRCYWxhbmNlclNvdXJjZVNlY3VyaXR5R3JvdXBPd25lckFsaWFzKCkge1xuICAgIHJldHVybiB0aGlzLmVsYi5hdHRyU291cmNlU2VjdXJpdHlHcm91cE93bmVyQWxpYXM7XG4gIH1cblxuICAvKipcbiAgICogQWxsb3cgY29ubmVjdGlvbnMgdG8gYWxsIGV4aXN0aW5nIHRhcmdldHMgb24gbmV3IGluc3RhbmNlIHBvcnRcbiAgICovXG4gIHByaXZhdGUgbmV3SW5zdGFuY2VQb3J0KGluc3RhbmNlUG9ydDogbnVtYmVyKSB7XG4gICAgdGhpcy50YXJnZXRzLmZvckVhY2godCA9PiB0aGlzLmFsbG93VGFyZ2V0Q29ubmVjdGlvbihpbnN0YW5jZVBvcnQsIHQpKTtcblxuICAgIC8vIEtlZXAgdHJhY2sgb2YgcG9ydCBmb3IgZnV0dXJlIHRhcmdldHNcbiAgICB0aGlzLmluc3RhbmNlUG9ydHMucHVzaChpbnN0YW5jZVBvcnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFsbG93IGNvbm5lY3Rpb25zIHRvIHRhcmdldCBvbiBhbGwgZXhpc3RpbmcgaW5zdGFuY2UgcG9ydHNcbiAgICovXG4gIHByaXZhdGUgbmV3VGFyZ2V0KHRhcmdldDogSUxvYWRCYWxhbmNlclRhcmdldCkge1xuICAgIHRoaXMuaW5zdGFuY2VQb3J0cy5mb3JFYWNoKHAgPT4gdGhpcy5hbGxvd1RhcmdldENvbm5lY3Rpb24ocCwgdGFyZ2V0KSk7XG5cbiAgICAvLyBLZWVwIHRyYWNrIG9mIHRhcmdldCBmb3IgZnV0dXJlIGxpc3RlbmVycy5cbiAgICB0aGlzLnRhcmdldHMucHVzaCh0YXJnZXQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFsbG93IGNvbm5lY3Rpb25zIGZvciBhIHNpbmdsZSAocG9ydCwgdGFyZ2V0KSBwYWlyXG4gICAqL1xuICBwcml2YXRlIGFsbG93VGFyZ2V0Q29ubmVjdGlvbihpbnN0YW5jZVBvcnQ6IG51bWJlciwgdGFyZ2V0OiBJTG9hZEJhbGFuY2VyVGFyZ2V0KSB7XG4gICAgdGhpcy5jb25uZWN0aW9ucy5hbGxvd1RvKFxuICAgICAgdGFyZ2V0LFxuICAgICAgUG9ydC50Y3AoaW5zdGFuY2VQb3J0KSxcbiAgICAgIGBQb3J0ICR7aW5zdGFuY2VQb3J0fSBMQiB0byBmbGVldGApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBpbnN0YW5jZSB0byB0aGUgbG9hZCBiYWxhbmNlci5cbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX2FkZEluc3RhbmNlSWQoaW5zdGFuY2VJZDogc3RyaW5nKSB7XG4gICAgdGhpcy5pbnN0YW5jZUlkcy5wdXNoKGluc3RhbmNlSWQpO1xuICB9XG59XG5cbi8qKlxuICogQW4gRUMyIGluc3RhbmNlIHRoYXQgaXMgdGhlIHRhcmdldCBmb3IgbG9hZCBiYWxhbmNpbmdcbiAqL1xuZXhwb3J0IGNsYXNzIEluc3RhbmNlVGFyZ2V0IGltcGxlbWVudHMgSUxvYWRCYWxhbmNlclRhcmdldCB7XG4gIHJlYWRvbmx5IGNvbm5lY3Rpb25zOiBDb25uZWN0aW9ucztcbiAgLyoqXG4gICAqIENyZWF0ZSBhIG5ldyBJbnN0YW5jZSB0YXJnZXQuXG4gICAqXG4gICAqIEBwYXJhbSBpbnN0YW5jZSBJbnN0YW5jZSB0byByZWdpc3RlciB0by5cbiAgICovXG4gIGNvbnN0cnVjdG9yKHB1YmxpYyByZWFkb25seSBpbnN0YW5jZTogSW5zdGFuY2UpIHtcbiAgICB0aGlzLmNvbm5lY3Rpb25zID0gaW5zdGFuY2UuY29ubmVjdGlvbnM7XG4gIH1cblxuICBwdWJsaWMgYXR0YWNoVG9DbGFzc2ljTEIobG9hZEJhbGFuY2VyOiBMb2FkQmFsYW5jZXIpOiB2b2lkIHtcbiAgICBsb2FkQmFsYW5jZXIuX2FkZEluc3RhbmNlSWQodGhpcy5pbnN0YW5jZS5pbnN0YW5jZUlkKTtcbiAgfVxufVxuXG4vKipcbiAqIFJlZmVyZW5jZSB0byBhIGxpc3RlbmVyJ3MgcG9ydCBqdXN0IGNyZWF0ZWQuXG4gKlxuICogVGhpcyBpbXBsZW1lbnRzIElDb25uZWN0YWJsZSB3aXRoIGEgZGVmYXVsdCBwb3J0ICh0aGUgcG9ydCB0aGF0IGFuIEVMQlxuICogbGlzdGVuZXIgd2FzIGp1c3QgY3JlYXRlZCBvbikgZm9yIGEgZ2l2ZW4gc2VjdXJpdHkgZ3JvdXAgc28gdGhhdCBpdCBjYW4gYmVcbiAqIGNvbnZlbmllbnRseSB1c2VkIGp1c3QgbGlrZSBhbnkgQ29ubmVjdGFibGUuIEUuZzpcbiAqXG4gKiAgICBjb25zdCBsaXN0ZW5lciA9IGVsYi5hZGRMaXN0ZW5lciguLi4pO1xuICpcbiAqICAgIGxpc3RlbmVyLmNvbm5lY3Rpb25zLmFsbG93RGVmYXVsdFBvcnRGcm9tQW55SVB2NCgpO1xuICogICAgLy8gb3JcbiAqICAgIGluc3RhbmNlLmNvbm5lY3Rpb25zLmFsbG93VG9EZWZhdWx0UG9ydChsaXN0ZW5lcik7XG4gKi9cbmV4cG9ydCBjbGFzcyBMaXN0ZW5lclBvcnQgaW1wbGVtZW50cyBJQ29ubmVjdGFibGUge1xuICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IENvbm5lY3Rpb25zO1xuXG4gIGNvbnN0cnVjdG9yKHNlY3VyaXR5R3JvdXA6IElTZWN1cml0eUdyb3VwLCBkZWZhdWx0UG9ydDogUG9ydCkge1xuICAgIHRoaXMuY29ubmVjdGlvbnMgPSBuZXcgQ29ubmVjdGlvbnMoeyBzZWN1cml0eUdyb3VwczogW3NlY3VyaXR5R3JvdXBdLCBkZWZhdWx0UG9ydCB9KTtcbiAgfVxufVxuXG5mdW5jdGlvbiB3ZWxsS25vd25Qcm90b2NvbChwb3J0OiBudW1iZXIpOiBMb2FkQmFsYW5jaW5nUHJvdG9jb2wge1xuICBjb25zdCBwcm90byA9IHRyeVdlbGxLbm93blByb3RvY29sKHBvcnQpO1xuICBpZiAoIXByb3RvKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBQbGVhc2Ugc3VwcGx5IHByb3RvY29sIHRvIGdvIHdpdGggcG9ydCAke3BvcnR9YCk7XG4gIH1cbiAgcmV0dXJuIHByb3RvO1xufVxuXG5mdW5jdGlvbiB0cnlXZWxsS25vd25Qcm90b2NvbChwb3J0OiBudW1iZXIpOiBMb2FkQmFsYW5jaW5nUHJvdG9jb2wgfCB1bmRlZmluZWQge1xuICBpZiAocG9ydCA9PT0gODApIHsgcmV0dXJuIExvYWRCYWxhbmNpbmdQcm90b2NvbC5IVFRQOyB9XG4gIGlmIChwb3J0ID09PSA0NDMpIHsgcmV0dXJuIExvYWRCYWxhbmNpbmdQcm90b2NvbC5IVFRQUzsgfVxuICByZXR1cm4gdW5kZWZpbmVkO1xufVxuXG5mdW5jdGlvbiBpc0h0dHBQcm90b2NvbChwcm90bzogTG9hZEJhbGFuY2luZ1Byb3RvY29sKTogYm9vbGVhbiB7XG4gIHJldHVybiBwcm90byA9PT0gTG9hZEJhbGFuY2luZ1Byb3RvY29sLkhUVFBTIHx8IHByb3RvID09PSBMb2FkQmFsYW5jaW5nUHJvdG9jb2wuSFRUUDtcbn1cblxuZnVuY3Rpb24gaWZVbmRlZmluZWQ8VD4oeDogVCB8IHVuZGVmaW5lZCwgZGVmOiBUKTogVCB7XG4gIHJldHVybiB4ICE9IG51bGwgPyB4IDogZGVmO1xufVxuXG5mdW5jdGlvbiBpZlVuZGVmaW5lZExhenk8VD4oeDogVCB8IHVuZGVmaW5lZCwgZGVmOiAoKSA9PiBUKTogVCB7XG4gIHJldHVybiB4ICE9IG51bGwgPyB4IDogZGVmKCk7XG59XG5cbi8qKlxuICogVHVybiBoZWFsdGggY2hlY2sgcGFyYW1ldGVycyBpbnRvIGEgcGFyYW1ldGVyIGJsb2IgZm9yIHRoZSBMQlxuICovXG5mdW5jdGlvbiBoZWFsdGhDaGVja1RvSlNPTihoZWFsdGhDaGVjazogSGVhbHRoQ2hlY2spOiBDZm5Mb2FkQmFsYW5jZXIuSGVhbHRoQ2hlY2tQcm9wZXJ0eSB7XG4gIGNvbnN0IHByb3RvY29sID0gaWZVbmRlZmluZWQoaGVhbHRoQ2hlY2sucHJvdG9jb2wsXG4gICAgaWZVbmRlZmluZWQodHJ5V2VsbEtub3duUHJvdG9jb2woaGVhbHRoQ2hlY2sucG9ydCksXG4gICAgICBMb2FkQmFsYW5jaW5nUHJvdG9jb2wuVENQKSk7XG5cbiAgY29uc3QgcGF0aCA9IHByb3RvY29sID09PSBMb2FkQmFsYW5jaW5nUHJvdG9jb2wuSFRUUCB8fCBwcm90b2NvbCA9PT0gTG9hZEJhbGFuY2luZ1Byb3RvY29sLkhUVFBTID8gaWZVbmRlZmluZWQoaGVhbHRoQ2hlY2sucGF0aCwgJy8nKSA6ICcnO1xuXG4gIGNvbnN0IHRhcmdldCA9IGAke3Byb3RvY29sLnRvVXBwZXJDYXNlKCl9OiR7aGVhbHRoQ2hlY2sucG9ydH0ke3BhdGh9YDtcblxuICByZXR1cm4ge1xuICAgIGhlYWx0aHlUaHJlc2hvbGQ6IGlmVW5kZWZpbmVkKGhlYWx0aENoZWNrLmhlYWx0aHlUaHJlc2hvbGQsIDIpLnRvU3RyaW5nKCksXG4gICAgaW50ZXJ2YWw6IChoZWFsdGhDaGVjay5pbnRlcnZhbCB8fCBEdXJhdGlvbi5zZWNvbmRzKDMwKSkudG9TZWNvbmRzKCkudG9TdHJpbmcoKSxcbiAgICB0YXJnZXQsXG4gICAgdGltZW91dDogKGhlYWx0aENoZWNrLnRpbWVvdXQgfHwgRHVyYXRpb24uc2Vjb25kcyg1KSkudG9TZWNvbmRzKCkudG9TdHJpbmcoKSxcbiAgICB1bmhlYWx0aHlUaHJlc2hvbGQ6IGlmVW5kZWZpbmVkKGhlYWx0aENoZWNrLnVuaGVhbHRoeVRocmVzaG9sZCwgNSkudG9TdHJpbmcoKSxcbiAgfTtcbn1cblxuZnVuY3Rpb24gbG9hZEJhbGFuY2VyU3VibmV0cyhwcm9wczogTG9hZEJhbGFuY2VyUHJvcHMpOiBTZWxlY3RlZFN1Ym5ldHMge1xuICBpZiAocHJvcHMuc3VibmV0U2VsZWN0aW9uICE9PSB1bmRlZmluZWQpIHtcbiAgICByZXR1cm4gcHJvcHMudnBjLnNlbGVjdFN1Ym5ldHMocHJvcHMuc3VibmV0U2VsZWN0aW9uKTtcbiAgfSBlbHNlIGlmIChwcm9wcy5pbnRlcm5ldEZhY2luZykge1xuICAgIHJldHVybiBwcm9wcy52cGMuc2VsZWN0U3VibmV0cyh7XG4gICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQyxcbiAgICB9KTtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gcHJvcHMudnBjLnNlbGVjdFN1Ym5ldHMoe1xuICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFX1dJVEhfRUdSRVNTLFxuICAgIH0pO1xuICB9XG59XG4iXX0=